# ActivitySim
# See full license in LICENSE.txt.

import os
import yaml

import logging
import orca

from activitysim import tracing


logger = logging.getLogger(__name__)


def add_dependent_columns(base_dfname, new_dfname):
    tbl = orca.get_table(new_dfname)
    for col in tbl.columns:
        logger.info("Adding dependent column %s" % col)
        orca.add_column(base_dfname, col, tbl[col])


def read_model_settings(configs_dir, file_name):
    settings = None
    file_path = os.path.join(configs_dir,  file_name)
    if os.path.isfile(file_path):
        with open(file_path) as f:
            settings = yaml.load(f)

    if settings is None:
        settings = {}

    return settings


def get_model_constants(model_settings):
    """
    Read constants from model settings file

    Returns
    -------
    constants : dict
        dictionary of constants to add to locals for use by expressions in model spec
    """
    return model_settings.get('CONSTANTS', {})


def get_logit_model_settings(model_settings):
    """
    Read nest spec (for nested logit) from model settings file

    Returns
    -------
    nests : dict
        dictionary specifying nesting structure and nesting coefficients

    constants : dict
        dictionary of constants to add to locals for use by expressions in model spec
    """
    nests = None

    if model_settings is not None:

        # default to MNL
        logit_type = model_settings.get('LOGIT_TYPE', 'MNL')

        if logit_type not in ['NL', 'MNL']:
            logging.error("Unrecognized logit type '%s'" % logit_type)
            raise RuntimeError("Unrecognized logit type '%s'" % logit_type)

        if logit_type == 'NL':
            nests = model_settings.get('NESTS', None)
            if nests is None:
                logger.error("No NEST found in model spec for NL model type")
                raise RuntimeError("No NEST found in model spec for NL model type")

    return nests
