# Publishing to PyPI - Maintainer Guide

This guide is for package maintainers who need to publish new versions of `admesh-weave-python` to PyPI.

## 📋 Pre-Publishing Checklist

Before publishing a new version, ensure:

- [ ] All tests pass locally and in CI
- [ ] Documentation is up-to-date
- [ ] CHANGELOG.md is updated with new version
- [ ] Version number follows semantic versioning
- [ ] Breaking changes are clearly documented
- [ ] Examples are tested and working
- [ ] README.md reflects any API changes
- [ ] All dependencies are correctly specified
- [ ] Security vulnerabilities are addressed

## 📦 Package Structure

The package includes:

**Core Package:**
- `admesh_weave/__init__.py` - Package entry point
- `admesh_weave/client.py` - Main client implementation
- `admesh_weave/types.py` - Type definitions
- `admesh_weave/py.typed` - PEP 561 type marker

**Distribution Files:**
- Source distribution (.tar.gz) - Full source code, docs, examples, tests
- Wheel distribution (.whl) - Optimized binary distribution
- LICENSE - MIT License
- README.md - Package documentation

## � Semantic Versioning

This package follows [Semantic Versioning 2.0.0](https://semver.org/):

**Version Format**: `MAJOR.MINOR.PATCH`

- **MAJOR** (1.0.0): Breaking changes, incompatible API changes
- **MINOR** (0.1.0): New features, backward-compatible
- **PATCH** (0.0.1): Bug fixes, backward-compatible

### Version Guidelines

**Breaking Changes (Major):**
- Removing or renaming public APIs
- Changing method signatures
- Removing support for Python versions
- Changing default behavior that breaks existing code

**New Features (Minor):**
- Adding new methods or classes
- Adding optional parameters with defaults
- Adding new functionality without breaking existing code
- Deprecating features (with warnings)

**Bug Fixes (Patch):**
- Fixing bugs without changing API
- Documentation updates
- Performance improvements
- Internal refactoring

### Pre-release Versions

For testing before official release:
- **Alpha**: `0.1.0a1`, `0.1.0a2` - Early testing
- **Beta**: `0.1.0b1`, `0.1.0b2` - Feature complete, testing
- **Release Candidate**: `0.1.0rc1` - Final testing before release

## 🔨 Building the Package

### Install Build Tools

```bash
pip install --upgrade build twine
```

### Build Distribution Files

```bash
# Clean previous builds
rm -rf dist/ build/ *.egg-info

# Build source distribution and wheel
python3 -m build
```

This creates:
- `dist/admesh_weave_python-X.Y.Z.tar.gz` - Source distribution
- `dist/admesh_weave_python-X.Y.Z-py3-none-any.whl` - Wheel distribution

### Verify Package

```bash
# Check package metadata and description
python3 -m twine check dist/*
```

All checks should pass before publishing.

## 🚀 Publishing to PyPI

### Option 1: Test on TestPyPI First (Recommended)

Always test on TestPyPI before publishing to production:

```bash
# Upload to TestPyPI
python3 -m twine upload --repository testpypi dist/*
```

Test installation:
```bash
# Install from TestPyPI
pip install --index-url https://test.pypi.org/simple/ admesh-weave-python

# Verify import works
python3 -c "from admesh_weave import AdMeshClient; print('✓ Import successful')"
```

### Option 2: Publish to PyPI (Production)

Once TestPyPI testing is successful:

```bash
# Upload to production PyPI
python3 -m twine upload dist/*
```

You'll be prompted for credentials (see Authentication section below).

## 🔑 Authentication

### PyPI API Tokens (Recommended)

API tokens are more secure than passwords and can be scoped to specific projects.

**Create API Token:**

1. **For PyPI**: Visit https://pypi.org/manage/account/token/
2. **For TestPyPI**: Visit https://test.pypi.org/manage/account/token/
3. Click "Add API token"
4. **Token name**: `admesh-weave-python` (or descriptive name)
5. **Scope**: Choose "Project: admesh-weave-python" (recommended) or "Entire account"
6. Click "Add token"
7. **Copy the token immediately** (you won't see it again!)

**Using the Token:**

When `twine upload` prompts for credentials:
- **Username**: `__token__` (exactly as shown, including underscores)
- **Password**: Paste your API token (starts with `pypi-`)

### Configure `.pypirc` (Optional)

To avoid entering credentials each time, create `~/.pypirc`:

```ini
[distutils]
index-servers =
    pypi
    testpypi

[pypi]
username = __token__
password = pypi-YOUR_PRODUCTION_TOKEN_HERE

[testpypi]
repository = https://test.pypi.org/legacy/
username = __token__
password = pypi-YOUR_TEST_TOKEN_HERE
```

**Security:**
```bash
# Set restrictive permissions
chmod 600 ~/.pypirc
```

⚠️ **Never commit `.pypirc` or tokens to version control!**

## 🎯 Post-Publishing Steps

### 1. Verify Installation

```bash
# Install from PyPI
pip install admesh-weave-python

# Test import and basic functionality
python3 -c "from admesh_weave import AdMeshClient; print('✓ Import successful')"
```

### 2. Create GitHub Release

Tag the release in git:

```bash
# Create and push tag
git tag -a v0.1.0 -m "Release v0.1.0"
git push origin v0.1.0
```

Create a GitHub release:
1. Go to repository releases page
2. Click "Create a new release"
3. Select the tag you just created
4. Add release notes from CHANGELOG.md
5. Publish release

### 3. Update Documentation

Add PyPI badges to README.md:

```markdown
[![PyPI version](https://badge.fury.io/py/admesh-weave-python.svg)](https://badge.fury.io/py/admesh-weave-python)
[![Python Versions](https://img.shields.io/pypi/pyversions/admesh-weave-python.svg)](https://pypi.org/project/admesh-weave-python/)
[![License: MIT](https://img.shields.io/badge/License-MIT-yellow.svg)](https://opensource.org/licenses/MIT)
```

### 4. Announce the Release

Consider announcing on:
- GitHub Discussions or repository README
- Project documentation site
- Relevant community forums or channels
- Social media (if applicable)

## 🔄 Publishing Updates

### Release Workflow

Follow this workflow for publishing new versions:

**1. Prepare the Release**

```bash
# Create a release branch (optional but recommended)
git checkout -b release/v0.2.0

# Update version in pyproject.toml
# Update CHANGELOG.md with changes
# Update README.md if needed
# Run tests to ensure everything works
pytest

# Commit changes
git add pyproject.toml CHANGELOG.md
git commit -m "Prepare release v0.2.0"
```

**2. Build and Test**

```bash
# Clean previous builds
rm -rf dist/ build/ *.egg-info

# Build new version
python3 -m build

# Verify package
python3 -m twine check dist/*

# Test on TestPyPI first
python3 -m twine upload --repository testpypi dist/*

# Install and test from TestPyPI
pip install --index-url https://test.pypi.org/simple/ admesh-weave-python==0.2.0
```

**3. Publish to Production**

```bash
# Upload to PyPI
python3 -m twine upload dist/*

# Merge release branch
git checkout main
git merge release/v0.2.0

# Create and push tag
git tag -a v0.2.0 -m "Release v0.2.0"
git push origin main
git push origin v0.2.0
```

**4. Post-Release**

- Create GitHub release with changelog
- Verify installation from PyPI
- Update documentation if needed
- Announce the release

## 🔧 Handling Breaking Changes

When introducing breaking changes:

**1. Deprecation Period (Recommended)**

Before removing features:

```python
import warnings

def old_method(self):
    warnings.warn(
        "old_method is deprecated and will be removed in v2.0.0. "
        "Use new_method instead.",
        DeprecationWarning,
        stacklevel=2
    )
    return self.new_method()
```

**2. Document Breaking Changes**

In CHANGELOG.md:

```markdown
## [2.0.0] - 2024-XX-XX

### Breaking Changes
- Removed `old_method()` (deprecated in v1.5.0)
- Changed `get_recommendations()` signature to require `session_id`
- Dropped support for Python 3.7

### Migration Guide
- Replace `old_method()` with `new_method()`
- Update calls to `get_recommendations()` to include `session_id` parameter
```

**3. Version Bump**

- Increment MAJOR version (e.g., 1.9.0 → 2.0.0)
- Clearly communicate changes to users
- Provide migration guide

## 🤝 Community Contributions

### Accepting Contributions

When accepting community contributions:

1. **Review thoroughly** - Check code quality, tests, documentation
2. **Update CHANGELOG.md** - Credit the contributor
3. **Decide version bump** - Based on change type (major/minor/patch)
4. **Thank contributors** - Acknowledge their work in release notes

### Contributor Recognition

In CHANGELOG.md:

```markdown
## [0.2.0] - 2024-XX-XX

### Added
- New feature X (thanks @contributor-username)

### Fixed
- Bug fix Y (thanks @another-contributor)
```

## 🛠️ Troubleshooting

### Package name already exists

If the package name is taken:
1. Choose a different name (e.g., `admesh-weave`, `admesh-weave-sdk`)
2. Update `name` field in `pyproject.toml`
3. Update all references in documentation
4. Rebuild and upload

### Upload fails with 403 Forbidden

**Possible causes:**
- Incorrect API token
- Token doesn't have correct scope
- Package name already claimed by another user
- Trying to upload existing version

**Solutions:**
- Verify token is correct and not expired
- Check token scope includes the project
- Ensure version number is incremented

### Version already exists on PyPI

**Error**: `File already exists`

**Solution**: You cannot re-upload the same version to PyPI
- Increment version number in `pyproject.toml`
- Update CHANGELOG.md
- Rebuild: `python3 -m build`
- Upload new version

### Build fails

**Common issues:**
- Missing dependencies in `pyproject.toml`
- Syntax errors in source files
- Invalid package structure

**Solutions:**
```bash
# Check for syntax errors
python3 -m py_compile src/admesh_weave/*.py

# Verify package structure
python3 -m build --verbose

# Check dependencies
pip install -e .
```

## 🔐 Security Best Practices

### Protecting Credentials

**Never commit secrets:**
```bash
# Add to .gitignore
echo ".pypirc" >> .gitignore
echo "*.token" >> .gitignore
```

**Secure `.pypirc` permissions:**
```bash
chmod 600 ~/.pypirc
```

**Use project-scoped tokens:**
- Limit token scope to specific project
- Reduces risk if token is compromised
- Easier to rotate without affecting other projects

**Rotate tokens regularly:**
- Set calendar reminder to rotate tokens every 6-12 months
- Immediately rotate if token may be compromised
- Delete old tokens after rotation

### Security Checklist

Before publishing:
- [ ] No hardcoded credentials in source code
- [ ] No API keys or tokens in examples
- [ ] Dependencies are from trusted sources
- [ ] No known security vulnerabilities (run `pip-audit`)
- [ ] `.pypirc` has restrictive permissions (600)

## 📊 Monitoring and Analytics

### Package Statistics

Track package health and usage:

- **PyPI Stats**: https://pypistats.org/packages/admesh-weave-python
- **Download trends**: Monitor adoption and version distribution
- **Libraries.io**: https://libraries.io/pypi/admesh-weave-python

### Useful Metrics

- Total downloads
- Downloads by version
- Downloads by Python version
- Geographic distribution

## 💬 Support Channels

### For Package Users

Direct users to appropriate support channels:

- **Documentation**: https://github.com/GouniManikumar12/admesh-weave-python
- **Issues**: https://github.com/GouniManikumar12/admesh-weave-python/issues
- **Discussions**: GitHub Discussions (if enabled)

### For Contributors

- **Contributing Guide**: See CONTRIBUTING.md
- **Code of Conduct**: Maintain respectful community
- **Pull Request Template**: Provide clear guidelines

## 📝 Quick Reference

### Essential Commands

```bash
# Install build tools
pip install --upgrade build twine

# Clean and build
rm -rf dist/ build/ *.egg-info
python3 -m build

# Verify package
python3 -m twine check dist/*

# Test on TestPyPI
python3 -m twine upload --repository testpypi dist/*

# Publish to PyPI
python3 -m twine upload dist/*

# Create git tag
git tag -a v0.1.0 -m "Release v0.1.0"
git push origin v0.1.0
```

### Version Bumping

```bash
# Patch release (0.1.0 → 0.1.1)
# Bug fixes only

# Minor release (0.1.0 → 0.2.0)
# New features, backward compatible

# Major release (0.1.0 → 1.0.0)
# Breaking changes
```

## ✅ Final Checklist

Before publishing to production PyPI:

- [ ] All tests pass
- [ ] Documentation updated
- [ ] CHANGELOG.md updated
- [ ] Version bumped appropriately
- [ ] Package builds successfully
- [ ] `twine check` passes
- [ ] Tested on TestPyPI
- [ ] Git tag created
- [ ] Release notes prepared
- [ ] Breaking changes documented (if any)

## 🚀 Ready to Publish

**Test first:**
```bash
python3 -m twine upload --repository testpypi dist/*
```

**Then publish:**
```bash
python3 -m twine upload dist/*
```

**Verify:**
```bash
pip install admesh-weave-python
python3 -c "from admesh_weave import AdMeshClient; print('✓ Success!')"
```

---

**Questions or issues?** Open an issue on GitHub or consult the [PyPI documentation](https://packaging.python.org/).

