"""
AdMesh Client - Backend SDK for Python (Database-backed, no Pub/Sub)

Simplified architecture:
- Call admesh-protocol /agent/recommend directly
- Backend returns cached recommendations if available, or generates then returns
- No SSE, no Pub/Sub; one HTTP call with short timeout
"""

import os
from typing import Any, Dict, List, Optional

import httpx

from .types import AdMeshRecommendation, AdMeshSubscriptionOptions, AdMeshWaitResult


class AdMeshClient:
    """
    AdMesh Client for consuming recommendations from the AdMesh Protocol service.

    This client provides a simple interface to fetch recommendations for Weave format
    by calling the /agent/recommend endpoint directly.

    Example:
        ```python
        from admesh_weave import AdMeshClient

        client = AdMeshClient(api_key="your-api-key")

        # In your LLM handler
        result = await client.get_recommendations_for_weave(
            session_id="sess_123",
            message_id="msg_1",
            query=user_query
        )

        if result["found"]:
            # Pass recommendations to LLM
            recommendations = result["recommendations"]
        ```
    """

    def __init__(self, api_key: str, api_base_url: Optional[str] = None) -> None:
        """
        Initialize the AdMesh Client.

        Args:
            api_key: Your AdMesh API key (required)
            api_base_url: Optional API base URL. Defaults to environment variable
                         ADMESH_API_BASE_URL or 'https://api.useadmesh.com'

        Raises:
            ValueError: If api_key is not provided
        """
        if not api_key:
            raise ValueError("AdMeshClient: api_key is required")

        self.api_key = api_key

        # Set API base URL with priority: config > environment variable > production default
        self.api_base_url = (
            api_base_url or os.environ.get("ADMESH_API_BASE_URL") or "https://api.useadmesh.com"
        )

    async def get_recommendations_for_weave(
        self,
        session_id: str,
        message_id: str,
        query: str,
        timeout_ms: Optional[int] = None,
    ) -> AdMeshWaitResult:
        """
        Get recommendations for Weave format by calling /agent/recommend.

        The backend will:
        - Return cached recommendations for (session_id, message_id) if present
        - Otherwise, generate recommendations, persist, and return them

        Args:
            session_id: Session ID
            message_id: Message ID for this conversation message
            query: User query (required for contextual recommendations)
            timeout_ms: Max wait time in milliseconds (default: 12000ms)

        Returns:
            AdMeshWaitResult dictionary with:
                - found: Whether recommendations were found
                - recommendations: List of recommendations (if found)
                - query: Original query
                - request_id: Request ID
                - error: Error message (if not found)

        Example:
            ```python
            result = await client.get_recommendations_for_weave(
                session_id="sess_123",
                message_id="msg_1",
                query="best laptops for programming",
                timeout_ms=30000
            )

            if result["found"]:
                print("Recommendations:", result["recommendations"])
            else:
                print("No recommendations found:", result.get("error"))
            ```
        """
        # Validate required parameters
        if not query or not query.strip():
            raise ValueError("AdMeshClient.get_recommendations_for_weave: 'query' parameter is required and cannot be empty")

        timeout_seconds = (timeout_ms or 12000) / 1000.0

        payload: Dict[str, Any] = {
            "query": query,
            "format": "weave",
            "session_id": session_id,
            "message_id": message_id,
            "source": "admesh_weave_python",  # Identify requests from the Weave Python backend SDK
        }

        url = f"{self.api_base_url.rstrip('/')}/agent/recommend"

        try:
            async with httpx.AsyncClient(timeout=timeout_seconds) as client:
                response = await client.post(
                    url,
                    json=payload,
                    headers={
                        "Content-Type": "application/json",
                        "Authorization": f"Bearer {self.api_key}",
                    },
                )

                if not response.is_success:
                    error_text = ""
                    try:
                        error_text = response.text
                    except Exception:
                        pass

                    raise Exception(
                        f"HTTP {response.status_code}: {response.reason_phrase}"
                        + (f" - {error_text}" if error_text else "")
                    )

                data = response.json()

                # Expected shape: { session_id, message_id, recommendations: [...] }
                recommendations: List[AdMeshRecommendation] = data.get("recommendations", [])

                if isinstance(recommendations, list) and len(recommendations) > 0:
                    return AdMeshWaitResult(
                        found=True,
                        recommendations=recommendations,
                        query=query,
                        request_id=data.get("session_id"),
                    )

                return AdMeshWaitResult(found=False, error="No recommendations found")

        except Exception as err:
            raise err

    def get_recommendations_for_weave_sync(
        self,
        session_id: str,
        message_id: str,
        query: str,
        timeout_ms: Optional[int] = None,
    ) -> AdMeshWaitResult:
        """
        Synchronous version of get_recommendations_for_weave.

        Get recommendations for Weave format by calling /agent/recommend.

        Args:
            session_id: Session ID
            message_id: Message ID for this conversation message
            query: User query (required for contextual recommendations)
            timeout_ms: Max wait time in milliseconds (default: 12000ms)

        Returns:
            AdMeshWaitResult dictionary with recommendations or error

        Example:
            ```python
            result = client.get_recommendations_for_weave_sync(
                session_id="sess_123",
                message_id="msg_1",
                query="best laptops for programming"
            )
            ```
        """
        # Validate required parameters
        if not query or not query.strip():
            raise ValueError("AdMeshClient.get_recommendations_for_weave_sync: 'query' parameter is required and cannot be empty")

        timeout_seconds = (timeout_ms or 12000) / 1000.0

        payload: Dict[str, Any] = {
            "query": query,
            "format": "weave",
            "session_id": session_id,
            "message_id": message_id,
            "source": "admesh_weave_python",
        }

        url = f"{self.api_base_url.rstrip('/')}/agent/recommend"

        try:
            with httpx.Client(timeout=timeout_seconds) as client:
                response = client.post(
                    url,
                    json=payload,
                    headers={
                        "Content-Type": "application/json",
                        "Authorization": f"Bearer {self.api_key}",
                    },
                )

                if not response.is_success:
                    error_text = ""
                    try:
                        error_text = response.text
                    except Exception:
                        pass

                    raise Exception(
                        f"HTTP {response.status_code}: {response.reason_phrase}"
                        + (f" - {error_text}" if error_text else "")
                    )

                data = response.json()

                recommendations: List[AdMeshRecommendation] = data.get("recommendations", [])

                if isinstance(recommendations, list) and len(recommendations) > 0:
                    return AdMeshWaitResult(
                        found=True,
                        recommendations=recommendations,
                        query=query,
                        request_id=data.get("session_id"),
                    )

                return AdMeshWaitResult(found=False, error="No recommendations found")

        except Exception as err:
            raise err

