import pytest
import numpy as np
from AES_Python import AES


# 128-bit key tests
@pytest.mark.parametrize("key,r_keys", [
    ("00000000000000000000000000000000", np.array([[[0, 0, 0, 0],
                                                    [0, 0, 0, 0],
                                                    [0, 0, 0, 0],
                                                    [0, 0, 0, 0]],
                                                   [[98, 99, 99, 99],
                                                    [98, 99, 99, 99],
                                                    [98, 99, 99, 99],
                                                    [98, 99, 99, 99]],
                                                   [[155, 152, 152, 201],
                                                    [249, 251, 251, 170],
                                                    [155, 152, 152, 201],
                                                    [249, 251, 251, 170]],
                                                   [[144, 151, 52, 80],
                                                    [105, 108, 207, 250],
                                                    [242, 244, 87, 51],
                                                    [11, 15, 172, 153]],
                                                   [[238, 6, 218, 123],
                                                    [135, 106, 21, 129],
                                                    [117, 158, 66, 178],
                                                    [126, 145, 238, 43]],
                                                   [[127, 46, 43, 136],
                                                    [248, 68, 62, 9],
                                                    [141, 218, 124, 187],
                                                    [243, 75, 146, 144]],
                                                   [[236, 97, 75, 133],
                                                    [20, 37, 117, 140],
                                                    [153, 255, 9, 55],
                                                    [106, 180, 155, 167]],
                                                   [[33, 117, 23, 135],
                                                    [53, 80, 98, 11],
                                                    [172, 175, 107, 60],
                                                    [198, 27, 240, 155]],
                                                   [[14, 249, 3, 51],
                                                    [59, 169, 97, 56],
                                                    [151, 6, 10, 4],
                                                    [81, 29, 250, 159]],
                                                   [[177, 212, 216, 226],
                                                    [138, 125, 185, 218],
                                                    [29, 123, 179, 222],
                                                    [76, 102, 73, 65]],
                                                   [[180, 239, 91, 203],
                                                    [62, 146, 226, 17],
                                                    [35, 233, 81, 207],
                                                    [111, 143, 24, 142]],
                                                   ])),
    ("ffffffffffffffffffffffffffffffff", np.array([[[0xff, 0xff, 0xff, 0xff],
                                                    [0xff, 0xff, 0xff, 0xff],
                                                    [0xff, 0xff, 0xff, 0xff],
                                                    [0xff, 0xff, 0xff, 0xff]],
                                                   [[0xe8, 0xe9, 0xe9, 0xe9],
                                                    [0x17, 0x16, 0x16, 0x16],
                                                    [0xe8, 0xe9, 0xe9, 0xe9],
                                                    [0x17, 0x16, 0x16, 0x16]],
                                                   [[0xad, 0xae, 0xae, 0x19],
                                                    [0xba, 0xb8, 0xb8, 0x0f],
                                                    [0x52, 0x51, 0x51, 0xe6],
                                                    [0x45, 0x47, 0x47, 0xf0]],
                                                   [[0x09, 0x0e, 0x22, 0x77],
                                                    [0xb3, 0xb6, 0x9a, 0x78],
                                                    [0xe1, 0xe7, 0xcb, 0x9e],
                                                    [0xa4, 0xa0, 0x8c, 0x6e]],
                                                   [[0xe1, 0x6a, 0xbd, 0x3e],
                                                    [0x52, 0xdc, 0x27, 0x46],
                                                    [0xb3, 0x3b, 0xec, 0xd8],
                                                    [0x17, 0x9b, 0x60, 0xb6]],
                                                   [[0xe5, 0xba, 0xf3, 0xce],
                                                    [0xb7, 0x66, 0xd4, 0x88],
                                                    [0x04, 0x5d, 0x38, 0x50],
                                                    [0x13, 0xc6, 0x58, 0xe6]],
                                                   [[0x71, 0xd0, 0x7d, 0xb3],
                                                    [0xc6, 0xb6, 0xa9, 0x3b],
                                                    [0xc2, 0xeb, 0x91, 0x6b],
                                                    [0xd1, 0x2d, 0xc9, 0x8d]],
                                                   [[0xe9, 0x0d, 0x20, 0x8d],
                                                    [0x2f, 0xbb, 0x89, 0xb6],
                                                    [0xed, 0x50, 0x18, 0xdd],
                                                    [0x3c, 0x7d, 0xd1, 0x50]],
                                                   [[0x96, 0x33, 0x73, 0x66],
                                                    [0xb9, 0x88, 0xfa, 0xd0],
                                                    [0x54, 0xd8, 0xe2, 0x0d],
                                                    [0x68, 0xa5, 0x33, 0x5d]],
                                                   [[0x8b, 0xf0, 0x3f, 0x23],
                                                    [0x32, 0x78, 0xc5, 0xf3],
                                                    [0x66, 0xa0, 0x27, 0xfe],
                                                    [0x0e, 0x05, 0x14, 0xa3]],
                                                   [[0xd6, 0x0a, 0x35, 0x88],
                                                    [0xe4, 0x72, 0xf0, 0x7b],
                                                    [0x82, 0xd2, 0xd7, 0x85],
                                                    [0x8c, 0xd7, 0xc3, 0x26]]])),
    ("000102030405060708090a0b0c0d0e0f", np.array([
                                                 [[0x00, 0x01, 0x02, 0x03],
                                                  [0x04, 0x05, 0x06, 0x07],
                                                  [0x08, 0x09, 0x0a, 0x0b],
                                                  [0x0c, 0x0d, 0x0e, 0x0f]],
                                                 [[0xd6, 0xaa, 0x74, 0xfd],
                                                  [0xd2, 0xaf, 0x72, 0xfa],
                                                  [0xda, 0xa6, 0x78, 0xf1],
                                                  [0xd6, 0xab, 0x76, 0xfe]],
                                                 [[0xb6, 0x92, 0xcf, 0x0b],
                                                  [0x64, 0x3d, 0xbd, 0xf1],
                                                  [0xbe, 0x9b, 0xc5, 0x00],
                                                  [0x68, 0x30, 0xb3, 0xfe]],
                                                 [[0xb6, 0xff, 0x74, 0x4e],
                                                  [0xd2, 0xc2, 0xc9, 0xbf],
                                                  [0x6c, 0x59, 0x0c, 0xbf],
                                                  [0x04, 0x69, 0xbf, 0x41]],
                                                 [[0x47, 0xf7, 0xf7, 0xbc],
                                                  [0x95, 0x35, 0x3e, 0x03],
                                                  [0xf9, 0x6c, 0x32, 0xbc],
                                                  [0xfd, 0x05, 0x8d, 0xfd]],
                                                 [[0x3c, 0xaa, 0xa3, 0xe8],
                                                  [0xa9, 0x9f, 0x9d, 0xeb],
                                                  [0x50, 0xf3, 0xaf, 0x57],
                                                  [0xad, 0xf6, 0x22, 0xaa]],
                                                 [[0x5e, 0x39, 0x0f, 0x7d],
                                                  [0xf7, 0xa6, 0x92, 0x96],
                                                  [0xa7, 0x55, 0x3d, 0xc1],
                                                  [0x0a, 0xa3, 0x1f, 0x6b]],
                                                 [[0x14, 0xf9, 0x70, 0x1a],
                                                  [0xe3, 0x5f, 0xe2, 0x8c],
                                                  [0x44, 0x0a, 0xdf, 0x4d],
                                                  [0x4e, 0xa9, 0xc0, 0x26]],
                                                 [[0x47, 0x43, 0x87, 0x35],
                                                  [0xa4, 0x1c, 0x65, 0xb9],
                                                  [0xe0, 0x16, 0xba, 0xf4],
                                                  [0xae, 0xbf, 0x7a, 0xd2]],
                                                 [[0x54, 0x99, 0x32, 0xd1],
                                                  [0xf0, 0x85, 0x57, 0x68],
                                                  [0x10, 0x93, 0xed, 0x9c],
                                                  [0xbe, 0x2c, 0x97, 0x4e]],
                                                 [[0x13, 0x11, 0x1d, 0x7f],
                                                  [0xe3, 0x94, 0x4a, 0x17],
                                                  [0xf3, 0x07, 0xa7, 0x8b],
                                                  [0x4d, 0x2b, 0x30, 0xc5]]])),
    ("4920e299a520526164696f476174756e", np.array([
                                                [[0x49, 0x20, 0xe2, 0x99],
                                                 [0xa5, 0x20, 0x52, 0x61],
                                                 [0x64, 0x69, 0x6f, 0x47],
                                                 [0x61, 0x74, 0x75, 0x6e]],
                                                [[0xda, 0xbd, 0x7d, 0x76],
                                                 [0x7f, 0x9d, 0x2f, 0x17],
                                                 [0x1b, 0xf4, 0x40, 0x50],
                                                 [0x7a, 0x80, 0x35, 0x3e]],
                                                [[0x15, 0x2b, 0xcf, 0xac],
                                                 [0x6a, 0xb6, 0xe0, 0xbb],
                                                 [0x71, 0x42, 0xa0, 0xeb],
                                                 [0x0b, 0xc2, 0x95, 0xd5]],
                                                [[0x34, 0x01, 0xcc, 0x87],
                                                 [0x5e, 0xb7, 0x2c, 0x3c],
                                                 [0x2f, 0xf5, 0x8c, 0xd7],
                                                 [0x24, 0x37, 0x19, 0x02]],
                                                [[0xa6, 0xd5, 0xbb, 0xb1],
                                                 [0xf8, 0x62, 0x97, 0x8d],
                                                 [0xd7, 0x97, 0x1b, 0x5a],
                                                 [0xf3, 0xa0, 0x02, 0x58]],
                                                [[0x56, 0xa2, 0xd1, 0xbc],
                                                 [0xae, 0xc0, 0x46, 0x31],
                                                 [0x79, 0x57, 0x5d, 0x6b],
                                                 [0x8a, 0xf7, 0x5f, 0x33]],
                                                [[0x1e, 0x6d, 0x12, 0xc2],
                                                 [0xb0, 0xad, 0x54, 0xf3],
                                                 [0xc9, 0xfa, 0x09, 0x98],
                                                 [0x43, 0x0d, 0x56, 0xab]],
                                                [[0x89, 0xdc, 0x70, 0xd8],
                                                 [0x39, 0x71, 0x24, 0x2b],
                                                 [0xf0, 0x8b, 0x2d, 0xb3],
                                                 [0xb3, 0x86, 0x7b, 0x18]],
                                                [[0x4d, 0xfd, 0xdd, 0xb5],
                                                 [0x74, 0x8c, 0xf9, 0x9e],
                                                 [0x84, 0x07, 0xd4, 0x2d],
                                                 [0x37, 0x81, 0xaf, 0x35]],
                                                [[0x5a, 0x84, 0x4b, 0x2f],
                                                 [0x2e, 0x08, 0xb2, 0xb1],
                                                 [0xaa, 0x0f, 0x66, 0x9c],
                                                 [0x9d, 0x8e, 0xc9, 0xa9]],
                                                [[0x75, 0x59, 0x98, 0x71],
                                                 [0x5b, 0x51, 0x2a, 0xc0],
                                                 [0xf1, 0x5e, 0x4c, 0x5c],
                                                 [0x6c, 0xd0, 0x85, 0xf5]]])),
])
def test_aes_key_expansion_128bit(key, r_keys):
    assert np.array_equal(AES.key_expand(key), r_keys)


# 192-bit key tests
@pytest.mark.parametrize("key,r_keys", [
    ("000000000000000000000000000000000000000000000000", np.array([
                                                                 [[0, 0, 0, 0],
                                                                  [0, 0, 0, 0],
                                                                  [0, 0, 0, 0],
                                                                  [0, 0, 0, 0]],
                                                                 [[0, 0, 0, 0],
                                                                  [0, 0, 0, 0],
                                                                  [98, 99, 99, 99],
                                                                  [98, 99, 99, 99]],
                                                                 [[98, 99, 99, 99],
                                                                  [98, 99, 99, 99],
                                                                  [98, 99, 99, 99],
                                                                  [98, 99, 99, 99]],
                                                                 [[155, 152, 152, 201],
                                                                  [249, 251, 251, 170],
                                                                  [155, 152, 152, 201],
                                                                  [249, 251, 251, 170]],
                                                                 [[155, 152, 152, 201],
                                                                  [249, 251, 251, 170],
                                                                  [144, 151, 52, 80],
                                                                  [105, 108, 207, 250]],
                                                                 [[242, 244, 87, 51],
                                                                  [11, 15, 172, 153],
                                                                  [144, 151, 52, 80],
                                                                  [105, 108, 207, 250]],
                                                                 [[200, 29, 25, 169],
                                                                  [161, 113, 214, 83],
                                                                  [83, 133, 129, 96],
                                                                  [88, 138, 45, 249]],
                                                                 [[200, 29, 25, 169],
                                                                  [161, 113, 214, 83],
                                                                  [123, 235, 244, 155],
                                                                  [218, 154, 34, 200]],
                                                                 [[137, 31, 163, 168],
                                                                  [209, 149, 142, 81],
                                                                  [25, 136, 151, 248],
                                                                  [184, 249, 65, 171]],
                                                                 [[194, 104, 150, 247],
                                                                  [24, 242, 180, 63],
                                                                  [145, 237, 23, 151],
                                                                  [64, 120, 153, 198]],
                                                                 [[89, 240, 14, 62],
                                                                  [225, 9, 79, 149],
                                                                  [131, 236, 188, 15],
                                                                  [155, 30, 8, 48]],
                                                                 [[10, 243, 31, 167],
                                                                  [74, 139, 134, 97],
                                                                  [19, 123, 136, 95],
                                                                  [242, 114, 199, 202]],
                                                                 [[67, 42, 200, 134],
                                                                  [216, 52, 192, 182],
                                                                  [210, 199, 223, 17],
                                                                  [152, 76, 89, 112]]])),
    ("ffffffffffffffffffffffffffffffffffffffffffffffff", np.array([
                                                                 [[0xff, 0xff, 0xff, 0xff],
                                                                  [0xff, 0xff, 0xff, 0xff],
                                                                  [0xff, 0xff, 0xff, 0xff],
                                                                  [0xff, 0xff, 0xff, 0xff]],
                                                                 [[0xff, 0xff, 0xff, 0xff],
                                                                  [0xff, 0xff, 0xff, 0xff],
                                                                  [0xe8, 0xe9, 0xe9, 0xe9],
                                                                  [0x17, 0x16, 0x16, 0x16]],
                                                                 [[0xe8, 0xe9, 0xe9, 0xe9],
                                                                  [0x17, 0x16, 0x16, 0x16],
                                                                  [0xe8, 0xe9, 0xe9, 0xe9],
                                                                  [0x17, 0x16, 0x16, 0x16]],
                                                                 [[0xad, 0xae, 0xae, 0x19],
                                                                  [0xba, 0xb8, 0xb8, 0x0f],
                                                                  [0x52, 0x51, 0x51, 0xe6],
                                                                  [0x45, 0x47, 0x47, 0xf0]],
                                                                 [[0xad, 0xae, 0xae, 0x19],
                                                                  [0xba, 0xb8, 0xb8, 0x0f],
                                                                  [0xc5, 0xc2, 0xd8, 0xed],
                                                                  [0x7f, 0x7a, 0x60, 0xe2]],
                                                                 [[0x2d, 0x2b, 0x31, 0x04],
                                                                  [0x68, 0x6c, 0x76, 0xf4],
                                                                  [0xc5, 0xc2, 0xd8, 0xed],
                                                                  [0x7f, 0x7a, 0x60, 0xe2]],
                                                                 [[0x17, 0x12, 0x40, 0x3f],
                                                                  [0x68, 0x68, 0x20, 0xdd],
                                                                  [0x45, 0x43, 0x11, 0xd9],
                                                                  [0x2d, 0x2f, 0x67, 0x2d]],
                                                                 [[0xe8, 0xed, 0xbf, 0xc0],
                                                                  [0x97, 0x97, 0xdf, 0x22],
                                                                  [0x8f, 0x8c, 0xd3, 0xb7],
                                                                  [0xe7, 0xe4, 0xf3, 0x6a]],
                                                                 [[0xa2, 0xa7, 0xe2, 0xb3],
                                                                  [0x8f, 0x88, 0x85, 0x9e],
                                                                  [0x67, 0x65, 0x3a, 0x5e],
                                                                  [0xf0, 0xf2, 0xe5, 0x7c]],
                                                                 [[0x26, 0x55, 0xc3, 0x3b],
                                                                  [0xc1, 0xb1, 0x30, 0x51],
                                                                  [0x63, 0x16, 0xd2, 0xe2],
                                                                  [0xec, 0x9e, 0x57, 0x7c]],
                                                                 [[0x8b, 0xfb, 0x6d, 0x22],
                                                                  [0x7b, 0x09, 0x88, 0x5e],
                                                                  [0x67, 0x91, 0x9b, 0x1a],
                                                                  [0xa6, 0x20, 0xab, 0x4b]],
                                                                 [[0xc5, 0x36, 0x79, 0xa9],
                                                                  [0x29, 0xa8, 0x2e, 0xd5],
                                                                  [0xa2, 0x53, 0x43, 0xf7],
                                                                  [0xd9, 0x5a, 0xcb, 0xa9]],
                                                                 [[0x59, 0x8e, 0x48, 0x2f],
                                                                  [0xff, 0xae, 0xe3, 0x64],
                                                                  [0x3a, 0x98, 0x9a, 0xcd],
                                                                  [0x13, 0x30, 0xb4, 0x18]]])),
    ("000102030405060708090a0b0c0d0e0f1011121314151617", np.array([
                                                                 [[0x00, 0x01, 0x02, 0x03],
                                                                  [0x04, 0x05, 0x06, 0x07],
                                                                  [0x08, 0x09, 0x0a, 0x0b],
                                                                  [0x0c, 0x0d, 0x0e, 0x0f]],
                                                                 [[0x10, 0x11, 0x12, 0x13],
                                                                  [0x14, 0x15, 0x16, 0x17],
                                                                  [0x58, 0x46, 0xf2, 0xf9],
                                                                  [0x5c, 0x43, 0xf4, 0xfe]],
                                                                 [[0x54, 0x4a, 0xfe, 0xf5],
                                                                  [0x58, 0x47, 0xf0, 0xfa],
                                                                  [0x48, 0x56, 0xe2, 0xe9],
                                                                  [0x5c, 0x43, 0xf4, 0xfe]],
                                                                 [[0x40, 0xf9, 0x49, 0xb3],
                                                                  [0x1c, 0xba, 0xbd, 0x4d],
                                                                  [0x48, 0xf0, 0x43, 0xb8],
                                                                  [0x10, 0xb7, 0xb3, 0x42]],
                                                                 [[0x58, 0xe1, 0x51, 0xab],
                                                                  [0x04, 0xa2, 0xa5, 0x55],
                                                                  [0x7e, 0xff, 0xb5, 0x41],
                                                                  [0x62, 0x45, 0x08, 0x0c]],
                                                                 [[0x2a, 0xb5, 0x4b, 0xb4],
                                                                  [0x3a, 0x02, 0xf8, 0xf6],
                                                                  [0x62, 0xe3, 0xa9, 0x5d],
                                                                  [0x66, 0x41, 0x0c, 0x08]],
                                                                 [[0xf5, 0x01, 0x85, 0x72],
                                                                  [0x97, 0x44, 0x8d, 0x7e],
                                                                  [0xbd, 0xf1, 0xc6, 0xca],
                                                                  [0x87, 0xf3, 0x3e, 0x3c]],
                                                                 [[0xe5, 0x10, 0x97, 0x61],
                                                                  [0x83, 0x51, 0x9b, 0x69],
                                                                  [0x34, 0x15, 0x7c, 0x9e],
                                                                  [0xa3, 0x51, 0xf1, 0xe0]],
                                                                 [[0x1e, 0xa0, 0x37, 0x2a],
                                                                  [0x99, 0x53, 0x09, 0x16],
                                                                  [0x7c, 0x43, 0x9e, 0x77],
                                                                  [0xff, 0x12, 0x05, 0x1e]],
                                                                 [[0xdd, 0x7e, 0x0e, 0x88],
                                                                  [0x7e, 0x2f, 0xff, 0x68],
                                                                  [0x60, 0x8f, 0xc8, 0x42],
                                                                  [0xf9, 0xdc, 0xc1, 0x54]],
                                                                 [[0x85, 0x9f, 0x5f, 0x23],
                                                                  [0x7a, 0x8d, 0x5a, 0x3d],
                                                                  [0xc0, 0xc0, 0x29, 0x52],
                                                                  [0xbe, 0xef, 0xd6, 0x3a]],
                                                                 [[0xde, 0x60, 0x1e, 0x78],
                                                                  [0x27, 0xbc, 0xdf, 0x2c],
                                                                  [0xa2, 0x23, 0x80, 0x0f],
                                                                  [0xd8, 0xae, 0xda, 0x32]],
                                                                 [[0xa4, 0x97, 0x0a, 0x33],
                                                                  [0x1a, 0x78, 0xdc, 0x09],
                                                                  [0xc4, 0x18, 0xc2, 0x71],
                                                                  [0xe3, 0xa4, 0x1d, 0x5d]]])),
])
def test_aes_key_expansion_192bit(key, r_keys):
    assert np.array_equal(AES.key_expand(key), r_keys)


# 256-bit key tests
@pytest.mark.parametrize("key,r_keys", [
    ("0000000000000000000000000000000000000000000000000000000000000000", np.array([
                                                                                 [[0x00, 0x00, 0x00, 0x00],
                                                                                  [0x00, 0x00, 0x00, 0x00],
                                                                                  [0x00, 0x00, 0x00, 0x00],
                                                                                  [0x00, 0x00, 0x00, 0x00]],
                                                                                 [[0x00, 0x00, 0x00, 0x00],
                                                                                  [0x00, 0x00, 0x00, 0x00],
                                                                                  [0x00, 0x00, 0x00, 0x00],
                                                                                  [0x00, 0x00, 0x00, 0x00]],
                                                                                 [[0x62, 0x63, 0x63, 0x63],
                                                                                  [0x62, 0x63, 0x63, 0x63],
                                                                                  [0x62, 0x63, 0x63, 0x63],
                                                                                  [0x62, 0x63, 0x63, 0x63]],
                                                                                 [[0xaa, 0xfb, 0xfb, 0xfb],
                                                                                  [0xaa, 0xfb, 0xfb, 0xfb],
                                                                                  [0xaa, 0xfb, 0xfb, 0xfb],
                                                                                  [0xaa, 0xfb, 0xfb, 0xfb]],
                                                                                 [[0x6f, 0x6c, 0x6c, 0xcf],
                                                                                  [0x0d, 0x0f, 0x0f, 0xac],
                                                                                  [0x6f, 0x6c, 0x6c, 0xcf],
                                                                                  [0x0d, 0x0f, 0x0f, 0xac]],
                                                                                 [[0x7d, 0x8d, 0x8d, 0x6a],
                                                                                  [0xd7, 0x76, 0x76, 0x91],
                                                                                  [0x7d, 0x8d, 0x8d, 0x6a],
                                                                                  [0xd7, 0x76, 0x76, 0x91]],
                                                                                 [[0x53, 0x54, 0xed, 0xc1],
                                                                                  [0x5e, 0x5b, 0xe2, 0x6d],
                                                                                  [0x31, 0x37, 0x8e, 0xa2],
                                                                                  [0x3c, 0x38, 0x81, 0x0e]],
                                                                                 [[0x96, 0x8a, 0x81, 0xc1],
                                                                                  [0x41, 0xfc, 0xf7, 0x50],
                                                                                  [0x3c, 0x71, 0x7a, 0x3a],
                                                                                  [0xeb, 0x07, 0x0c, 0xab]],
                                                                                 [[0x9e, 0xaa, 0x8f, 0x28],
                                                                                  [0xc0, 0xf1, 0x6d, 0x45],
                                                                                  [0xf1, 0xc6, 0xe3, 0xe7],
                                                                                  [0xcd, 0xfe, 0x62, 0xe9]],
                                                                                 [[0x2b, 0x31, 0x2b, 0xdf],
                                                                                  [0x6a, 0xcd, 0xdc, 0x8f],
                                                                                  [0x56, 0xbc, 0xa6, 0xb5],
                                                                                  [0xbd, 0xbb, 0xaa, 0x1e]],
                                                                                 [[0x64, 0x06, 0xfd, 0x52],
                                                                                  [0xa4, 0xf7, 0x90, 0x17],
                                                                                  [0x55, 0x31, 0x73, 0xf0],
                                                                                  [0x98, 0xcf, 0x11, 0x19]],
                                                                                 [[0x6d, 0xbb, 0xa9, 0x0b],
                                                                                  [0x07, 0x76, 0x75, 0x84],
                                                                                  [0x51, 0xca, 0xd3, 0x31],
                                                                                  [0xec, 0x71, 0x79, 0x2f]],
                                                                                 [[0xe7, 0xb0, 0xe8, 0x9c],
                                                                                  [0x43, 0x47, 0x78, 0x8b],
                                                                                  [0x16, 0x76, 0x0b, 0x7b],
                                                                                  [0x8e, 0xb9, 0x1a, 0x62]],
                                                                                 [[0x74, 0xed, 0x0b, 0xa1],
                                                                                  [0x73, 0x9b, 0x7e, 0x25],
                                                                                  [0x22, 0x51, 0xad, 0x14],
                                                                                  [0xce, 0x20, 0xd4, 0x3b]],
                                                                                 [[0x10, 0xf8, 0x0a, 0x17],
                                                                                  [0x53, 0xbf, 0x72, 0x9c],
                                                                                  [0x45, 0xc9, 0x79, 0xe7],
                                                                                  [0xcb, 0x70, 0x63, 0x85]]])),
    ("ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff", np.array([
                                                                                 [[0xff, 0xff, 0xff, 0xff],
                                                                                  [0xff, 0xff, 0xff, 0xff],
                                                                                  [0xff, 0xff, 0xff, 0xff],
                                                                                  [0xff, 0xff, 0xff, 0xff]],
                                                                                 [[0xff, 0xff, 0xff, 0xff],
                                                                                  [0xff, 0xff, 0xff, 0xff],
                                                                                  [0xff, 0xff, 0xff, 0xff],
                                                                                  [0xff, 0xff, 0xff, 0xff]],
                                                                                 [[0xe8, 0xe9, 0xe9, 0xe9],
                                                                                  [0x17, 0x16, 0x16, 0x16],
                                                                                  [0xe8, 0xe9, 0xe9, 0xe9],
                                                                                  [0x17, 0x16, 0x16, 0x16]],
                                                                                 [[0x0f, 0xb8, 0xb8, 0xb8],
                                                                                  [0xf0, 0x47, 0x47, 0x47],
                                                                                  [0x0f, 0xb8, 0xb8, 0xb8],
                                                                                  [0xf0, 0x47, 0x47, 0x47]],
                                                                                 [[0x4a, 0x49, 0x49, 0x65],
                                                                                  [0x5d, 0x5f, 0x5f, 0x73],
                                                                                  [0xb5, 0xb6, 0xb6, 0x9a],
                                                                                  [0xa2, 0xa0, 0xa0, 0x8c]],
                                                                                 [[0x35, 0x58, 0x58, 0xdc],
                                                                                  [0xc5, 0x1f, 0x1f, 0x9b],
                                                                                  [0xca, 0xa7, 0xa7, 0x23],
                                                                                  [0x3a, 0xe0, 0xe0, 0x64]],
                                                                                 [[0xaf, 0xa8, 0x0a, 0xe5],
                                                                                  [0xf2, 0xf7, 0x55, 0x96],
                                                                                  [0x47, 0x41, 0xe3, 0x0c],
                                                                                  [0xe5, 0xe1, 0x43, 0x80]],
                                                                                 [[0xec, 0xa0, 0x42, 0x11],
                                                                                  [0x29, 0xbf, 0x5d, 0x8a],
                                                                                  [0xe3, 0x18, 0xfa, 0xa9],
                                                                                  [0xd9, 0xf8, 0x1a, 0xcd]],
                                                                                 [[0xe6, 0x0a, 0xb7, 0xd0],
                                                                                  [0x14, 0xfd, 0xe2, 0x46],
                                                                                  [0x53, 0xbc, 0x01, 0x4a],
                                                                                  [0xb6, 0x5d, 0x42, 0xca]],
                                                                                 [[0xa2, 0xec, 0x6e, 0x65],
                                                                                  [0x8b, 0x53, 0x33, 0xef],
                                                                                  [0x68, 0x4b, 0xc9, 0x46],
                                                                                  [0xb1, 0xb3, 0xd3, 0x8b]],
                                                                                 [[0x9b, 0x6c, 0x8a, 0x18],
                                                                                  [0x8f, 0x91, 0x68, 0x5e],
                                                                                  [0xdc, 0x2d, 0x69, 0x14],
                                                                                  [0x6a, 0x70, 0x2b, 0xde]],
                                                                                 [[0xa0, 0xbd, 0x9f, 0x78],
                                                                                  [0x2b, 0xee, 0xac, 0x97],
                                                                                  [0x43, 0xa5, 0x65, 0xd1],
                                                                                  [0xf2, 0x16, 0xb6, 0x5a]],
                                                                                 [[0xfc, 0x22, 0x34, 0x91],
                                                                                  [0x73, 0xb3, 0x5c, 0xcf],
                                                                                  [0xaf, 0x9e, 0x35, 0xdb],
                                                                                  [0xc5, 0xee, 0x1e, 0x05]],
                                                                                 [[0x06, 0x95, 0xed, 0x13],
                                                                                  [0x2d, 0x7b, 0x41, 0x84],
                                                                                  [0x6e, 0xde, 0x24, 0x55],
                                                                                  [0x9c, 0xc8, 0x92, 0x0f]],
                                                                                 [[0x54, 0x6d, 0x42, 0x4f],
                                                                                  [0x27, 0xde, 0x1e, 0x80],
                                                                                  [0x88, 0x40, 0x2b, 0x5b],
                                                                                  [0x4d, 0xae, 0x35, 0x5e]]])),
    ("000102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f", np.array([
                                                                                 [[0x00, 0x01, 0x02, 0x03],
                                                                                  [0x04, 0x05, 0x06, 0x07],
                                                                                  [0x08, 0x09, 0x0a, 0x0b],
                                                                                  [0x0c, 0x0d, 0x0e, 0x0f]],
                                                                                 [[0x10, 0x11, 0x12, 0x13],
                                                                                  [0x14, 0x15, 0x16, 0x17],
                                                                                  [0x18, 0x19, 0x1a, 0x1b],
                                                                                  [0x1c, 0x1d, 0x1e, 0x1f]],
                                                                                 [[0xa5, 0x73, 0xc2, 0x9f],
                                                                                  [0xa1, 0x76, 0xc4, 0x98],
                                                                                  [0xa9, 0x7f, 0xce, 0x93],
                                                                                  [0xa5, 0x72, 0xc0, 0x9c]],
                                                                                 [[0x16, 0x51, 0xa8, 0xcd],
                                                                                  [0x02, 0x44, 0xbe, 0xda],
                                                                                  [0x1a, 0x5d, 0xa4, 0xc1],
                                                                                  [0x06, 0x40, 0xba, 0xde]],
                                                                                 [[0xae, 0x87, 0xdf, 0xf0],
                                                                                  [0x0f, 0xf1, 0x1b, 0x68],
                                                                                  [0xa6, 0x8e, 0xd5, 0xfb],
                                                                                  [0x03, 0xfc, 0x15, 0x67]],
                                                                                 [[0x6d, 0xe1, 0xf1, 0x48],
                                                                                  [0x6f, 0xa5, 0x4f, 0x92],
                                                                                  [0x75, 0xf8, 0xeb, 0x53],
                                                                                  [0x73, 0xb8, 0x51, 0x8d]],
                                                                                 [[0xc6, 0x56, 0x82, 0x7f],
                                                                                  [0xc9, 0xa7, 0x99, 0x17],
                                                                                  [0x6f, 0x29, 0x4c, 0xec],
                                                                                  [0x6c, 0xd5, 0x59, 0x8b]],
                                                                                 [[0x3d, 0xe2, 0x3a, 0x75],
                                                                                  [0x52, 0x47, 0x75, 0xe7],
                                                                                  [0x27, 0xbf, 0x9e, 0xb4],
                                                                                  [0x54, 0x07, 0xcf, 0x39]],
                                                                                 [[0x0b, 0xdc, 0x90, 0x5f],
                                                                                  [0xc2, 0x7b, 0x09, 0x48],
                                                                                  [0xad, 0x52, 0x45, 0xa4],
                                                                                  [0xc1, 0x87, 0x1c, 0x2f]],
                                                                                 [[0x45, 0xf5, 0xa6, 0x60],
                                                                                  [0x17, 0xb2, 0xd3, 0x87],
                                                                                  [0x30, 0x0d, 0x4d, 0x33],
                                                                                  [0x64, 0x0a, 0x82, 0x0a]],
                                                                                 [[0x7c, 0xcf, 0xf7, 0x1c],
                                                                                  [0xbe, 0xb4, 0xfe, 0x54],
                                                                                  [0x13, 0xe6, 0xbb, 0xf0],
                                                                                  [0xd2, 0x61, 0xa7, 0xdf]],
                                                                                 [[0xf0, 0x1a, 0xfa, 0xfe],
                                                                                  [0xe7, 0xa8, 0x29, 0x79],
                                                                                  [0xd7, 0xa5, 0x64, 0x4a],
                                                                                  [0xb3, 0xaf, 0xe6, 0x40]],
                                                                                 [[0x25, 0x41, 0xfe, 0x71],
                                                                                  [0x9b, 0xf5, 0x00, 0x25],
                                                                                  [0x88, 0x13, 0xbb, 0xd5],
                                                                                  [0x5a, 0x72, 0x1c, 0x0a]],
                                                                                 [[0x4e, 0x5a, 0x66, 0x99],
                                                                                  [0xa9, 0xf2, 0x4f, 0xe0],
                                                                                  [0x7e, 0x57, 0x2b, 0xaa],
                                                                                  [0xcd, 0xf8, 0xcd, 0xea]],
                                                                                 [[0x24, 0xfc, 0x79, 0xcc],
                                                                                  [0xbf, 0x09, 0x79, 0xe9],
                                                                                  [0x37, 0x1a, 0xc2, 0x3c],
                                                                                  [0x6d, 0x68, 0xde, 0x36]]])),
])
def test_aes_key_expansion_256bit(key, r_keys):
    assert np.array_equal(AES.key_expand(key), r_keys)
