"""
Build {content_type}.json for non-doc content type (blog, tutorial, example, guide)

Combines all individual content JSON files from MyST build output into a single
manifest file for SvelteKit to consume (e.g., blog.json, tutorial.json).
"""

from __future__ import annotations

from typing import TYPE_CHECKING

if TYPE_CHECKING:
    from afterpython._typing import tContentType

import json
from pathlib import Path

import click

import afterpython as ap
from afterpython.const import CONTENT_TYPES


def standardize_data(content_type: tContentType, content: dict) -> dict:
    """
    Standardize the content dictionary to a single level.
    """
    from afterpython.tools._afterpython import read_afterpython

    afterpython = read_afterpython()
    website = afterpython.get("website", {})
    content_type_config = website.get(content_type, {})
    featured_post = content_type_config.get("featured_post", "")

    frontmatter = content.get("frontmatter", {})
    # e.g. featured_post = "blog4.md" or "blog4"
    is_featured = content["slug"] == featured_post or content["location"].endswith(
        featured_post
    )
    return {
        "version": content["version"],
        "slug": content["slug"],
        "location": content["location"],
        "featured": is_featured,
        # frontmatter fields
        "title": frontmatter.get("title", ""),
        "description": frontmatter.get("description", ""),
        "tags": frontmatter.get("tags", []),
        "date": frontmatter.get("date", ""),
        "thumbnail": frontmatter.get("thumbnail", ""),
        "thumbnailOptimized": frontmatter.get("thumbnailOptimized", ""),
        "authors": frontmatter.get("authors", []),
    }


def convert_paths(content_type: tContentType, output_file: Path):
    """
    Convert paths (generated by MyST build) in the content_json file to use the new paths in the build output.
    e.g. convert /blog_thumbnail-08d4f63a34eb7b55b6a00c25218c3295.png to /blog/build/blog_thumbnail-08d4f63a34eb7b55b6a00c25218c3295.png
    """
    with open(output_file, encoding="utf-8") as f:
        content = json.load(f)

    for item in content:
        if not item.get("thumbnailOptimized"):
            item["thumbnailOptimized"] = item["thumbnail"]
        item["thumbnail"] = f"/{content_type}/build" + item["thumbnail"]
        item["thumbnailOptimized"] = (
            f"/{content_type}/build" + item["thumbnailOptimized"]
        )

    with open(output_file, "w", encoding="utf-8") as f:
        json.dump(content, f, indent=2, ensure_ascii=False)

    click.echo(f"Completed path conversion in {output_file}")


def build_content_json():
    """Build {content_type}.json manifest for non-doc content types.

    Reads all JSON files from afterpython/_build/{content_type}/site/content/
    (excluding index.json) and combines them into a single manifest file at
    afterpython/_build/{content_type}.json.

    This manifest is used by SvelteKit landing pages to list all content items.

    Args:
        content_type: Content type (blog, tutorial, example, guide)
    """
    for content_type in CONTENT_TYPES:
        if content_type == "doc":
            continue  # Doc doesn't need a combined manifest

        content_path = ap.paths.afterpython_path / content_type
        content_dir = content_path / "_build" / "site" / "content"
        myst_yml_path = content_path / "myst.yml"
        if (
            not content_path.exists()
            or not myst_yml_path.exists()
            or not content_dir.exists()
        ):
            click.echo(f"Skipping {content_type}.json (missing files)")
            continue

        click.echo(f"Building {content_type}.json...")

        # Collect all JSON files except index.json, which is created by a placeholder index.md file
        content_items = []
        for json_file in content_dir.glob("*.json"):
            if json_file.stem == "index":
                continue  # Skip index.json

            try:
                with open(json_file, encoding="utf-8") as f:
                    data = json.load(f)

                # Preserve all data from MyST JSON files
                # The project-website-template will parse the fields it needs
                content_items.append(standardize_data(content_type, data))
            except (json.JSONDecodeError, KeyError) as e:
                click.echo(f"Warning: Failed to parse {json_file}: {e}")
                continue

        # Write combined manifest
        output_file = ap.paths.build_path / f"{content_type}.json"
        with open(output_file, "w", encoding="utf-8") as f:
            json.dump(content_items, f, indent=2, ensure_ascii=False)

        click.echo(f"Generated {output_file} with {len(content_items)} items")

        convert_paths(content_type, output_file)
