import logging
import httpx
import re
from datetime import datetime

from typing import Dict, Optional, overload

from agentbox.api.client.models import SandboxADB, InstanceAuthInfo
from agentbox.api.client.types import Unset
from agentbox.connection_config import ConnectionConfig, ProxyTypes
from agentbox.envd.api import ENVD_API_HEALTH_ROUTE, handle_envd_api_exception
from agentbox.exceptions import SandboxException, format_request_timeout_error
from agentbox.sandbox.main import SandboxSetup
from agentbox.sandbox.utils import class_method_variant
# from agentbox.sandbox_sync.adb_shell.adb_shell import ADBShell
from agentbox.sandbox_sync.adb_shell2.adb_shell2 import ADBShell2
from agentbox.sandbox_sync.filesystem.filesystem import Filesystem
from agentbox.sandbox_sync.commands.command import Commands
from agentbox.sandbox_sync.commands.pty import Pty
from agentbox.sandbox_sync.sandbox_api import SandboxApi, SandboxInfo
from agentbox.sandbox_sync.commands_ssh.command_ssh import SSHCommands
from agentbox.sandbox_sync.filesystem_ssh.filesystem_ssh import SSHSyncFilesystem
from agentbox.sandbox_sync.commands_ssh2.command_ssh2 import SSHCommands2

logger = logging.getLogger(__name__)


class TransportWithLogger(httpx.HTTPTransport):
    def handle_request(self, request):
        url = f"{request.url.scheme}://{request.url.host}{request.url.path}"
        logger.info(f"Request: {request.method} {url}")
        response = super().handle_request(request)

        # data = connect.GzipCompressor.decompress(response.read()).decode()
        logger.info(f"Response: {response.status_code} {url}")

        return response


class Sandbox(SandboxSetup, SandboxApi):
    """
    E2B cloud sandbox is a secure and isolated cloud environment.

    The sandbox allows you to:
    - Access Linux OS
    - Create, list, and delete files and directories
    - Run commands
    - Run isolated code
    - Access the internet

    Check docs [here](https://agentbox.dev/docs).

    Use the `Sandbox()` to create a new sandbox.

    Example:
    ```python
    from agentbox import Sandbox

    sandbox = Sandbox()
    ```
    """

    @property
    def files(self) -> Filesystem:
        """
        Module for interacting with the sandbox filesystem.
        """
        return self._filesystem

    @property
    def commands(self) -> Commands:
        """
        Module for running commands in the sandbox.
        """
        return self._commands

    @property
    def adb_shell(self) -> ADBShell2:
        """
        Module for adb shell in the sandbox.
        """
        return self._adb_shell
    
    @property
    def adb_shell2(self) -> ADBShell2:
        """
        Module for adb shell2 in the sandbox.
        """
        return self._adb_shell2

    @property
    def pty(self) -> Pty:
        """
        Module for interacting with the sandbox pseudo-terminal.
        """
        return self._pty

    @property
    def sandbox_id(self) -> str:
        """
        Unique identifier of the sandbox
        """
        return self._sandbox_id

    @property
    def envd_api_url(self) -> str:
        return self._envd_api_url

    @property
    def _envd_access_token(self) -> str:
        """Private property to access the envd token"""
        return self.__envd_access_token

    @_envd_access_token.setter
    def _envd_access_token(self, value: Optional[str]):
        """Private setter for envd token"""
        self.__envd_access_token = value

    @property
    def connection_config(self) -> ConnectionConfig:
        return self._connection_config

    def __init__(
        self,
        template: Optional[str] = None,
        timeout: Optional[int] = None,
        metadata: Optional[Dict[str, str]] = None,
        envs: Optional[Dict[str, str]] = None,
        secure: Optional[bool] = None,
        api_key: Optional[str] = None,
        domain: Optional[str] = None,
        debug: Optional[bool] = None,
        sandbox_id: Optional[str] = None,
        request_timeout: Optional[float] = None,
        proxy: Optional[ProxyTypes] = None,
    ):
        """
        Create a new sandbox.

        By default, the sandbox is created from the default `base` sandbox template.

        :param template: Sandbox template name or ID
        :param timeout: Timeout for the sandbox in **seconds**, default to 300 seconds. Maximum time a sandbox can be kept alive is 24 hours (86_400 seconds) for Pro users and 1 hour (3_600 seconds) for Hobby users
        :param metadata: Custom metadata for the sandbox
        :param envs: Custom environment variables for the sandbox
        :param api_key: E2B API Key to use for authentication, defaults to `AGENTBOX_API_KEY` environment variable
        :param request_timeout: Timeout for the request in **seconds**
        :param proxy: Proxy to use for the request and for the **requests made to the returned sandbox**

        :return: sandbox instance for the new sandbox
        """
        super().__init__()

        if sandbox_id and (metadata is not None or template is not None):
            raise SandboxException(
                "Cannot set metadata or timeout when connecting to an existing sandbox. "
                "Use Sandbox.connect method instead.",
            )

        connection_headers = {}

        if debug:
            self._sandbox_id = "debug_sandbox_id"
            self._envd_version = None
            self._envd_access_token = None
            self._ssh_host = "127.0.0.1"
            self._ssh_port = 22
            self._ssh_username = "debug"
            self._ssh_password = "debug"
            self._adb_info = SandboxADB(
                adb_auth_command="adb shell",
                auth_password="debug",
                connect_command="adb connect 127.0.0.1",
                expire_time="",
                forwarder_command="",
                instance_no="debug_sandbox_id"
            )
        elif sandbox_id is not None:
            response = SandboxApi.get_info(sandbox_id=sandbox_id, api_key=api_key, domain=domain)

            self._sandbox_id = sandbox_id
            self._envd_version = response.envd_version
            self._envd_access_token = response._envd_access_token

            if response._envd_access_token is not None and not isinstance(
                    response._envd_access_token, Unset
            ):
                connection_headers["X-Access-Token"] = response._envd_access_token

        else:
            template = template or self.default_template
            timeout = timeout or self.default_sandbox_timeout
            response = SandboxApi._create_sandbox(
                template=template,
                api_key=api_key,
                timeout=timeout,
                metadata=metadata,
                env_vars=envs,
                domain=domain,
                debug=debug,
                request_timeout=request_timeout,
                secure=secure or False,
                proxy=proxy,
            )
            self._sandbox_id = response.sandbox_id
            self._envd_version = response.envd_version

            if response.envd_access_token is not None and not isinstance(
                response.envd_access_token, Unset
            ):
                self._envd_access_token = response.envd_access_token
                connection_headers["X-Access-Token"] = response.envd_access_token
            else:
                self._envd_access_token = None

        self._transport = TransportWithLogger(limits=self._limits, proxy=proxy)
        self._connection_config = ConnectionConfig(
            api_key=api_key,
            domain=domain,
            debug=debug,
            request_timeout=request_timeout,
            headers=connection_headers,
            proxy=proxy,
        )

        # 根据 sandbox id 进行区分 commands 类型
        if "brd" in self._sandbox_id.lower():
            # ssh info
            ssh_info = SandboxApi._get_ssh(
                sandbox_id=self._sandbox_id,
                api_key=api_key,
                domain=domain,
                debug=debug,
                request_timeout=request_timeout,
                proxy=proxy,
            )
            # print("ssh_info:", ssh_info)
            # Parse SSH connection details from the connect command
            pattern = r'ssh\s+-p\s+(\d+).*?\s+([^@\s]+)@([\w\.-]+)'
            ssh_match = re.search(pattern, ssh_info.connect_command)
            if ssh_match:
                self._ssh_port = int(ssh_match.group(1))
                self._ssh_username = ssh_match.group(2)
                self._ssh_host = ssh_match.group(3)
                self._ssh_password = ssh_info.auth_password
            else:
                raise Exception("Could not parse SSH connection details")
            # Get adb connection details
            self._adb_info = SandboxApi._get_adb(
                sandbox_id=self._sandbox_id,
                api_key=api_key,
                domain=domain,
                debug=debug,
                proxy=proxy,
            )
            # self._watch_commands = SSHCommands(
            #     self._ssh_host,
            #     self._ssh_port,
            #     self._ssh_username,
            #     self._ssh_password,
            #     self.connection_config,
            # )
            # self._commands = SSHCommands(
            #     self._ssh_host,
            #     self._ssh_port,
            #     self._ssh_username,
            #     self._ssh_password,
            #     self.connection_config,
            # )
            self._watch_commands = SSHCommands2(
                self._ssh_host,
                self._ssh_port,
                self._ssh_username,
                self._ssh_password,
                self.connection_config,
            )
            self._commands = SSHCommands2(
                self._ssh_host,
                self._ssh_port,
                self._ssh_username,
                self._ssh_password,
                self.connection_config,
            )
            self._filesystem = SSHSyncFilesystem(
                self._ssh_host,
                self._ssh_port,
                self._ssh_username,
                self._ssh_password,
                self.connection_config,
                self._commands,
                self._watch_commands,
            )
            # self._adb_shell = ADBShell(
            #     forwarder_command=self._adb_info.forwarder_command,
            #     connect_command=self._adb_info.connect_command,
            #     adb_auth_command=self._adb_info.adb_auth_command,
            #     adb_auth_password=self._adb_info.auth_password,
            # )
            self._adb_shell2 = ADBShell2(
                connection_config=self.connection_config,
                sandbox_id=self._sandbox_id
            )
            self._adb_shell = self._adb_shell2
        else:  
            self._envd_api_url = f"{'http' if self.connection_config.debug else 'https'}://{self.get_host(self.envd_port)}"
            self._envd_api = httpx.Client(
                base_url=self.envd_api_url,
                transport=self._transport,
                headers=self.connection_config.headers,
            )

            self._filesystem = Filesystem(
                self.envd_api_url,
                self._envd_version,
                self.connection_config,
                self._transport._pool,
                self._envd_api,
            )
            self._commands = Commands(
                self.envd_api_url,
                self.connection_config,
                self._transport._pool,
            )
            self._pty = Pty(
                self.envd_api_url,
                self.connection_config,
                self._transport._pool,
            )

    def is_running(self, request_timeout: Optional[float] = None) -> bool:
        """
        Check if the sandbox is running.

        :param request_timeout: Timeout for the request in **seconds**

        :return: `True` if the sandbox is running, `False` otherwise

        Example
        ```python
        sandbox = Sandbox()
        sandbox.is_running() # Returns True

        sandbox.kill()
        sandbox.is_running() # Returns False
        ```
        """
        try:
            r = self._envd_api.get(
                ENVD_API_HEALTH_ROUTE,
                timeout=self.connection_config.get_request_timeout(request_timeout),
            )

            if r.status_code == 502:
                return False

            err = handle_envd_api_exception(r)

            if err:
                raise err

        except httpx.TimeoutException:
            raise format_request_timeout_error()

        return True

    @classmethod
    def connect(
        cls,
        sandbox_id: str,
        api_key: Optional[str] = None,
        domain: Optional[str] = None,
        debug: Optional[bool] = None,
        proxy: Optional[ProxyTypes] = None,
    ):
        """
        Connects to an existing Sandbox.
        With sandbox ID you can connect to the same sandbox from different places or environments (serverless functions, etc).

        :param sandbox_id: Sandbox ID
        :param api_key: E2B API Key to use for authentication, defaults to `AGENTBOX_API_KEY` environment variable
        :param proxy: Proxy to use for the request and for the **requests made to the returned sandbox**

        :return: sandbox instance for the existing sandbox

        @example
        ```python
        sandbox = Sandbox()
        sandbox_id = sandbox.sandbox_id

        # Another code block
        same_sandbox = Sandbox.connect(sandbox_id)
        ```
        """

        return cls(
            sandbox_id=sandbox_id,
            api_key=api_key,
            domain=domain,
            debug=debug,
            proxy=proxy,
        )

    def __enter__(self):
        return self

    def __exit__(self, exc_type, exc_value, traceback):
        self.kill()

    @overload
    def kill(self, request_timeout: Optional[float] = None) -> bool:
        """
        Kill the sandbox.

        :param request_timeout: Timeout for the request in **seconds**

        :return: `True` if the sandbox was killed, `False` if the sandbox was not found
        """
        ...

    @overload
    @staticmethod
    def kill(
        sandbox_id: str,
        api_key: Optional[str] = None,
        domain: Optional[str] = None,
        debug: Optional[bool] = None,
        request_timeout: Optional[float] = None,
        proxy: Optional[ProxyTypes] = None,
    ) -> bool:
        """
        Kill the sandbox specified by sandbox ID.

        :param sandbox_id: Sandbox ID
        :param api_key: E2B API Key to use for authentication, defaults to `AGENTBOX_API_KEY` environment variable
        :param request_timeout: Timeout for the request in **seconds**
        :param proxy: Proxy to use for the request

        :return: `True` if the sandbox was killed, `False` if the sandbox was not found
        """
        ...

    @class_method_variant("_cls_kill")
    def kill(self, request_timeout: Optional[float] = None) -> bool:  # type: ignore
        """
        Kill the sandbox.

        :param request_timeout: Timeout for the request
        :return: `True` if the sandbox was killed, `False` if the sandbox was not found
        """
        config_dict = self.connection_config.__dict__
        config_dict.pop("access_token", None)
        config_dict.pop("api_url", None)

        if request_timeout:
            config_dict["request_timeout"] = request_timeout

        return SandboxApi._cls_kill(
            sandbox_id=self.sandbox_id,
            **config_dict,
        )

    @overload
    def set_timeout(
        self,
        timeout: int,
        request_timeout: Optional[float] = None,
    ) -> None:
        """
        Set the timeout of the sandbox.
        After the timeout expires the sandbox will be automatically killed.
        This method can extend or reduce the sandbox timeout set when creating the sandbox or from the last call to `.set_timeout`.

        Maximum time a sandbox can be kept alive is 24 hours (86_400 seconds) for Pro users and 1 hour (3_600 seconds) for Hobby users.

        :param timeout: Timeout for the sandbox in **seconds**
        :param request_timeout: Timeout for the request in **seconds**
        """
        ...

    @overload
    @staticmethod
    def set_timeout(
        sandbox_id: str,
        timeout: int,
        api_key: Optional[str] = None,
        domain: Optional[str] = None,
        debug: Optional[bool] = None,
        request_timeout: Optional[float] = None,
        proxy: Optional[ProxyTypes] = None,
    ) -> None:
        """
        Set the timeout of the sandbox specified by sandbox ID.
        After the timeout expires the sandbox will be automatically killed.
        This method can extend or reduce the sandbox timeout set when creating the sandbox or from the last call to `.set_timeout`.

        Maximum time a sandbox can be kept alive is 24 hours (86_400 seconds) for Pro users and 1 hour (3_600 seconds) for Hobby users.

        :param sandbox_id: Sandbox ID
        :param timeout: Timeout for the sandbox in **seconds**
        :param api_key: E2B API Key to use for authentication, defaults to `AGENTBOX_API_KEY` environment variable
        :param request_timeout: Timeout for the request in **seconds**
        :param proxy: Proxy to use for the request
        """
        ...

    @class_method_variant("_cls_set_timeout")
    def set_timeout(  # type: ignore
        self,
        timeout: int,
        request_timeout: Optional[float] = None,
    ) -> None:
        config_dict = self.connection_config.__dict__
        config_dict.pop("access_token", None)
        config_dict.pop("api_url", None)

        if request_timeout:
            config_dict["request_timeout"] = request_timeout

        SandboxApi._cls_set_timeout(
            sandbox_id=self.sandbox_id,
            timeout=timeout,
            **config_dict,
        )

    def get_info(  # type: ignore
        self,
        request_timeout: Optional[float] = None,
    ) -> SandboxInfo:
        """
        Get sandbox information like sandbox ID, template, metadata, started at/end at date.
        :param request_timeout: Timeout for the request in **seconds**
        :return: Sandbox info
        """
        config_dict = self.connection_config.__dict__
        config_dict.pop("access_token", None)
        config_dict.pop("api_url", None)

        if request_timeout:
            config_dict["request_timeout"] = request_timeout

        return SandboxApi.get_info(
            sandbox_id=self.sandbox_id,
            **config_dict,
        )
    
    def get_instance_no(  # type: ignore
        self,
        request_timeout: Optional[float] = None,
    ) -> str:
        """
        Get sandbox instance number.
        :param request_timeout: Timeout for the request in **seconds**
        :return: Sandbox instance number
        """
        config_dict = self.connection_config.__dict__
        config_dict.pop("access_token", None)
        config_dict.pop("api_url", None)

        if request_timeout:
            config_dict["request_timeout"] = request_timeout

        return SandboxApi.get_instance_no(
            sandbox_id=self.sandbox_id,
            **config_dict,
        )
    
    def get_instance_auth_info(  # type: ignore
        self,
        valid_time: Optional[int] = 3600,
        request_timeout: Optional[float] = None,
    ) -> InstanceAuthInfo:
        """
        Get sandbox instance auth info.
        :param request_timeout: Timeout for the request in **seconds**
        :return: Sandbox instance auth info
        """
        config_dict = self.connection_config.__dict__
        config_dict.pop("access_token", None)
        config_dict.pop("api_url", None)

        if request_timeout:
            config_dict["request_timeout"] = request_timeout

        return SandboxApi.get_instance_auth_info(
            sandbox_id=self.sandbox_id,
            valid_time=valid_time,
            **config_dict,
        )

    @classmethod
    def resume(
        cls,
        sandbox_id: str,
        timeout: Optional[int] = None,
        api_key: Optional[str] = None,
        domain: Optional[str] = None,
        debug: Optional[bool] = None,
        request_timeout: Optional[float] = None,
    ):
        """
        Resume the sandbox.

        The **default sandbox timeout of 300 seconds** will be used for the resumed sandbox.
        If you pass a custom timeout via the `timeout` parameter, it will be used instead.

        :param sandbox_id: sandbox ID
        :param timeout: Timeout for the sandbox in **seconds**
        :param api_key: E2B API Key to use for authentication
        :param domain: Domain of the sandbox server
        :param debug: Enable debug mode
        :param request_timeout: Timeout for the request in **seconds**

        :return: A running sandbox instance
        """

        timeout = timeout or cls.default_sandbox_timeout

        SandboxApi._cls_resume(
            sandbox_id=sandbox_id,
            request_timeout=request_timeout,
            timeout=timeout,
            api_key=api_key,
            domain=domain,
            debug=debug,
        )

        return cls.connect(
            sandbox_id=sandbox_id,
            api_key=api_key,
            domain=domain,
            debug=debug,
        )

    @overload
    def pause(
        self,
        request_timeout: Optional[float] = None,
    ) -> str:
        """
        Pause the sandbox.

        :param request_timeout: Timeout for the request in **seconds**

        :return: sandbox ID that can be used to resume the sandbox
        """
        ...

    @overload
    @staticmethod
    def pause(
        sandbox_id: str,
        api_key: Optional[str] = None,
        domain: Optional[str] = None,
        debug: Optional[bool] = None,
        request_timeout: Optional[float] = None,
    ) -> str:
        """
        Pause a sandbox by its ID.

        :param sandbox_id: Sandbox ID
        :param api_key: E2B API Key to use for authentication
        :param domain: Domain of the sandbox server
        :param debug: Enable debug mode
        :param request_timeout: Timeout for the request in **seconds**

        :return: sandbox ID that can be used to resume the sandbox
        """
        ...

    @class_method_variant("_cls_pause")
    def pause( # type: ignore
        self,
        request_timeout: Optional[float] = None,
    ) -> str:
        """
        Pause the sandbox.

        :param request_timeout: Timeout for the request in **seconds**

        :return: sandbox ID that can be used to resume the sandbox
        """
        SandboxApi._cls_pause(
            sandbox_id=self.sandbox_id,
            api_key=self.connection_config.api_key,
            domain=self.connection_config.domain,
            debug=self.connection_config.debug,
            request_timeout=request_timeout,
        )

        return self.sandbox_id