# Copyright (c) 2025 Beijing Volcano Engine Technology Co., Ltd. and/or its affiliates.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

from typing import Dict

from agentkit.client import BaseAgentkitClient
from agentkit.utils import get_logger

# 导入自动生成的类型
from agentkit.knowledge.knowledge_all_types import (
    ListKnowledgeBasesRequest,
    ListKnowledgeBasesResponse,
    AddKnowledgeBaseRequest,
    AddKnowledgeBaseResponse,
    GetKnowledgeConnectionInfoRequest,
    GetKnowledgeConnectionInfoResponse,
    DeleteKnowledgeBaseRequest,
    DeleteKnowledgeBaseResponse,
)

logger = get_logger(__name__)


class AgentkitKnowledge(BaseAgentkitClient):
    """AgentKit Knowledge Base Management Service"""
    
    # Define all API actions for this service
    API_ACTIONS: Dict[str, str] = {
        "ListKnowledgeBases": "ListKnowledgeBases",
        "AddKnowledgeBase": "AddKnowledgeBase",
        "GetKnowledgeConnectionInfo": "GetKnowledgeConnectionInfo",
        "DeleteKnowledgeBase": "DeleteKnowledgeBase",
    }
    
    def __init__(
        self,
        access_key: str = "",
        secret_key: str = "",
        region: str = "",
        session_token: str = "",
    ) -> None:
        super().__init__(
            access_key=access_key,
            secret_key=secret_key,
            region=region,
            session_token=session_token,
            service_name="knowledge",
        )

    def list_knowledge_bases(
        self, request: ListKnowledgeBasesRequest
    ) -> ListKnowledgeBasesResponse:
        """List all knowledge bases from Volcengine."""
        return self._invoke_api(
            api_action="ListKnowledgeBases",
            request=request,
            response_type=ListKnowledgeBasesResponse,
        )

    def add_knowledge_base(
        self, request: AddKnowledgeBaseRequest
    ) -> AddKnowledgeBaseResponse:
        """Add a new knowledge base to Volcengine."""
        return self._invoke_api(
            api_action="AddKnowledgeBase",
            request=request,
            response_type=AddKnowledgeBaseResponse,
        )

    def get_knowledge_connection_info(
        self, request: GetKnowledgeConnectionInfoRequest
    ) -> GetKnowledgeConnectionInfoResponse:
        """Get connection information for a specific knowledge base."""
        return self._invoke_api(
            api_action="GetKnowledgeConnectionInfo",
            request=request,
            response_type=GetKnowledgeConnectionInfoResponse,
        )

    def delete_knowledge_base(
        self, request: DeleteKnowledgeBaseRequest
    ) -> DeleteKnowledgeBaseResponse:
        """Delete a knowledge base from Volcengine."""
        return self._invoke_api(
            api_action="DeleteKnowledgeBase",
            request=request,
            response_type=DeleteKnowledgeBaseResponse,
        )


if __name__ == "__main__":
    # list knowledge bases
    knowledge = AgentkitKnowledge()
    req = ListKnowledgeBasesRequest(page_number=1, page_size=10)
    res = knowledge.list_knowledge_bases(req)
    print(res)
    