# Copyright (c) 2025 Beijing Volcano Engine Technology Co., Ltd. and/or its affiliates.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import json
from typing import Dict

from agentkit.client import BaseAgentkitClient
from agentkit.utils import get_logger

# 导入自动生成的类型（强类型版本）
from agentkit.runtime.runtime_all_types import (
    GetRuntimeRequest,
    GetRuntimeResponse,
    ListRuntimesRequest,
    ListRuntimesResponse,
    UpdateRuntimeRequest,
    UpdateRuntimeResponse,
    DeleteRuntimeRequest,
    DeleteRuntimeResponse,
    ReleaseRuntimeRequest,
    ReleaseRuntimeResponse,
    GetRuntimeVersionRequest,
    GetRuntimeVersionResponse,
    ListRuntimeVersionsRequest,
    ListRuntimeVersionsResponse,
    GetRuntimeCozeTokenRequest,
    GetRuntimeCozeTokenResponse,
    ListRuntimeCrRegistriesRequest,
    ListRuntimeCrRegistriesResponse,
    CreateRuntimeRequest,
    CreateRuntimeResponse,
)

logger = get_logger(__name__)


class AgentkitRuntime(BaseAgentkitClient):
    """AgentKit Runtime Management Service"""
    
    # Define all API actions for this service
    API_ACTIONS: Dict[str, str] = {
        "CreateRuntime": "CreateRuntime",
        "GetRuntime": "GetRuntime",
        "UpdateRuntime": "UpdateRuntime",
        "DeleteRuntime": "DeleteRuntime",
        "ListRuntimes": "ListRuntimes",
        "ReleaseRuntime": "ReleaseRuntime",
        "GetRuntimeVersion": "GetRuntimeVersion",
        "ListRuntimeVersions": "ListRuntimeVersions",
        "GetRuntimeCozeToken": "GetRuntimeCozeToken",
        "ListRuntimeCrRegistries": "ListRuntimeCrRegistries",
    }
    
    def __init__(
        self,
        access_key: str = "",
        secret_key: str = "",
        region: str = "",
        session_token: str = "",
    ) -> None:
        super().__init__(
            access_key=access_key,
            secret_key=secret_key,
            region=region,
            session_token=session_token,
            service_name="runtime",
        )

    def create(
        self, request: CreateRuntimeRequest
    ) -> CreateRuntimeResponse:
        """Create a new AgentKit Runtime on Volcengine."""
        return self._invoke_api(
            api_action="CreateRuntime",
            request=request,
            response_type=CreateRuntimeResponse,
        )

    def get(self, request: GetRuntimeRequest) -> GetRuntimeResponse:
        """Get details of a specific AgentKit Runtime from Volcengine."""
        logger.debug("Get runtime request: %s", json.dumps(request.model_dump(by_alias=True)))
        try:
            resp = self._invoke_api(
                api_action="GetRuntime",
                request=request,
                response_type=GetRuntimeResponse,
            )
        except Exception as e:
            if "InvalidAgentKitRuntime.NotFound" in str(e):
                return None
            raise Exception(f"Failed to get agentkit runtime: {str(e)}")
        return resp

    def update(
        self, request: UpdateRuntimeRequest
    ) -> UpdateRuntimeResponse:
        """Update an existing AgentKit Runtime on Volcengine."""
        logger.debug("Update runtime request: %s", json.dumps(request.model_dump(by_alias=True, exclude_none=True)))
        return self._invoke_api(
            api_action="UpdateRuntime",
            request=request,
            response_type=UpdateRuntimeResponse,
        )

    def delete(
        self, request: DeleteRuntimeRequest
    ) -> DeleteRuntimeResponse:
        """Delete an AgentKit Runtime on Volcengine."""
        return self._invoke_api(
            api_action="DeleteRuntime",
            request=request,
            response_type=DeleteRuntimeResponse,
        )

    def list(
        self, request: ListRuntimesRequest
    ) -> ListRuntimesResponse:
        """List all AgentKit Runtimes from Volcengine."""
        return self._invoke_api(
            api_action="ListRuntimes",
            request=request,
            response_type=ListRuntimesResponse,
        )

    def release(
        self, request: ReleaseRuntimeRequest
    ) -> ReleaseRuntimeResponse:
        """Release an AgentKit Runtime on Volcengine."""
        return self._invoke_api(
            api_action="ReleaseRuntime",
            request=request,
            response_type=ReleaseRuntimeResponse,
        )

    def get_version(
        self, request: GetRuntimeVersionRequest
    ) -> GetRuntimeVersionResponse:
        """Get the version of a specific AgentKit Runtime."""
        return self._invoke_api(
            api_action="GetRuntimeVersion",
            request=request,
            response_type=GetRuntimeVersionResponse,
        )

    def list_versions(
        self, request: ListRuntimeVersionsRequest
    ) -> ListRuntimeVersionsResponse:
        """List all versions of a specific AgentKit Runtime."""
        return self._invoke_api(
            api_action="ListRuntimeVersions",
            request=request,
            response_type=ListRuntimeVersionsResponse,
        )

    def get_coze_token(
        self, request: GetRuntimeCozeTokenRequest
    ) -> GetRuntimeCozeTokenResponse:
        """Get Coze JWT token for a specific AgentKit Runtime."""
        return self._invoke_api(
            api_action="GetRuntimeCozeToken",
            request=request,
            response_type=GetRuntimeCozeTokenResponse,
        )

    def list_cr_registries(
        self, request: ListRuntimeCrRegistriesRequest
    ) -> ListRuntimeCrRegistriesResponse:
        """List container registry repositories for Runtime."""
        return self._invoke_api(
            api_action="ListRuntimeCrRegistries",
            request=request,
            response_type=ListRuntimeCrRegistriesResponse,
        )


if __name__ == "__main__":
    # list all runtimes
    runtime = AgentkitRuntime()
    runtimes = runtime.list(ListRuntimesRequest())
    print(runtimes)
    