# Copyright (c) 2025 Beijing Volcano Engine Technology Co., Ltd. and/or its affiliates.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"""AgentKit Configuration Manager"""

from pathlib import Path
from typing import Dict, Any, Optional, Union
import os
import yaml
from .auto_prompt import generate_config_from_dataclass
from .dataclass_utils import AutoSerializableMixin
from dataclasses import dataclass, field
from .docker_build_config import DockerBuildConfig

WORKFLOW_NAME_IN_YAML = "launch_types"
GLOBAL_CONFIG_FILE_PATH = ""

def get_global_config_file_path() -> str:
    """Get global configuration file path"""
    return GLOBAL_CONFIG_FILE_PATH

def set_global_config_file_path(config_path: str) -> None:
    """Set global configuration file path"""
    global GLOBAL_CONFIG_FILE_PATH
    GLOBAL_CONFIG_FILE_PATH = config_path

@dataclass
class CommonConfig(AutoSerializableMixin):
    """Common configuration - automatic prompt generation support"""
    agent_name: str = field(
        default="", 
        metadata={
            "description": "Agent application name", 
            "icon": "🤖",
            "validation": {
                "required": True,
                "pattern": r"^[a-zA-Z0-9_\-.]+$",
                "message": "Only letters, numbers, underscore(_), hyphen(-) and dot(.) are allowed"
            }
        }
    )
    entry_point: str = field(
        default="", 
        metadata={
            "description": "Agent application entry file (path allowed), e.g. simple_agent.py", 
            "icon": "📝",
            "validation": {
                "required": True,
                # allow relative paths with directories and either .py or .go endings
                "pattern": r"^[\w\-/\.]+?\.(py)$",
                "message": "Must be a Python (.py), containing only letters, numbers, underscore(_), hyphen(-), dot(.) and '/' for directories"
            }
        }
    )
    description: str = field(default="", metadata={"description": "Application description", "icon": "📄"})
    language: str = field(default="Python", metadata={
        "description": "Agent application Language, defaults to Python",
        "icon": "✏️",
        "choices": [
            {"value": "Python", "description": "Python"},
            # {"value": "Golang", "description": "Go (Golang)"}
        ]
    })
    language_version: str = field(
        default="", 
        metadata={
            "description": "Agent application Language version", 
            "icon": "🐍",
            "validation": {
                "type": "conditional",
                "depends_on": "language",
                "rules": {
                    "Python": {
                        "choices": ["3.10", "3.11", "3.12", "3.13"],
                        "message": "Python 版本必须是 3.10, 3.11, 3.12, 3.13 之一"
                    },
                    # "Golang": {
                    #     "pattern": r"^1\.\d+$",
                    #     "message": "Go 版本格式应为 '1.x' (如 1.21, 1.23)",
                    #     "hint": "[格式: 1.x]"
                    # }
                }
            }
        }
    )
    agent_type: str = field(default="Basic App", metadata={"description" : "Agent application Type", "icon": "📩", "hidden": True})
    dependencies_file: str = field(default="requirements.txt", metadata={"description": "Agent application Dependencies file", "icon": "📦"})
    runtime_envs: Dict[str, str] = field(
        default_factory=dict,
        metadata={
            "description": "Application-level runtime environment variables (shared across all deployment modes)",
            "icon": "🔐",
            "hidden": False  # 在交互式配置中显示
        }
    )
    launch_type: str = field(
        default="local",
        metadata={
            "description": "Deployment and runtime mode, defaults to local (local build and deploy), optional hybrid (local build, cloud deploy)",
            "icon": "🚀",
            "choices": [
                {"value": "local", "description": "Local build and deploy"},
                {"value": "hybrid", "description": "Local build, cloud deploy"},
                {"value": "cloud", "description": "Cloud build and deploy base on Volcano Engine Agentkit Platform"}
            ]
        }
    )
    
    _config_metadata = {
        'name': '基础配置',
        'welcome_message': '欢迎使用 AgentKit 配置向导',
        'next_step_hint': '本向导将帮助您完成Agent应用配置，请根据提示输入相关信息，或直接按Enter键使用默认值。',
        'completion_message': '太棒了！通用配置已完成！',
        'next_action_hint': '下面将开始针对您选择的部署模式进行配置。'
    }
    
    @staticmethod
    def _recommended_for_language(language: str) -> Dict[str, str]:
        """Return recommended language_version and dependencies_file for supported languages."""
        mapping = {
            "python": {"language_version": "3.12", "dependencies_file": "requirements.txt"},
            "golang": {"language_version": "1.23", "dependencies_file": "go.mod"},
        }
        return mapping.get((language or "python").lower(), mapping["python"])
    
    def __post_init__(self):
        """
        Ensure language-specific defaults for language_version and dependencies_file.
        Heuristic: only replace when current values look like the other language's defaults
        or are empty/unset to avoid overwriting explicit user choices.
        """
        rec = self._recommended_for_language(self.language)
        # Normalize current values
        lv = (self.language_version or "").strip()
        df = (self.dependencies_file or "").strip()
        lang = (self.language or "python").lower()
    
        # Update language_version when empty or clearly mismatched
        if not lv:
            self.language_version = rec["language_version"]
        else:
            # if language is go but version looks like python default -> replace
            if lang == "golang" and (lv.startswith("3.") or lv == "3.12"):
                self.language_version = rec["language_version"]
            # if language is python but version looks like go default -> replace
            if lang == "python" and lv.startswith("1."):
                self.language_version = rec["language_version"]
    
        # Update dependencies_file when empty or clearly mismatched
        if not df:
            self.dependencies_file = rec["dependencies_file"]
        else:
            if lang == "golang" and df == "requirements.txt":
                self.dependencies_file = rec["dependencies_file"]
            if lang == "python" and df == "go.mod":
                self.dependencies_file = rec["dependencies_file"]
    
    def set_language(self, language: str):
        """
        Change language and apply recommended language_version and dependencies_file
        when user hasn't provided custom values (uses same heuristic as __post_init__).
        """
        self.language = language
        rec = self._recommended_for_language(language)
        lv = (self.language_version or "").strip()
        df = (self.dependencies_file or "").strip()
    
        if not lv or lv in ("3.12", "1.23") or lv.startswith("3.") and language.lower() == "go":
            self.language_version = rec["language_version"]
        if not df or df in ("requirements.txt", "go.mod"):
            self.dependencies_file = rec["dependencies_file"]
    
    @classmethod
    def interactive_create(cls, existing_config: Optional[Dict[str, Any]] = None) -> "CommonConfig":
        """Generate interactive configuration based on dataclass"""
        from .auto_prompt import auto_prompt
        existing = cls.from_dict(existing_config or {})
        config_dict = auto_prompt.generate_config(cls, existing.to_dict())
        return cls.from_dict(config_dict)

class ConfigUpdateResult:
    """Configuration update result for "return and rewrite" mode"""
    
    def __init__(self):
        self.updates: Dict[str, Any] = {}
    
    def add_update(self, key_path: str, value: Any):
        """Add configuration update, supports arbitrary level key paths"""
        self.updates[key_path] = value
    
    def add_common_update(self, key: str, value: Any):
        """Add common configuration update"""
        self.add_update(f"common.{key}", value)
    
    def add_workflow_update(self, workflow_name: str, key: str, value: Any):
        """Add workflow configuration update"""
        self.add_update(f"workflows.{workflow_name}.{key}", value)
    
    def has_updates(self) -> bool:
        """Check if there are updates needed"""
        return bool(self.updates)
    
    def get_updates(self) -> Dict[str, Any]:
        """Get all updates"""
        return self.updates


class AgentkitConfigManager:
    """Agentkit configuration manager - fully dynamic, no predefined workflow structure"""
    
    def __init__(self, config_path: Optional[Union[str, Path]] = None):
        if config_path is None or config_path == "":
            config_path = Path.cwd() / "agentkit.yaml"
        self.config_path = Path(config_path)
        self._data: Dict[str, Any] = {}
        self._load_config()
    
    @classmethod
    def from_dict(
        cls,
        config_dict: Dict[str, Any],
        base_config_path: Optional[Union[str, Path]] = None
    ) -> 'AgentkitConfigManager':
        """Create config manager from dictionary.
        
        This method supports creating a configuration manager directly from a
        dictionary, optionally merging with an existing configuration file.
        
        Args:
            config_dict: Configuration dictionary. Should contain 'common' and
                        optionally 'launch_types' keys matching the YAML structure.
            base_config_path: Optional path to existing config file to use as base.
                            If provided and exists, config_dict will be deep-merged
                            into the base configuration.
        
        Returns:
            AgentkitConfigManager instance with the specified configuration.
            Note: This instance is NOT saved to disk automatically.
        
        Example:
            >>> # Create from pure dict
            >>> config = AgentkitConfigManager.from_dict({
            ...     "common": {
            ...         "agent_name": "my_agent",
            ...         "launch_type": "hybrid"
            ...     }
            ... })
            >>> 
            >>> # Create by merging with existing config
            >>> config = AgentkitConfigManager.from_dict(
            ...     config_dict={"common": {"launch_type": "hybrid"}},
            ...     base_config_path="agentkit.yaml"
            ... )
        """
        # Create instance without loading from file
        instance = cls.__new__(cls)
        instance.config_path = Path(base_config_path) if base_config_path else Path("agentkit.yaml")
        
        if base_config_path and Path(base_config_path).exists():
            # Load base config and merge with config_dict
            with open(base_config_path, 'r', encoding='utf-8') as f:
                base_data = yaml.safe_load(f) or {}
            instance._data = instance._deep_merge(base_data, config_dict)
        else:
            # Use config_dict directly, ensuring required structure
            if "common" not in config_dict:
                # If no common config provided, use defaults
                config_dict = {
                    "common": CommonConfig().to_dict(),
                    WORKFLOW_NAME_IN_YAML: config_dict.get(WORKFLOW_NAME_IN_YAML, {})
                }
            instance._data = config_dict
            
            # Ensure launch_types key exists
            if WORKFLOW_NAME_IN_YAML not in instance._data:
                instance._data[WORKFLOW_NAME_IN_YAML] = {}
        
        return instance
    
    def _load_config(self):
        """Load configuration file"""
        if self.config_path.exists():
            with open(self.config_path, 'r', encoding='utf-8') as f:
                self._data = yaml.safe_load(f) or {}
        else:
            self._data = self._get_default_config()
            self._save_config()
    
    def _get_default_config(self) -> Dict[str, Any]:
        """Get default configuration - contains only common config and empty workflow container"""
        return {
            "common": CommonConfig().to_dict(),
            WORKFLOW_NAME_IN_YAML: {},  # Fully dynamic, no predefined workflows
            "docker_build": {}  # Optional Docker build configuration
        }
    
    def _save_config(self):
        """Save configuration file"""
        os.makedirs(self.config_path.parent, exist_ok=True)
        with open(self.config_path, 'w', encoding='utf-8') as f:
            yaml.dump(self._data, f, default_flow_style=False, sort_keys=False)
    
    def get_common_config(self) -> CommonConfig:
        """Get common configuration"""
        return CommonConfig.from_dict(self._data.get("common", {}))
    
    def update_common_config(self, config: CommonConfig):
        """Update common configuration"""
        self._data["common"] = config.to_dict()
        self._save_config()
    
    def get_docker_build_config(self) -> DockerBuildConfig:
        """Get Docker build configuration"""
        return DockerBuildConfig.from_dict(self._data.get("docker_build", {}))
    
    def update_docker_build_config(self, config: DockerBuildConfig):
        """Update Docker build configuration"""
        self._data["docker_build"] = config.to_dict()
        self._save_config()
    
    def set_docker_build_runtime_param(self, key: str, value: Any):
        """Set runtime parameter for docker_build (not persisted to file)
        
        Args:
            key: Parameter name
            value: Parameter value
        """
        if "docker_build" not in self._data:
            self._data["docker_build"] = {}
        self._data["docker_build"][key] = value
    
    def get_workflow_config(self, workflow_name: str) -> Dict[str, Any]:
        """Get specified workflow configuration (fully dynamic, no structure validation)"""
        return self._data.get(WORKFLOW_NAME_IN_YAML, {}).get(workflow_name, {})
    
    def update_workflow_config(self, workflow_name: str, config: Dict[str, Any]):
        """Update workflow configuration (fully dynamic, no structure validation)
        
        使用深度合并策略：
        - 保留原有的系统字段（如 runtime_id, runtime_endpoint 等）
        - 更新用户配置的字段
        - 新增的字段会被添加
        """
        if WORKFLOW_NAME_IN_YAML not in self._data:
            self._data[WORKFLOW_NAME_IN_YAML] = {}
        
        # 获取现有配置
        existing_config = self._data[WORKFLOW_NAME_IN_YAML].get(workflow_name, {})
        
        # 深度合并：新配置覆盖旧配置，但保留旧配置中新配置没有的字段
        merged_config = self._deep_merge(existing_config, config)
        
        self._data[WORKFLOW_NAME_IN_YAML][workflow_name] = merged_config
        self._save_config()
    
    def _deep_merge(self, base: Dict[str, Any], update: Dict[str, Any]) -> Dict[str, Any]:
        """深度合并两个字典
        
        Args:
            base: 基础字典（旧配置）
            update: 更新字典（新配置）
        
        Returns:
            合并后的字典
        """
        result = base.copy()
        
        for key, value in update.items():
            if key in result and isinstance(result[key], dict) and isinstance(value, dict):
                # 递归合并嵌套字典
                result[key] = self._deep_merge(result[key], value)
            else:
                # 直接更新值
                result[key] = value
        
        return result
    
    def list_workflows(self) -> list[str]:
        """List all configured workflow names"""
        return list(self._data.get(WORKFLOW_NAME_IN_YAML, {}).keys())
    
    def workflow_exists(self, workflow_name: str) -> bool:
        """Check if workflow exists"""
        return workflow_name in self._data.get(WORKFLOW_NAME_IN_YAML, {})
    
    def apply_updates(self, update_result: ConfigUpdateResult):
        """Apply update results for "return and rewrite" mode"""
        if not update_result.has_updates():
            return
        
        updates = update_result.get_updates()
        
        for key_path, value in updates.items():
            keys = key_path.split('.')
            current = self._data
            
            # Navigate to parent level
            for key in keys[:-1]:
                if key not in current:
                    current[key] = {}
                current = current[key]
            
            # Set final value
            current[keys[-1]] = value
        
        self._save_config()
    
    def get_config_path(self) -> Path:
        """Get configuration file path"""
        return self.config_path
    
    def reload(self):
        """Reload configuration"""
        self._load_config()
    
    def reset_to_default(self):
        """Reset to default configuration"""
        self._data = self._get_default_config()
        self._save_config()
    
    def get_raw_data(self) -> Dict[str, Any]:
        """Get raw configuration data (for debugging or advanced operations)"""
        return self._data.copy()
    
    def set_raw_value(self, key_path: str, value: Any):
        """Directly set value at any path (advanced use case)"""
        keys = key_path.split('.')
        current = self._data
        
        for key in keys[:-1]:
            if key not in current:
                current[key] = {}
            current = current[key]
        
        current[keys[-1]] = value
        self._save_config()
    
    def get_raw_value(self, key_path: str, default: Any = None) -> Any:
        """Get value at any path (advanced use case)"""
        keys = key_path.split('.')
        current = self._data
        
        try:
            for key in keys:
                current = current[key]
            return current
        except (KeyError, TypeError):
            return default

# Config instance cache for singleton pattern
_config_instances: Dict[str, AgentkitConfigManager] = {}


def get_config(config_path: Optional[str] = None, force_reload: bool = False) -> AgentkitConfigManager:
    """
    Get configuration instance with singleton pattern.
    
    Args:
        config_path: Path to configuration file. If None, uses global config path.
        force_reload: If True, reload configuration from disk even if cached.
        
    Returns:
        AgentkitConfigManager instance.
        
    Note:
        This implements a singleton pattern per config path to ensure
        different parts of the codebase share the same Config instance,
        avoiding state inconsistency issues.
    """
    config_path = config_path or get_global_config_file_path()
    
    # Update global config path if provided
    if config_path and config_path != "":
        set_global_config_file_path(config_path)
    
    # Normalize path for cache key
    from pathlib import Path
    cache_key = str(Path(config_path).resolve()) if config_path else "default"
    
    # Return cached instance unless force_reload is True
    if not force_reload and cache_key in _config_instances:
        return _config_instances[cache_key]
    
    # Create new instance and cache it
    instance = AgentkitConfigManager(config_path)
    _config_instances[cache_key] = instance
    
    return instance


def clear_config_cache():
    """
    Clear the config instance cache.
    
    Useful for testing or when you need to ensure
    a fresh configuration is loaded.
    """
    global _config_instances
    _config_instances.clear()

def create_config_update_result() -> ConfigUpdateResult:
    """Create configuration update result instance"""
    return ConfigUpdateResult()