# Copyright (c) 2025 Beijing Volcano Engine Technology Co., Ltd. and/or its affiliates.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"""全局配置模块

提供全局配置文件 (~/.agentkit/config.yaml) 的定义和管理功能。
全局配置用于存储跨项目共享的配置，如火山引擎凭证、CR实例名等。

优先级：
- 环境变量 > 项目配置 > 全局配置 > 默认值
"""

from dataclasses import dataclass, field
from pathlib import Path
from typing import Optional
import yaml
import logging

logger = logging.getLogger(__name__)


@dataclass
class VolcengineCredentials:
    """火山引擎凭证配置
    
    用于存储火山引擎 API 凭证，作为环境变量的备选方案。
    优先级低于环境变量。
    """
    access_key: str = ""
    secret_key: str = ""
    region: str = "cn-beijing"
    
    def to_dict(self):
        return {
            "access_key": self.access_key,
            "secret_key": self.secret_key,
            "region": self.region,
        }
    
    @classmethod
    def from_dict(cls, data: dict):
        return cls(
            access_key=data.get("access_key", ""),
            secret_key=data.get("secret_key", ""),
            region=data.get("region", "cn-beijing"),
        )


@dataclass
class CRGlobalConfig:
    """容器镜像服务 (CR) 全局配置
    
    当项目配置中 cr_instance_name 或 cr_namespace_name 为空或 "Auto" 时，
    将使用这里的全局配置。
    """
    instance_name: str = ""
    namespace_name: str = ""
    
    def to_dict(self):
        return {
            "instance_name": self.instance_name,
            "namespace_name": self.namespace_name,
        }
    
    @classmethod
    def from_dict(cls, data: dict):
        return cls(
            instance_name=data.get("instance_name", ""),
            namespace_name=data.get("namespace_name", ""),
        )


@dataclass
class TOSGlobalConfig:
    """对象存储 (TOS) 全局配置
    
    当项目配置中 tos_bucket、tos_prefix 或 tos_region 为空或 "Auto" 时，
    将使用这里的全局配置。
    """
    bucket: str = ""
    prefix: str = "agentkit-builds"
    region: str = "cn-beijing"
    
    def to_dict(self):
        return {
            "bucket": self.bucket,
            "prefix": self.prefix,
            "region": self.region,
        }
    
    @classmethod
    def from_dict(cls, data: dict):
        return cls(
            bucket=data.get("bucket", ""),
            prefix=data.get("prefix", "agentkit-builds"),
            region=data.get("region", "cn-beijing"),
        )


@dataclass
class GlobalConfig:
    """全局配置
    
    存储在 ~/.agentkit/config.yaml 中的全局配置。
    用于跨项目共享配置，减少重复配置。
    """
    volcengine: VolcengineCredentials = field(default_factory=VolcengineCredentials)
    cr: CRGlobalConfig = field(default_factory=CRGlobalConfig)
    tos: TOSGlobalConfig = field(default_factory=TOSGlobalConfig)
    
    def to_dict(self):
        return {
            "volcengine": self.volcengine.to_dict(),
            "cr": self.cr.to_dict(),
            "tos": self.tos.to_dict(),
        }
    
    @classmethod
    def from_dict(cls, data: dict):
        return cls(
            volcengine=VolcengineCredentials.from_dict(data.get("volcengine", {})),
            cr=CRGlobalConfig.from_dict(data.get("cr", {})),
            tos=TOSGlobalConfig.from_dict(data.get("tos", {})),
        )


class GlobalConfigManager:
    """全局配置管理器
    
    负责加载、保存和管理全局配置文件。
    默认路径：~/.agentkit/config.yaml
    """
    
    DEFAULT_PATH = Path.home() / ".agentkit" / "config.yaml"
    
    def __init__(self, config_path: Optional[Path] = None):
        """初始化全局配置管理器
        
        Args:
            config_path: 配置文件路径，默认为 ~/.agentkit/config.yaml
        """
        self.config_path = config_path or self.DEFAULT_PATH
        self._config: Optional[GlobalConfig] = None
    
    def load(self) -> GlobalConfig:
        """加载全局配置
        
        如果配置文件不存在或加载失败，返回空配置（不抛出异常）。
        
        Returns:
            GlobalConfig 实例
        """
        if not self.config_path.exists():
            logger.debug(f"全局配置文件不存在: {self.config_path}")
            return GlobalConfig()
        
        try:
            with open(self.config_path, 'r', encoding='utf-8') as f:
                data = yaml.safe_load(f) or {}
            logger.debug(f"成功加载全局配置: {self.config_path}")
            return GlobalConfig.from_dict(data)
        except Exception as e:
            logger.debug(f"加载全局配置失败（使用空配置）: {e}")
            return GlobalConfig()
    
    def save(self, config: GlobalConfig):
        """保存全局配置
        
        Args:
            config: 要保存的配置对象
        """
        # 创建目录
        self.config_path.parent.mkdir(parents=True, exist_ok=True)
        
        # 保存配置
        with open(self.config_path, 'w', encoding='utf-8') as f:
            yaml.dump(
                config.to_dict(), 
                f, 
                default_flow_style=False, 
                allow_unicode=True,
                sort_keys=False
            )
        
        # 设置文件权限（仅所有者可读写）
        try:
            self.config_path.chmod(0o600)
        except Exception as e:
            logger.warning(f"设置配置文件权限失败: {e}")
        
        logger.info(f"全局配置已保存: {self.config_path}")
    
    def get_config(self, force_reload: bool = False) -> GlobalConfig:
        """获取配置（带缓存）
        
        Args:
            force_reload: 是否强制重新加载
            
        Returns:
            GlobalConfig 实例
        """
        if self._config is None or force_reload:
            self._config = self.load()
        return self._config
    
    def exists(self) -> bool:
        """检查全局配置文件是否存在
        
        Returns:
            True 如果配置文件存在
        """
        return self.config_path.exists()


# 单例实例
_global_config_manager: Optional[GlobalConfigManager] = None


def get_global_config_manager() -> GlobalConfigManager:
    """获取全局配置管理器（单例）
    
    Returns:
        GlobalConfigManager 实例
    """
    global _global_config_manager
    if _global_config_manager is None:
        _global_config_manager = GlobalConfigManager()
    return _global_config_manager


def get_global_config(force_reload: bool = False) -> GlobalConfig:
    """获取全局配置（单例）
    
    这是最常用的 API，用于获取全局配置。
    
    Args:
        force_reload: 是否强制重新加载
        
    Returns:
        GlobalConfig 实例
        
    Example:
        >>> global_config = get_global_config()
        >>> print(global_config.volcengine.access_key)
        >>> print(global_config.cr.instance_name)
    """
    return get_global_config_manager().get_config(force_reload=force_reload)


def save_global_config(config: GlobalConfig):
    """保存全局配置
    
    Args:
        config: 要保存的配置对象
        
    Example:
        >>> global_config = get_global_config()
        >>> global_config.cr.instance_name = "my-cr-instance"
        >>> save_global_config(global_config)
    """
    get_global_config_manager().save(config)


def global_config_exists() -> bool:
    """检查全局配置文件是否存在
    
    Returns:
        True 如果配置文件存在
    """
    return get_global_config_manager().exists()
