# Copyright (c) 2025 Beijing Volcano Engine Technology Co., Ltd. and/or its affiliates.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"""全局配置解析器

提供全局配置的自动注入功能，在配置对象创建时自动填充未设置的字段。
这个模块被 dataclass_utils.py 中的 from_dict() 方法调用。
"""

from typing import Any, Dict
import logging

logger = logging.getLogger(__name__)


def apply_global_config_defaults(
    config_obj: Any, 
    project_data: Dict[str, Any]
) -> Any:
    """应用全局配置默认值（仅在项目配置为空/Auto时）
    
    这个函数在 from_dict() 创建配置对象后调用，
    自动填充项目配置中未设置的字段。
    
    优先级：
    1. 项目配置（有效值）- 最高优先级，不覆盖
    2. 全局配置（有效值）- 中等优先级，用于填充
    3. 类默认值 - 最低优先级
    
    Args:
        config_obj: 配置对象实例（已通过 from_dict 创建）
        project_data: 项目配置原始字典（当前未使用，保留用于未来扩展）
        
    Returns:
        应用全局配置后的对象
        
    Example:
        >>> # 项目配置中 cr_instance_name 为空
        >>> config = HybridVeAgentkitConfig.from_dict({"cr_instance_name": ""})
        >>> # apply_global_config_defaults 自动从全局配置填充
        >>> # config.cr_instance_name = "my-team-cr-instance"（来自全局配置）
    """
    # 延迟导入，避免循环依赖
    try:
        from .workflow_configs import HybridVeAgentkitConfig, VeAgentkitConfig
        from .global_config import get_global_config
        from .utils import is_valid_config
    except ImportError as e:
        logger.debug(f"无法导入配置模块，跳过全局配置应用: {e}")
        return config_obj
    
    # 只处理工作流配置类
    if not isinstance(config_obj, (HybridVeAgentkitConfig, VeAgentkitConfig)):
        return config_obj
    
    try:
        # 加载全局配置
        global_config = get_global_config()
        
        # 定义字段映射：配置字段 -> (全局配置section, 全局配置字段)
        field_mappings = {
            'cr_instance_name': ('cr', 'instance_name'),
            'cr_namespace_name': ('cr', 'namespace_name'),
        }
        
        # 对于 VeAgentkitConfig，还需要处理 TOS 配置
        if isinstance(config_obj, VeAgentkitConfig):
            field_mappings.update({
                'tos_bucket': ('tos', 'bucket'),
                'tos_prefix': ('tos', 'prefix'),
                'tos_region': ('tos', 'region'),
            })
        
        # 应用全局配置
        for field_name, (section, attr) in field_mappings.items():
            # 检查字段是否存在
            if not hasattr(config_obj, field_name):
                continue
            
            # 获取项目配置值
            project_value = getattr(config_obj, field_name)
            
            # 如果项目配置有效，不覆盖
            if is_valid_config(project_value):
                logger.debug(
                    f"使用项目配置: {field_name} = {project_value}"
                )
                continue
            
            # 从全局配置获取值
            section_obj = getattr(global_config, section, None)
            if section_obj is None:
                continue
            
            global_value = getattr(section_obj, attr, None)
            
            # 如果全局配置有效，应用它
            if global_value and is_valid_config(global_value):
                logger.info(
                    f"应用全局配置: {field_name} = {global_value} "
                    f"(来源: ~/.agentkit/config.yaml)"
                )
                setattr(config_obj, field_name, global_value)
        
    except Exception as e:
        # 全局配置应用失败不应该影响主流程
        logger.debug(f"应用全局配置时出错（已忽略）: {e}")
    
    return config_obj
