# Copyright (c) 2025 Beijing Volcano Engine Technology Co., Ltd. and/or its affiliates.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"""Status query result model."""

from dataclasses import dataclass, field
from typing import Optional, Dict, Any, List
from enum import Enum


class ServiceStatus(str, Enum):
    """Service status enumeration."""
    RUNNING = "running"
    STOPPED = "stopped"
    STARTING = "starting"
    STOPPING = "stopping"
    NOT_DEPLOYED = "not_deployed"
    ERROR = "error"
    UNKNOWN = "unknown"


@dataclass
class StatusResult:
    """Result of a status query."""
    
    success: bool
    """Whether the status query was successful."""
    
    status: ServiceStatus = ServiceStatus.UNKNOWN
    """Current service status."""
    
    container_id: Optional[str] = None
    """Container ID (for local deployments)."""
    
    service_id: Optional[str] = None
    """Service ID (for cloud deployments)."""
    
    endpoint_url: Optional[str] = None
    """Service endpoint URL."""
    
    uptime: Optional[int] = None
    """Service uptime."""
    
    error: Optional[str] = None
    """Error message if query failed."""
    
    error_type: Optional[str] = None  # For compatibility with adapter
    
    details: Dict[str, Any] = field(default_factory=dict)
    """Additional status details."""
    
    def __bool__(self) -> bool:
        """Allow using result in boolean context: if result: ..."""
        return self.success
    
    def is_running(self) -> bool:
        """Check if service is running."""
        return self.status == ServiceStatus.RUNNING
    
    def __str__(self) -> str:
        """String representation."""
        if self.success:
            return f"StatusResult(status={self.status.value}, endpoint={self.endpoint_url})"
        else:
            return f"StatusResult(success=False, error={self.error})"
