# Copyright (c) 2025 Beijing Volcano Engine Technology Co., Ltd. and/or its affiliates.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"""Base service class with common functionality."""

import logging
from typing import Optional, Dict, Any
from pathlib import Path


class BaseService:
    """Base class for all services with common functionality."""
    
    def __init__(self):
        """Initialize base service."""
        self.logger = logging.getLogger(f"{__name__}.{self.__class__.__name__}")
    
    def _load_config(self, config_dict: Optional[Dict[str, Any]], config_file: Optional[str]):
        """
        Load configuration with priority: config_dict > config_file > default.
        
        Args:
            config_dict: Configuration dictionary.
            config_file: Configuration file path.
            
        Returns:
            Configuration object.
            
        Raises:
            FileNotFoundError: If config_file specified but doesn't exist.
            ValueError: If configuration is invalid.
        """
        from ...config import get_config, AgentkitConfigManager
        
        # Priority: config_dict > config_file > default
        if config_dict:
            # Create config from dict, optionally merging with config_file
            if config_file:
                # Merge mode: config_file as base, config_dict as overrides
                config_path = Path(config_file)
                if not config_path.exists():
                    raise FileNotFoundError(f"Configuration file not found: {config_file}")
                self.logger.debug(f"Creating config from dict with base file: {config_file}")
                return AgentkitConfigManager.from_dict(
                    config_dict=config_dict,
                    base_config_path=config_path
                )
            else:
                # Pure dict mode: create config entirely from dict
                self.logger.debug("Creating config from dict (no base file)")
                return AgentkitConfigManager.from_dict(config_dict=config_dict)
        
        # No config_dict, use traditional file loading
        if config_file:
            config_path = Path(config_file)
            if not config_path.exists():
                raise FileNotFoundError(f"Configuration file not found: {config_file}")
            return get_config(config_path=config_path)
        else:
            return get_config()
    
    def _validate_config(self, config) -> None:
        """
        Validate configuration has required fields.
        
        Args:
            config: Configuration object to validate.
            
        Raises:
            ValueError: If configuration is invalid.
        """
        common_config = config.get_common_config()
        
        # Check required fields
        if not common_config.agent_name:
            raise ValueError("Configuration missing required field: agent_name")
        
        if not common_config.entry_point:
            raise ValueError("Configuration missing required field: entry_point")
        
        if not common_config.launch_type:
            raise ValueError("Configuration missing required field: launch_type")
        
        self.logger.debug(f"Configuration validated: agent={common_config.agent_name}, "
                         f"launch_type={common_config.launch_type}")
    
    def _get_workflow(self, workflow_name: str):
        """
        Get workflow instance by name.
        
        Args:
            workflow_name: Workflow name (local/cloud/hybrid).
            
        Returns:
            Workflow instance or None if not found.
            
        Raises:
            ValueError: If workflow not found.
        """
        from ...workflows import WORKFLOW_REGISTRY
        
        if workflow_name not in WORKFLOW_REGISTRY:
            available = ", ".join(WORKFLOW_REGISTRY.keys())
            raise ValueError(
                f"Unknown workflow type '{workflow_name}'. "
                f"Available workflows: {available}"
            )
        
        return WORKFLOW_REGISTRY[workflow_name]()
    
    def _handle_exception(self, operation: str, error: Exception) -> Dict[str, Any]:
        """
        Standardized exception handling.
        
        Args:
            operation: Name of the operation that failed.
            error: The exception that occurred.
            
        Returns:
            Dictionary with error information for result object.
        """
        self.logger.error(f"{operation} error: {error}", exc_info=True)
        
        # Provide more specific error messages for common errors
        error_message = str(error)
        if isinstance(error, FileNotFoundError):
            error_message = f"File not found: {error}"
        elif isinstance(error, ValueError):
            error_message = f"Invalid configuration: {error}"
        elif isinstance(error, PermissionError):
            error_message = f"Permission denied: {error}"
        elif isinstance(error, TimeoutError):
            error_message = f"Operation timed out: {error}"
        
        return {
            "success": False,
            "error": error_message,
            "error_type": type(error).__name__
        }
