# Copyright (c) 2025 Beijing Volcano Engine Technology Co., Ltd. and/or its affiliates.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"""Deploy service - Core business logic for deploying agents."""

from typing import Optional, Dict, Any

from ..models.deploy_result import DeployResult
from ..field_mapper import get_deploy_info
from .base_service import BaseService


class DeployService(BaseService):
    """Service for deploying agents."""
    
    def deploy(
        self,
        config_dict: Optional[Dict[str, Any]] = None,
        config_file: Optional[str] = None
    ) -> DeployResult:
        """
        Deploy agent to target environment.
        
        Args:
            config_dict: Configuration dictionary (highest priority).
            config_file: Configuration file path.
            
        Returns:
            DeployResult: Deploy operation result.
        """
        try:
            # Load and validate configuration
            config = self._load_config(config_dict, config_file)
            self._validate_config(config)
            
            # Get workflow
            common_config = config.get_common_config()
            workflow_name = common_config.launch_type
            workflow_class = self._get_workflow(workflow_name)
            
            # Get workflow config
            workflow_config = config.get_workflow_config(workflow_name)
            
            # Execute deploy
            self.logger.info(f"Starting deploy with workflow '{workflow_name}'")
            
            # Create workflow instance with progress reporting
            from ...workflows.progress import ProgressReporter, create_cli_progress_listener
            progress_reporter = ProgressReporter()
            progress_reporter.add_listener(create_cli_progress_listener())
            
            # Instantiate workflow with dependencies
            if callable(workflow_class):
                workflow_instance = workflow_class(
                    config_manager=config,
                    progress_reporter=progress_reporter,
                    logger=self.logger
                )
            else:
                workflow_instance = workflow_class
                workflow_instance.progress = progress_reporter
                workflow_instance.config_manager = config
                workflow_instance.logger = self.logger
            
            # Call workflow and get DeployInfo
            deploy_info = workflow_instance.deploy(workflow_config)
            
            # Convert DeployInfo to DeployResult
            return DeployResult(
                success=deploy_info.success,
                container_id=deploy_info.container_id,
                container_name=deploy_info.container_name,
                service_id=deploy_info.service_id,
                endpoint_url=deploy_info.endpoint_url,
                deploy_timestamp=deploy_info.deploy_timestamp,
                error=deploy_info.error,
                error_type=str(deploy_info.error_type) if deploy_info.error_type else None,
                metadata={
                    "workflow": workflow_name,
                    "details": deploy_info.details
                }
            )
                
        except Exception as e:
            error_info = self._handle_exception("Deploy", e)
            return DeployResult(**error_info)
