# Copyright (c) 2025 Beijing Volcano Engine Technology Co., Ltd. and/or its affiliates.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"""Invoke service - Core business logic for invoking agents."""

import logging
from typing import Optional, Dict, Any

from ..models.invoke_result import InvokeResult
from .base_service import BaseService


logger = logging.getLogger(__name__)


class InvokeService(BaseService):
    """Service for invoking deployed agents."""
    
    def __init__(self):
        """Initialize invoke service."""
        super().__init__()
    
    def invoke(
        self,
        payload: Dict[str, Any],
        config_dict: Optional[Dict[str, Any]] = None,
        config_file: Optional[str] = None,
        headers: Optional[Dict[str, str]] = None,
        apikey: Optional[str] = None
    ) -> InvokeResult:
        """
        Invoke deployed agent.
        
        Args:
            payload: Request payload to send to agent.
            config_dict: Configuration dictionary (highest priority).
            config_file: Configuration file path.
            headers: Optional HTTP headers.
            apikey: Optional API key for authentication.
            
        Returns:
            InvokeResult: Invocation result.
        """
        try:
            # Load configuration
            config = self._load_config(config_dict, config_file)
            common_config = config.get_common_config()
            
            # Get workflow
            workflow_name = common_config.launch_type
            workflow = self._get_workflow(workflow_name)
            
            # Get workflow config
            workflow_config = config.get_workflow_config(workflow_name)
            
            # Prepare invocation args
            invoke_args = {
                "payload": payload,
                "headers": headers or {},
                "apikey": apikey
            }
            
            # Execute invocation
            self.logger.info(f"Invoking agent with workflow '{workflow_name}'")
            
            # Instantiate workflow with dependencies if it's a class
            if callable(workflow):
                workflow_instance = workflow(config_manager=config, logger=self.logger)
            else:
                workflow_instance = workflow
                workflow_instance.config_manager = config
                workflow_instance.logger = self.logger
            
            # Call workflow and get InvokeInfo
            invoke_info = workflow_instance.invoke(workflow_config, invoke_args)
            
            # Convert InvokeInfo to InvokeResult
            if invoke_info.success:
                return InvokeResult(
                    success=True,
                    response=invoke_info.response,
                    is_streaming=invoke_info.is_streaming,
                    metadata={
                        "workflow": workflow_name,
                        "details": invoke_info.details
                    }
                )
            else:
                return InvokeResult(
                    success=False,
                    error=invoke_info.error or "Invocation failed",
                    error_type=str(invoke_info.error_type) if invoke_info.error_type else None,
                    response=invoke_info.response,
                    metadata={
                        "workflow": workflow_name,
                        "details": invoke_info.details
                    }
                )
                
        except Exception as e:
            error_info = self._handle_exception("Invoke", e)
            return InvokeResult(**error_info)
