# Copyright (c) 2025 Beijing Volcano Engine Technology Co., Ltd. and/or its affiliates.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"""Status service - Core business logic for querying agent status."""

from typing import Optional, Dict, Any

from ..models.status_result import StatusResult, ServiceStatus
from ..field_mapper import get_status_info
from .base_service import BaseService


class StatusService(BaseService):
    """Service for querying agent status."""
    
    def status(
        self,
        config_dict: Optional[Dict[str, Any]] = None,
        config_file: Optional[str] = None
    ) -> StatusResult:
        """
        Query agent runtime status.
        
        Args:
            config_dict: Configuration dictionary (highest priority).
            config_file: Configuration file path.
            
        Returns:
            StatusResult: Status query result.
        """
        try:
            # Load configuration
            config = self._load_config(config_dict, config_file)
            common_config = config.get_common_config()
            
            # Get workflow
            workflow_name = common_config.launch_type
            workflow = self._get_workflow(workflow_name)
            if workflow is None:
                return StatusResult(
                    success=False,
                    error=f"Unknown workflow type '{workflow_name}'"
                )
            
            # Get workflow config
            workflow_config = config.get_workflow_config(workflow_name)
            
            # Query status
            self.logger.info(f"Querying status with workflow '{workflow_name}'")
            
            # Instantiate workflow
            if callable(workflow):
                workflow_instance = workflow(config_manager=config, logger=self.logger)
            else:
                workflow_instance = workflow
                workflow_instance.config_manager = config
                workflow_instance.logger = self.logger
            
            # Call workflow and get StatusInfo
            status_info = workflow_instance.status(workflow_config)
            
            # Map status string to ServiceStatus enum
            status_enum_map = {
                'running': ServiceStatus.RUNNING,
                'stopped': ServiceStatus.STOPPED,
                'starting': ServiceStatus.STARTING,
                'stopping': ServiceStatus.STOPPING,
                'not_deployed': ServiceStatus.NOT_DEPLOYED,
                'error': ServiceStatus.ERROR,
                'unknown': ServiceStatus.UNKNOWN,
            }
            status_enum = status_enum_map.get(status_info.status or 'unknown', ServiceStatus.UNKNOWN)
            
            return StatusResult(
                success=status_info.success,
                status=status_enum,
                container_id=status_info.container_id,
                service_id=status_info.service_id,
                endpoint_url=status_info.endpoint_url,
                uptime=status_info.uptime,
                error=status_info.error,
                error_type=status_info.error_type,
                details=status_info.details
            )
                
        except Exception as e:
            error_info = self._handle_exception("Status query", e)
            return StatusResult(**error_info)
    
    def _parse_status(self, status_info: Any, workflow_name: str) -> StatusResult:
        """
        Parse status information from workflow.
        
        Args:
            status_info: Status information from workflow.
            workflow_name: Workflow name.
            
        Returns:
            StatusResult: Parsed status result.
        """
        # If status_info is a dict, extract fields using FieldMapper
        if isinstance(status_info, dict):
            # Extract and normalize status information
            status_data = get_status_info(status_info)
            
            # Map normalized status string to ServiceStatus enum
            normalized_status = status_data.get('status', 'unknown')
            status_enum_map = {
                'running': ServiceStatus.RUNNING,
                'stopped': ServiceStatus.STOPPED,
                'starting': ServiceStatus.STARTING,
                'stopping': ServiceStatus.STOPPING,
                'error': ServiceStatus.ERROR,
                'unknown': ServiceStatus.UNKNOWN,
            }
            status = status_enum_map.get(normalized_status, ServiceStatus.UNKNOWN)
            
            return StatusResult(
                success=True,
                status=status,
                container_id=status_data.get('container_id'),
                service_id=status_data.get('service_id'),
                endpoint_url=status_data.get('endpoint_url'),
                uptime=status_data.get('uptime'),
                details=status_info
            )
        else:
            # If it's a string or other format, store as details
            return StatusResult(
                success=True,
                status=ServiceStatus.UNKNOWN,
                details={"raw_status": str(status_info)}
            )
