# Copyright (c) 2025 Beijing Volcano Engine Technology Co., Ltd. and/or its affiliates.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"""Base workflow class for AgentKit."""

from abc import ABC, abstractmethod
from typing import Dict, Any, Optional
import logging
from .models import BuildInfo, DeployInfo, InvokeInfo, StatusInfo
from .progress import ProgressReporter
from pathlib import Path



class Workflow(ABC):
    """
    Base workflow class with improved interface.
    
    All methods now return detailed result objects instead of bool,
    providing better error information and operation details.
    
    Subclasses can optionally receive config_manager to avoid
    creating new instances internally (dependency injection).
    """
    
    def __init__(self, 
                 config_manager=None, 
                 progress_reporter: Optional[ProgressReporter] = None,
                 logger=None):
        """
        Initialize workflow.
        
        Args:
            config_manager: Config manager instance (dependency injection).
            progress_reporter: Progress reporter for UI updates.
            logger: Logger instance.
        """
        self.config_manager = config_manager
        self.progress = progress_reporter or ProgressReporter()
        self.logger = logger or logging.getLogger(self.__class__.__name__)
    
    def prompt_for_config(self, current_config: Dict[str, Any]) -> Dict[str, Any]:
        """
        Prompt for configuration (for backward compatibility).
        
        Subclasses should implement this or generate_config.
        """
        # Default implementation delegates to generate_config if it exists
        if hasattr(self, 'generate_config'):
            return self.generate_config(current_config)
        return current_config
    
    def generate_config(self, config: Dict[str, Any]) -> Dict[str, Any]:
        """
        Generate workflow-specific configuration.
        
        Args:
            config: The base configuration.
            
        Returns:
            The workflow-specific configuration.
        """
        pass
    
    @abstractmethod
    def build(self, config: Dict[str, Any]) -> BuildInfo:
        """
        Build the agent image.
        
        Args:
            config: The configuration of the workflow.
            
        Returns:
            BuildInfo with detailed build result.
        """
        pass
    
    @abstractmethod
    def deploy(self, config: Dict[str, Any]) -> DeployInfo:
        """
        Deploy the agent.
        
        Args:
            config: The configuration of the workflow.
            
        Returns:
            DeployInfo with detailed deploy result.
        """
        pass
    
    @abstractmethod
    def invoke(self, config: Dict[str, Any], args: Dict[str, Any]) -> InvokeInfo:
        """
        Invoke the agent.
        
        Args:
            config: The configuration of the workflow.
            args: The arguments for the invocation.
            
        Returns:
            InvokeInfo with response data.
        """
        pass
    
    @abstractmethod
    def status(self, config: Dict[str, Any] = None) -> StatusInfo:
        """
        Query the status of the agent.
        
        Args:
            config: The configuration of the workflow (optional).
                   If None, will load from global config.
            
        Returns:
            StatusInfo with current status.
        """
        pass
    
    @abstractmethod
    def stop(self, config: Dict[str, Any] = None) -> bool:
        """
        Stop the agent runtime (without destroying resources).
        
        Args:
            config: The configuration of the workflow (optional).
                   If None, will load from global config.
            
        Returns:
            True if successful, False otherwise.
        """
        pass
    
    @abstractmethod
    def destroy(self, config: Dict[str, Any] = None, force: bool = False) -> bool:
        """
        Destroy the agent runtime.
        
        Args:
            config: The configuration of the workflow.
            force: Force destroy even if there are errors.
            
        Returns:
            True if successful, False otherwise.
            
        Note:
            This method still returns bool for backward compatibility.
            May be updated to return DestroyInfo in future versions.
        """
        pass
