# Copyright (c) 2025 Beijing Volcano Engine Technology Co., Ltd. and/or its affiliates.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

from agentkit.toolkit.workflows import Workflow
from agentkit.toolkit.workflows.models import BuildInfo, DeployInfo, StatusInfo, InvokeInfo, WorkflowErrorType
from agentkit.toolkit.workflows.progress import ProgressLevel
from typing import Dict, Any, List, Optional, Tuple
from datetime import datetime
import os
import json

from agentkit.toolkit.config import (
    AUTO_CREATE_VE,
    get_config,
    HybridVeAgentkitConfig,
    CommonConfig,
)
from agentkit.toolkit.config.auto_prompt import auto_prompt
from agentkit.toolkit.integrations.services import CRService, CRServiceConfig, DefaultCRConfigCallback
from agentkit.toolkit.integrations.runner import VeAgentkitRuntimeRunner


class HybridVeAgentkitWorkflow(Workflow):
    
    def __init__(self, config_manager=None, progress_reporter=None, logger=None):
        super().__init__(config_manager, progress_reporter, logger)
    
    def prompt_for_config(self, current_config: Dict[str, Any] = None) -> Dict[str, Any]:
        agent_config = get_config()
        common_config = agent_config.get_common_config()
        # 兼容旧字段名
        if current_config.get("cr_instance_name") is None or current_config["cr_instance_name"] == AUTO_CREATE_VE or current_config["cr_instance_name"] == "":
            # 也检查旧字段名
            if current_config.get("ve_cr_instance_name"):
                current_config["cr_instance_name"] = current_config["ve_cr_instance_name"]
            else:
                current_config["cr_instance_name"] = CRService.default_cr_instance_name_template()
        if current_config.get("cr_repo_name") is None or current_config["cr_repo_name"] == AUTO_CREATE_VE or current_config["cr_repo_name"] == "":
            if current_config.get("ve_cr_repo_name"):
                current_config["cr_repo_name"] = current_config["ve_cr_repo_name"]
            else:
                current_config["cr_repo_name"] = common_config.agent_name

        ve_config = auto_prompt.generate_config(HybridVeAgentkitConfig, current_config)
        return ve_config

    def build(self, config: Dict[str, Any]) -> BuildInfo:
        """Build the agent image using LocalDockerBuilder."""
        try:
            from agentkit.toolkit.integrations.builder.local_docker_builder import LocalDockerBuilder, LocalDockerBuilderConfig, LocalDockerBuilderResult
        except ImportError as e:
            self.logger.error(f"Docker dependencies missing: {e}")
            self.progress.report("依赖错误: Docker未安装", 0, level=ProgressLevel.ERROR)
            return BuildInfo(
                success=False,
                error="缺少Docker相关依赖，请安装agentkit[docker] extras",
                error_type=WorkflowErrorType.DEPENDENCY_ERROR
            )
        try:
            hybrid_ve_config = HybridVeAgentkitConfig.from_dict(config)
            # 如果已有远程镜像，可以跳过构建
            # if hybrid_ve_config.cr_image_full_url:
            #     self.logger.info(f"已配置远程镜像: {hybrid_ve_config.cr_image_full_url}，跳过本地构建")
            #     return True
            
            agent_config = get_config()
            common_config = agent_config.get_common_config()

            # 使用LocalDockerBuilderConfig类构建配置，避免硬编码字符串
            builder_config_obj = LocalDockerBuilderConfig(
                common_config=common_config,
                image_name=common_config.agent_name or "agentkit-app",
                image_tag=hybrid_ve_config.image_tag
            )
            builder_config = builder_config_obj.to_dict()
            
            # 添加Docker构建配置（如果存在）
            docker_build_config = agent_config.get_docker_build_config()
            builder_config["docker_build_config"] = docker_build_config.to_dict()

            builder = LocalDockerBuilder()
            success, build_result = builder.build(builder_config)
            result_obj = LocalDockerBuilderResult.from_dict(build_result)            
            if success:
                hybrid_ve_config.full_image_name = result_obj.full_image_name
                hybrid_ve_config.image_id = result_obj.image_id
                hybrid_ve_config.build_timestamp = result_obj.build_timestamp
                
                agent_config.update_workflow_config("hybrid", hybrid_ve_config.to_persist_dict())
                
                self.progress.report("构建完成", 100, level=ProgressLevel.SUCCESS)
                self.logger.info(f"Build successful: {hybrid_ve_config.full_image_name}")
                
                # Extract tag from full image name
                image_tag = None
                if hybrid_ve_config.full_image_name and ':' in hybrid_ve_config.full_image_name:
                    image_tag = hybrid_ve_config.full_image_name.rsplit(':', 1)[1]
                
                return BuildInfo(
                    success=True,
                    image_name=hybrid_ve_config.full_image_name,
                    image_id=hybrid_ve_config.image_id,
                    image_tag=image_tag,
                    build_timestamp=hybrid_ve_config.build_timestamp,
                    details={"workflow": "hybrid", "builder": "LocalDockerBuilder"}
                )
            else:
                error_msg = "镜像构建失败"
                build_logs = result_obj.build_logs or []
                if build_logs:
                    if isinstance(build_logs, list):
                        error_msg = "\n".join([log for log in build_logs if log.strip()])
                    else:
                        error_msg = str(build_logs)
                
                self.logger.error(f"Build failed: {error_msg}")
                self.progress.report("构建失败", 0, level=ProgressLevel.ERROR)
                
                return BuildInfo(
                    success=False,
                    error=error_msg,
                    error_type=WorkflowErrorType.BUILD_FAILURE,
                    details={"build_logs": build_logs}
                )
                
        except Exception as e:
            self.logger.error(f"Build error: {e}", exc_info=True)
            self.progress.report(f"构建错误: {e}", 0, level=ProgressLevel.ERROR)
            return BuildInfo(
                success=False,
                error=str(e),
                error_type=WorkflowErrorType.UNKNOWN_ERROR,
                details={"exception": type(e).__name__}
            )

    def deploy(self, config: Dict[str, Any]) -> DeployInfo:
        """简化后的主部署函数 - 仅负责流程编排"""
        try:
            hybrid_ve_config = HybridVeAgentkitConfig.from_dict(config)
            agent_config = get_config()
            common_config = agent_config.get_common_config()
            
            self.progress.report("开始部署", 0, level=ProgressLevel.INFO)
            
            # 1. 镜像准备阶段
            self.progress.report("准备和推送镜像", 30, level=ProgressLevel.INFO)
            if not self._prepare_and_push_image(hybrid_ve_config, common_config):
                error_msg = "镜像准备或推送失败"
                self.logger.error(error_msg)
                self.progress.report(error_msg, 0, level=ProgressLevel.ERROR)
                return DeployInfo(
                    success=False,
                    error=error_msg,
                    error_type=WorkflowErrorType.DEPLOY_FAILURE
                )
            
            # 2. Runtime 部署阶段  
            self.progress.report("部署 Runtime", 60, level=ProgressLevel.INFO)
            return self._deploy_runtime(hybrid_ve_config, common_config)
            
        except Exception as e:
            self.logger.error(f"Deploy error: {e}", exc_info=True)
            self.progress.report(f"部署错误: {e}", 0, level=ProgressLevel.ERROR)
            return DeployInfo(
                success=False,
                error=str(e),
                error_type=WorkflowErrorType.UNKNOWN_ERROR,
                details={"exception": type(e).__name__}
            )
    
    def _prepare_and_push_image(self, config: HybridVeAgentkitConfig, common_config: CommonConfig) -> bool:
        """镜像准备和推送"""
        # 如果有远程镜像，但没有本地镜像
        if config.cr_image_full_url and not self._check_local_image(config, common_config):
            return True
            
        if not self._check_local_image(config, common_config):
            return False
            
        return self._push_image_to_cr(config, common_config)
    
    def _check_local_image(self, config: HybridVeAgentkitConfig, common_config: CommonConfig) -> bool:
        """检查本地镜像"""
        try:
            from agentkit.toolkit.integrations.container import DockerManager
        except ImportError as e:
            self.logger.error(f"Docker dependencies missing: {e}")
            return False
            
        docker_manager = DockerManager()
        image_exists, image_info, actual_image_id = docker_manager.check_image_exists(
            config.full_image_name or f"{common_config.agent_name or 'agentkit-app'}:{config.image_tag}", 
            config.image_id
        )
        
        if not image_exists:
            self.logger.error(f"Image does not exist: {config.full_image_name}")
            return False
            
        # 更新镜像ID
        config.image_id = actual_image_id
        return True
    
    def _push_image_to_cr(self, config: HybridVeAgentkitConfig, common_config: CommonConfig) -> bool:
        """推送镜像到CR - 使用新的CR服务"""
        # 创建CR配置回调
        def config_updater(workflow_name: str, cr_config_dict: Dict[str, Any]) -> None:
            """配置更新回调"""
            # 将CR配置同步到工作流配置
            if "instance_name" in cr_config_dict:
                config.cr_instance_name = cr_config_dict["instance_name"]
            if "namespace_name" in cr_config_dict:
                config.cr_namespace_name = cr_config_dict["namespace_name"]
            if "repo_name" in cr_config_dict:
                config.cr_repo_name = cr_config_dict["repo_name"]
            if "image_full_url" in cr_config_dict:
                config.cr_image_full_url = cr_config_dict["image_full_url"]
            
            # 更新工作流配置
            get_config().update_workflow_config("hybrid", config.to_persist_dict())
        
        # 创建CR服务配置
        cr_service_config = CRServiceConfig(
            instance_name=config.cr_instance_name,
            namespace_name=config.cr_namespace_name,
            repo_name=config.cr_repo_name,
            image_full_url=config.cr_image_full_url
        )
        
        # 创建CR服务
        cr_service = CRService(config_callback=DefaultCRConfigCallback(config_updater=config_updater))
        
        # 确保CR资源存在
        cr_result = cr_service.ensure_cr_resources(cr_service_config, common_config)
        if not cr_result.success:
            self.logger.error(f"CR resource preparation failed: {cr_result.error}")
            return False
        
        # 确保公网访问
        public_result = cr_service.ensure_public_endpoint(cr_service_config)
        if not public_result.success:
            self.logger.error(f"CR public endpoint configuration failed: {public_result.error}")
            return False
        
        # 登录并推送镜像
        success, remote_image_full_url = cr_service.login_and_push_image(
            cr_service_config,
            config.image_id,
            config.image_tag,
            cr_result.namespace_name
        )
        
        if success:
            config.cr_image_full_url = remote_image_full_url
            get_config().update_workflow_config("hybrid", config.to_persist_dict())
            return True
        else:
            return False
    
    # 原有的CR相关方法已经移除，逻辑已迁移到CRService中
    
    def _deploy_runtime(self, config: HybridVeAgentkitConfig, common_config) -> DeployInfo:
        """部署Runtime - 使用VeAgentkitRuntimeRunner"""
        try:
            # 合并应用级和 Workflow 级环境变量
            from agentkit.toolkit.config import merge_runtime_envs
            merged_envs = merge_runtime_envs(common_config, config.to_dict())
            
            # 创建Runner配置
            runner_config = {
                "common_config": common_config.to_dict(),
                "runtime_id": config.runtime_id or AUTO_CREATE_VE,
                "runtime_name": config.runtime_name,
                "runtime_role_name": config.runtime_role_name,
                "runtime_apikey": config.runtime_apikey,
                "runtime_apikey_name": config.runtime_apikey_name,
                "runtime_endpoint": config.runtime_endpoint,
                "runtime_envs": merged_envs,
                "image_url": config.cr_image_full_url
            }
            
            # 使用Runner部署
            runner = VeAgentkitRuntimeRunner()
            success, result = runner.deploy(runner_config)
            
            if success:
                # 更新配置
                config.runtime_id = result.get("runtime_id", config.runtime_id)
                config.runtime_name = result.get("runtime_name", config.runtime_name)
                config.runtime_endpoint = result.get("runtime_endpoint", config.runtime_endpoint)
                config.runtime_apikey = result.get("runtime_apikey", config.runtime_apikey)
                config.runtime_apikey_name = result.get("runtime_apikey_name", config.runtime_apikey_name)
                config.runtime_role_name = result.get("runtime_role_name", config.runtime_role_name)
                config.deploy_timestamp = datetime.now().isoformat()
                
                # 保存配置
                agent_config = get_config()
                agent_config.update_workflow_config("hybrid", config.to_persist_dict())
                
                self.progress.report("部署完成", 100, level=ProgressLevel.SUCCESS)
                self.logger.info(f"Deploy successful: {result.get('message')}")
                
                return DeployInfo(
                    success=True,
                    service_id=config.runtime_id,
                    endpoint_url=config.runtime_endpoint,
                    deploy_timestamp=config.deploy_timestamp,
                    details={
                        "workflow": "hybrid",
                        "runtime_name": config.runtime_name,
                        "image_url": config.cr_image_full_url,
                        "message": result.get('message', '')
                    }
                )
            else:
                error_msg = result.get('error', '未知错误')
                self.logger.error(f"Deploy failed: {error_msg}")
                self.progress.report("部署失败", 0, level=ProgressLevel.ERROR)
                
                return DeployInfo(
                    success=False,
                    error=error_msg,
                    error_type=WorkflowErrorType.DEPLOY_FAILURE
                )
                
        except Exception as e:
            self.logger.error(f"Deploy error: {e}", exc_info=True)
            self.progress.report(f"部署异常: {e}", 0, level=ProgressLevel.ERROR)
            
            return DeployInfo(
                success=False,
                error=str(e),
                error_type=WorkflowErrorType.UNKNOWN_ERROR,
                details={"exception": type(e).__name__}
            )

    def invoke(self, config: Dict[str, Any], args: Dict[str, Any]) -> Tuple[bool, Any]:
        """Invoke the workflow - 使用VeAgentkitRuntimeRunner.
        Args:
            config (Dict[str, Any]): The configuration of the workflow.
        Returns:
            bool: True if the invocation was successful, False otherwise.
        """
        hybrid_ve_config = HybridVeAgentkitConfig.from_dict(config)
        if not hybrid_ve_config.runtime_id:
            self.logger.error("No runtime ID configured")
            return InvokeInfo(
                success=False,
                error="暂未部署到Agentkit Platform",
                error_type=WorkflowErrorType.CONFIG_ERROR
            )
        try:
            # 创建Runner配置
            runner_config = {
                "common_config": {},
                "runtime_id": hybrid_ve_config.runtime_id,
                "runtime_endpoint": hybrid_ve_config.runtime_endpoint,
                "runtime_apikey": hybrid_ve_config.runtime_apikey
            }
            payload = args.get("payload", {"prompt": "北京天气怎么样"})
            if isinstance(payload, str):
                payload = json.loads(payload)
            headers = args.get("headers", {"user_id": "agentkit_user", "session_id": "agentkit_sample_session"})
            if isinstance(headers, str):
                headers = json.loads(headers)
            # 使用Runner调用
            runner = VeAgentkitRuntimeRunner()
            success, result = runner.invoke(runner_config, payload, headers)
            if success:
                # 检测是否是流式响应（生成器）
                is_streaming = hasattr(result, '__iter__') and not isinstance(result, (dict, str, list, bytes))
                
                return InvokeInfo(
                    success=True,
                    response=result,
                    is_streaming=is_streaming
                )
            else:
                self.logger.error(f"Invoke failed: {result}")
                return InvokeInfo(
                    success=False,
                    error=str(result),
                    error_type=WorkflowErrorType.UNKNOWN_ERROR
                )
        except Exception as e:
            self.logger.error(f"Invoke error: {e}", exc_info=True)
            return InvokeInfo(
                success=False,
                error=str(e),
                error_type=WorkflowErrorType.UNKNOWN_ERROR,
                details={"exception": type(e).__name__}
            )
        

    def status(self, config: Dict[str, Any] = None) -> StatusInfo:
        """Get the status of the workflow - 使用VeAgentkitRuntimeRunner."""
        agent_config = get_config()
        if config is None:
            config = agent_config.get_workflow_config("hybrid")
        
        hybrid_ve_config = HybridVeAgentkitConfig.from_dict(config)
        if not hybrid_ve_config.runtime_id or hybrid_ve_config.runtime_id == AUTO_CREATE_VE:
            self.logger.warning("No runtime ID configured")
            return StatusInfo(
                success=True,
                status="not_deployed",
                details={"message": "暂未部署到Agentkit Platform"}
            )
        try:
            # 创建Runner配置
            runner_config = {
                "common_config": {},
                "runtime_id": hybrid_ve_config.runtime_id,
                "runtime_endpoint": hybrid_ve_config.runtime_endpoint,
                "runtime_apikey": hybrid_ve_config.runtime_apikey
            }
            # 使用Runner获取状态
            runner = VeAgentkitRuntimeRunner()
            status_data = runner.status(runner_config)
            
            # Normalize status
            raw_status = status_data.get("status", "unknown")
            if raw_status == "Ready":
                normalized_status = "running"
            else:
                normalized_status = raw_status.lower()
            
            return StatusInfo(
                success=True,
                status=normalized_status,
                endpoint_url=status_data.get("endpoint") or hybrid_ve_config.runtime_endpoint,
                service_id=hybrid_ve_config.runtime_id,
                details=status_data
            )
        except Exception as e:
            self.logger.error(f"Status query error: {e}", exc_info=True)
            return StatusInfo(
                success=False,
                status="error",
                error=str(e),
                error_type=WorkflowErrorType.UNKNOWN_ERROR,
                details={"exception": type(e).__name__}
            )
        

    def stop(self, config: Dict[str, Any] = None) -> bool:
        """Stop the workflow - 使用VeAgentkitRuntimeRunner.
        
        Args:
            config: 工作流配置（可选）
        
        Returns:
            bool: 是否成功停止
        """
        agent_config = get_config()
        if config is None:
            config = agent_config.get_workflow_config("hybrid")
        
        hybrid_ve_config = HybridVeAgentkitConfig.from_dict(config)
        if not hybrid_ve_config.runtime_id or hybrid_ve_config.runtime_id == AUTO_CREATE_VE:
            self.logger.info("No runtime ID configured, nothing to stop")
            self.progress.report("未配置Runtime ID，无需停止", 100, level=ProgressLevel.INFO)
            return True
        
        try:
            self.progress.report("停止Hybrid Runtime", 50, level=ProgressLevel.INFO)
            
            # 创建Runner配置
            runner_config = {
                "common_config": {},
                "runtime_id": hybrid_ve_config.runtime_id,
            }
            runner = VeAgentkitRuntimeRunner()
            success = runner.stop(runner_config)
            
            if success:
                self.logger.info("Runtime stopped successfully")
                self.progress.report("Runtime停止成功", 100, level=ProgressLevel.SUCCESS)
            else:
                self.logger.error("Runtime stop failed")
                self.progress.report("Runtime停止失败", 0, level=ProgressLevel.ERROR)
            
            return success
            
        except Exception as e:
            self.logger.error(f"Runtime stop error: {e}", exc_info=True)
            self.progress.report(f"Runtime停止异常: {e}", 0, level=ProgressLevel.ERROR)
            return False

    def destroy(self, config: Dict[str, Any] = None, force: bool = False) -> bool:
        """Stop and destroy the workflow resources - 使用VeAgentkitRuntimeRunner.
        
        Args:
            config: 工作流配置（可选）
            force: 强制销毁，即使遇到错误也继续
        
        Returns:
            bool: 是否成功销毁
        """
        agent_config = get_config()
        if config is None:
            config = agent_config.get_workflow_config("hybrid")
        
        hybrid_ve_config = HybridVeAgentkitConfig.from_dict(config)
        if not hybrid_ve_config.runtime_id or hybrid_ve_config.runtime_id == AUTO_CREATE_VE:
            self.logger.info("No runtime ID configured, nothing to destroy")
            self.progress.report("未配置Runtime ID，无需销毁", 100, level=ProgressLevel.INFO)
            return True
        
        try:
            self.progress.report("销毁Hybrid Runtime资源", 50, level=ProgressLevel.INFO)
            
            # 创建Runner配置
            runner_config = {
                "common_config": {},
                "runtime_id": hybrid_ve_config.runtime_id
            }
            runner = VeAgentkitRuntimeRunner()
            success = runner.destroy(runner_config)
            
            if success:
                # 清空配置
                hybrid_ve_config.runtime_id = ""
                hybrid_ve_config.runtime_endpoint = ""
                hybrid_ve_config.runtime_apikey = ""
                agent_config.update_workflow_config("hybrid", hybrid_ve_config.to_persist_dict())
                self.logger.info("Runtime destroyed successfully")
                self.progress.report("Runtime销毁成功", 100, level=ProgressLevel.SUCCESS)
            else:
                error_msg = "Runtime destroy failed"
                self.logger.error(error_msg)
                self.progress.report("Runtime销毁失败", 0, level=ProgressLevel.ERROR)
                if not force:
                    return False
            
            return True
            
        except Exception as e:
            self.logger.error(f"Runtime destroy error: {e}", exc_info=True)
            self.progress.report(f"Runtime销毁异常: {e}", 0, level=ProgressLevel.ERROR)
            if not force:
                return False
            return True  # 强制模式下即使失败也返回 True