# Copyright (c) 2025 Beijing Volcano Engine Technology Co., Ltd. and/or its affiliates.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"""
Workflow result models.

These models represent the results of workflow operations,
providing detailed information to the Service layer.
"""

from dataclasses import dataclass, field
from typing import Optional, Dict, Any


class WorkflowErrorType:
    """Error type constants for workflow operations."""
    CONFIG_ERROR = "configuration_error"
    NETWORK_ERROR = "network_error"
    PERMISSION_ERROR = "permission_error"
    BUILD_FAILURE = "build_failure"
    DEPLOY_FAILURE = "deploy_failure"
    INVOKE_FAILURE = "invoke_failure"
    RESOURCE_NOT_FOUND = "resource_not_found"
    TIMEOUT_ERROR = "timeout_error"
    DEPENDENCY_ERROR = "dependency_error"
    VALIDATION_ERROR = "validation_error"
    IMAGE_PUSH_ERROR = "image_push_error"
    CONTAINER_START_ERROR = "container_start_error"
    UNKNOWN_ERROR = "unknown_error"


@dataclass
class WorkflowResult:
    """Base class for workflow operation results."""
    success: bool
    error: Optional[str] = None
    error_type: Optional[str] = None
    details: Dict[str, Any] = field(default_factory=dict)
    
    def __bool__(self):
        """Allow boolean context: if result: ..."""
        return self.success
    
    def get_user_friendly_message(self) -> str:
        """Get user-friendly error message."""
        if not self.error:
            return "操作成功"
        
        if self.error_type == WorkflowErrorType.CONFIG_ERROR:
            return f"配置错误: {self.error}"
        elif self.error_type == WorkflowErrorType.NETWORK_ERROR:
            return f"网络错误: {self.error}. 请检查网络连接并重试"
        elif self.error_type == WorkflowErrorType.PERMISSION_ERROR:
            return f"权限错误: {self.error}. 请检查访问权限"
        elif self.error_type == WorkflowErrorType.TIMEOUT_ERROR:
            return f"超时错误: {self.error}. 操作超时，请稍后重试"
        elif self.error_type == WorkflowErrorType.INVOKE_FAILURE:
            return f"调用失败: {self.error}. 请检查服务状态和输入参数"
        else:
            return f"错误: {self.error}"


@dataclass
class BuildInfo(WorkflowResult):
    """
    Build operation result from workflow.
    
    Contains detailed information about the built image.
    """
    image_name: Optional[str] = None
    image_id: Optional[str] = None
    image_tag: Optional[str] = None
    build_timestamp: Optional[str] = None
    
    def get_image_repository(self) -> Optional[str]:
        """
        Extract repository from image name.
        
        Returns:
            Repository part of image name (without tag).
            
        Example:
            >>> info = BuildInfo(image_name="myrepo:v1.0")
            >>> info.get_image_repository()
            "myrepo"
        """
        if self.image_name and ':' in self.image_name:
            return self.image_name.rsplit(':', 1)[0]
        return self.image_name
    
    def has_warnings(self) -> bool:
        """Check if build succeeded but has warnings."""
        return self.success and bool(self.details.get('warnings'))
    
    def __str__(self):
        if self.success:
            return f"BuildInfo(success=True, image={self.image_name})"
        else:
            return f"BuildInfo(success=False, error={self.error})"


@dataclass
class DeployInfo(WorkflowResult):
    """
    Deploy operation result from workflow.
    
    Contains detailed information about the deployed service.
    """
    endpoint_url: Optional[str] = None
    service_id: Optional[str] = None
    container_id: Optional[str] = None
    container_name: Optional[str] = None
    deploy_timestamp: Optional[str] = None
    
    def __str__(self):
        if self.success:
            return f"DeployInfo(success=True, endpoint={self.endpoint_url})"
        else:
            return f"DeployInfo(success=False, error={self.error})"


@dataclass
class InvokeInfo(WorkflowResult):
    """
    Invoke operation result from workflow.
    
    Contains response data and streaming information.
    """
    response: Any = None
    is_streaming: bool = False
    response_time_ms: Optional[float] = None
    
    def __str__(self):
        if self.success:
            return f"InvokeInfo(success=True, streaming={self.is_streaming})"
        else:
            return f"InvokeInfo(success=False, error={self.error})"


@dataclass
class StatusInfo(WorkflowResult):
    """
    Status query result from workflow.
    
    Contains current status of the service.
    """
    status: Optional[str] = None
    endpoint_url: Optional[str] = None
    service_id: Optional[str] = None
    container_id: Optional[str] = None
    uptime: Optional[int] = None
    
    def __str__(self):
        if self.success:
            return f"StatusInfo(success=True, status={self.status})"
        else:
            return f"StatusInfo(success=False, error={self.error})"
