# Copyright (c) 2025 Beijing Volcano Engine Technology Co., Ltd. and/or its affiliates.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

from typing import Dict

from agentkit.client import BaseAgentkitClient
from agentkit.utils import get_logger

# 导入自动生成的类型
from agentkit.mcp.mcp_all_types import (
    CreateMCPServiceRequest,
    CreateMCPServiceResponse,
    UpdateMCPServiceRequest,
    UpdateMCPServiceResponse,
    DeleteMCPServiceRequest,
    DeleteMCPServiceResponse,
    GetMCPServiceRequest,
    GetMCPServiceResponse,
    ListMCPServicesRequest,
    ListMCPServicesResponse,
    CreateMCPToolsetRequest,
    CreateMCPToolsetResponse,
    UpdateMCPToolsetRequest,
    UpdateMCPToolsetResponse,
    DeleteMCPToolsetRequest,
    DeleteMCPToolsetResponse,
    GetMCPToolsetRequest,
    GetMCPToolsetResponse,
    ListMCPToolsetsRequest,
    ListMCPToolsetsResponse,
    UpdateMCPToolsRequest,
    UpdateMCPToolsResponse,
    GetMCPToolsRequest,
    GetMCPToolsResponse,
    ListMCPToolsRequest,
    ListMCPToolsResponse,
)

logger = get_logger(__name__)


class AgentkitMCP(BaseAgentkitClient):
    """AgentKit MCP (Model Context Protocol) Management Service"""
    
    # Define all API actions for this service
    API_ACTIONS: Dict[str, str] = {
        "CreateMCPService": "CreateMCPService",
        "UpdateMCPService": "UpdateMCPService",
        "DeleteMCPService": "DeleteMCPService",
        "GetMCPService": "GetMCPService",
        "ListMCPServices": "ListMCPServices",
        "CreateMCPToolset": "CreateMCPToolset",
        "UpdateMCPToolset": "UpdateMCPToolset",
        "DeleteMCPToolset": "DeleteMCPToolset",
        "GetMCPToolset": "GetMCPToolset",
        "ListMCPToolsets": "ListMCPToolsets",
        "UpdateMCPTools": "UpdateMCPTools",
        "GetMCPTools": "GetMCPTools",
        "ListMCPTools": "ListMCPTools",
    }
    
    def __init__(
        self,
        access_key: str = "",
        secret_key: str = "",
        region: str = "",
        session_token: str = "",
    ) -> None:
        super().__init__(
            access_key=access_key,
            secret_key=secret_key,
            region=region,
            session_token=session_token,
            service_name="mcp",
        )

    # ==================== MCP Service APIs ====================
    
    def create_mcp_service(
        self, request: CreateMCPServiceRequest
    ) -> CreateMCPServiceResponse:
        """Create a new MCP service."""
        return self._invoke_api(
            api_action="CreateMCPService",
            request=request,
            response_type=CreateMCPServiceResponse,
        )

    def update_mcp_service(
        self, request: UpdateMCPServiceRequest
    ) -> UpdateMCPServiceResponse:
        """Update an existing MCP service."""
        return self._invoke_api(
            api_action="UpdateMCPService",
            request=request,
            response_type=UpdateMCPServiceResponse,
        )

    def delete_mcp_service(
        self, request: DeleteMCPServiceRequest
    ) -> DeleteMCPServiceResponse:
        """Delete an MCP service."""
        return self._invoke_api(
            api_action="DeleteMCPService",
            request=request,
            response_type=DeleteMCPServiceResponse,
        )

    def get_mcp_service(
        self, request: GetMCPServiceRequest
    ) -> GetMCPServiceResponse:
        """Get details of a specific MCP service."""
        return self._invoke_api(
            api_action="GetMCPService",
            request=request,
            response_type=GetMCPServiceResponse,
        )

    def list_mcp_services(
        self, request: ListMCPServicesRequest
    ) -> ListMCPServicesResponse:
        """List all MCP services."""
        return self._invoke_api(
            api_action="ListMCPServices",
            request=request,
            response_type=ListMCPServicesResponse,
        )

    # ==================== MCP Toolset APIs ====================

    def create_mcp_toolset(
        self, request: CreateMCPToolsetRequest
    ) -> CreateMCPToolsetResponse:
        """Create a new MCP toolset."""
        return self._invoke_api(
            api_action="CreateMCPToolset",
            request=request,
            response_type=CreateMCPToolsetResponse,
        )

    def update_mcp_toolset(
        self, request: UpdateMCPToolsetRequest
    ) -> UpdateMCPToolsetResponse:
        """Update an existing MCP toolset."""
        return self._invoke_api(
            api_action="UpdateMCPToolset",
            request=request,
            response_type=UpdateMCPToolsetResponse,
        )

    def delete_mcp_toolset(
        self, request: DeleteMCPToolsetRequest
    ) -> DeleteMCPToolsetResponse:
        """Delete an MCP toolset."""
        return self._invoke_api(
            api_action="DeleteMCPToolset",
            request=request,
            response_type=DeleteMCPToolsetResponse,
        )

    def get_mcp_toolset(
        self, request: GetMCPToolsetRequest
    ) -> GetMCPToolsetResponse:
        """Get details of a specific MCP toolset."""
        return self._invoke_api(
            api_action="GetMCPToolset",
            request=request,
            response_type=GetMCPToolsetResponse,
        )

    def list_mcp_toolsets(
        self, request: ListMCPToolsetsRequest
    ) -> ListMCPToolsetsResponse:
        """List all MCP toolsets."""
        return self._invoke_api(
            api_action="ListMCPToolsets",
            request=request,
            response_type=ListMCPToolsetsResponse,
        )

    # ==================== MCP Tools APIs ====================

    def update_mcp_tools(
        self, request: UpdateMCPToolsRequest
    ) -> UpdateMCPToolsResponse:
        """Update tools for an MCP service."""
        return self._invoke_api(
            api_action="UpdateMCPTools",
            request=request,
            response_type=UpdateMCPToolsResponse,
        )

    def get_mcp_tools(
        self, request: GetMCPToolsRequest
    ) -> GetMCPToolsResponse:
        """Get tools from an MCP toolset."""
        return self._invoke_api(
            api_action="GetMCPTools",
            request=request,
            response_type=GetMCPToolsResponse,
        )

    def list_mcp_tools(
        self, request: ListMCPToolsRequest
    ) -> ListMCPToolsResponse:
        """List all MCP tools from specified toolsets."""
        return self._invoke_api(
            api_action="ListMCPTools",
            request=request,
            response_type=ListMCPToolsResponse,
        )


if __name__ == "__main__":
    # list mcp toolsets
    mcp = AgentkitMCP()
    req = ListMCPToolsetsRequest()
    res = mcp.list_mcp_toolsets(req)
    print(res)