# Copyright (c) 2025 Beijing Volcano Engine Technology Co., Ltd. and/or its affiliates.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"""AgentKit CLI - Build command implementation."""

import typer
from pathlib import Path
from rich.console import Console

# Note: 不要在文件开头导很重的包，不然会导致命令很卡(import包很慢)

console = Console()


def build_command(
    config_file: Path = typer.Option("agentkit.yaml", help="Configuration file"),
    platform: str = typer.Option("auto", help="Build platform"),
    regenerate_dockerfile: bool = typer.Option(
        False,
        "--regenerate-dockerfile",
        help="Force regenerate Dockerfile even if it exists"
    ),
):
    """Build Docker image for the Agent."""
    from agentkit.toolkit import sdk
    
    console.print(f"[cyan]Building image with {config_file}[/cyan]")
    
    # Call SDK
    result = sdk.build(
        config_file=str(config_file),
        platform=platform,
        regenerate_dockerfile=regenerate_dockerfile
    )
    
    # Format output
    if result.success:
        if result.image_name:
            console.print(f"[green]✅ Build completed successfully![/green]")
            console.print(f"[green]📦 Image: {result.image_name}[/green]")
        else:
            console.print(f"[green]✅ Build completed![/green]")
        
        if result.image_id:
            console.print(f"[dim]Image ID: {result.image_id}[/dim]")
        if result.image_tag:
            console.print(f"[dim]Tag: {result.image_tag}[/dim]")
    else:
        console.print(f"[red]❌ Build failed: {result.error}[/red]")
        if result.build_logs:
            for log in result.build_logs:
                if log.strip():
                    console.print(f"[red]{log}[/red]")
        raise typer.Exit(1)
