# Copyright (c) 2025 Beijing Volcano Engine Technology Co., Ltd. and/or its affiliates.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"""AgentKit CLI - Config command implementation."""

from typing import Optional, List
import typer
from rich.console import Console

from agentkit.toolkit.config import get_config, CommonConfig
from agentkit.toolkit.config.config_handler import (
    ConfigParamHandler, 
    NonInteractiveConfigHandler
)

# Note: 不要在文件开头导很重的包，不然会导致命令很卡(import包很慢)

console = Console()


def config_command(
    config_file: Optional[str] = typer.Option(None, "--config", "-c", help="配置文件路径"),
    
    # 模式控制
    interactive: bool = typer.Option(False, "--interactive", "-i", help="强制交互式模式"),
    dry_run: bool = typer.Option(False, "--dry-run", help="预览配置变更但不保存"),
    show: bool = typer.Option(False, "--show", "-s", help="显示当前配置"),
    
    # 全局配置支持
    global_config: bool = typer.Option(False, "--global", "-g", help="操作全局配置 (~/.agentkit/config.yaml)"),
    set_field: Optional[str] = typer.Option(None, "--set", help="设置字段 (格式: key=value，如 cr.instance_name=my-cr)"),
    init_global: bool = typer.Option(False, "--init", help="初始化全局配置文件（创建模板）"),
    
    # CommonConfig 参数
    agent_name: Optional[str] = typer.Option(None, "--agent_name", help="Agent应用名称"),
    entry_point: Optional[str] = typer.Option(None, "--entry_point", help="入口文件 (例如: agent.py)"),
    description: Optional[str] = typer.Option(None, "--description", help="应用描述"),
    language: Optional[str] = typer.Option(None, "--language", help="编程语言: Python/Golang"),
    language_version: Optional[str] = typer.Option(None, "--language_version", help="语言版本 (例如: Python: 3.10-3.13, Go: 1.x)"),
    python_version: Optional[str] = typer.Option(None, "--python_version", help="[已弃用] Python版本，请使用 --language_version"),
    dependencies_file: Optional[str] = typer.Option(None, "--dependencies_file", help="依赖文件 (例如: requirements.txt)"),
    launch_type: Optional[str] = typer.Option(None, "--launch_type", help="部署模式: local/hybrid/cloud"),
    
    # 应用级环境变量（CommonConfig）
    runtime_envs: Optional[List[str]] = typer.Option(None, "--runtime_envs", "-e", help="应用级环境变量 KEY=VALUE，所有部署模式共享（可多次使用）"),
    
    # Workflow 级环境变量
    workflow_runtime_envs: Optional[List[str]] = typer.Option(None, "--workflow-runtime-envs", help="Workflow级环境变量 KEY=VALUE，仅当前部署模式使用（可多次使用）"),
    
    # Hybrid/Cloud workflow 参数
    region: Optional[str] = typer.Option(None, "--region", help="区域 (例如: cn-beijing)"),
    tos_bucket: Optional[str] = typer.Option(None, "--tos_bucket", help="TOS存储桶名称"),
    image_tag: Optional[str] = typer.Option(None, "--image_tag", help="镜像标签 (例如: 0.0.1)"),
    cr_instance_name: Optional[str] = typer.Option(None, "--cr_instance_name", "--ve_cr_instance_name", help="CR 实例名称"),
    cr_namespace_name: Optional[str] = typer.Option(None, "--cr_namespace_name", "--ve_cr_namespace_name", help="CR 命名空间"),
    cr_repo_name: Optional[str] = typer.Option(None, "--cr_repo_name", "--ve_cr_repo_name", help="CR 仓库名称"),
):
    """config AgentKit(support interactive and non-interactive mode)
    
    示例:
    
    \b
    # 交互式配置（默认）
    agentkit config
    
    \b
    # 非交互式配置（完整）
    agentkit config \\
        --agent_name myAgent \\
        --entry_point agent.py \\
        --language Python \\
        --language_version 3.12 \\
        --launch_type cloud \\
        --region cn-beijing \\
        --tos_bucket agentkit \\
        --image_tag 0.0.1 \\
        --runtime_envs API_KEY=xxxxx \\
        --runtime_envs MODEL_ENDPOINT=https://api.example.com
    
    \b
    # 配置 Golang 应用
    agentkit config \\
        --agent_name myGoAgent \\
        --entry_point main.go \\
        --language Golang \\
        --language_version 1.23
    
    \b
    # 应用级 vs Workflow 级环境变量
    # --runtime_envs/-e: 应用级（所有部署模式共享）
    # --workflow-runtime-envs: 仅当前部署模式使用
    agentkit config \\
        -e API_KEY=shared_key \\
        --workflow-runtime-envs DEBUG=true
    
    \b
    # 增量更新（只修改部分配置）
    agentkit config --language_version 3.13
    
    \b
    # 预览配置变更
    agentkit config --entry_point agent.py --dry-run
    
    \b
    # 显示当前配置
    agentkit config --show
    """
    
    try:
        # 处理全局配置操作
        if global_config:
            _handle_global_config(show, set_field, init_global)
            return
        
        # 处理 --show 选项
        if show:
            handler = NonInteractiveConfigHandler(config_path=config_file)
            handler.show_current_config()
            return
        
        # 收集 CLI 参数
        cli_params = ConfigParamHandler.collect_cli_params(
            agent_name=agent_name,
            entry_point=entry_point,
            description=description,
            language=language,
            language_version=language_version,
            python_version=python_version,  # 向后兼容
            dependencies_file=dependencies_file,
            launch_type=launch_type,
            runtime_envs=runtime_envs,
            workflow_runtime_envs=workflow_runtime_envs,
            region=region,
            tos_bucket=tos_bucket,
            image_tag=image_tag,
            cr_instance_name=cr_instance_name,
            cr_namespace_name=cr_namespace_name,
            cr_repo_name=cr_repo_name,
        )
        
        has_cli_params = ConfigParamHandler.has_cli_params(cli_params)
        
        # 决定使用哪种模式
        if interactive or (not has_cli_params and not interactive):
            # 交互式模式（保持原有行为）
            _interactive_config(config_file)
        else:
            # 非交互式模式
            handler = NonInteractiveConfigHandler(config_path=config_file)
            success = handler.update_config(
                common_params=cli_params['common'],
                workflow_params=cli_params['workflow'],
                dry_run=dry_run
            )
            
            if not success:
                raise typer.Exit(code=1)
    
    except KeyboardInterrupt:
        console.print("\n\n[yellow]⚠️  配置已取消[/yellow]")
        raise typer.Exit(code=130)  # 130 is the standard exit code for Ctrl+C


def _interactive_config(config_file: Optional[str] = None):
    """交互式配置（原有逻辑）"""
    config = get_config(config_path=config_file)
    
    common_config = CommonConfig.interactive_create(config.get_common_config().to_dict())
    config.update_common_config(common_config)
    
    workflow_name = common_config.launch_type

    from agentkit.toolkit.workflows import WORKFLOW_REGISTRY, Workflow
    
    if workflow_name in WORKFLOW_REGISTRY:
        workflow: Workflow = WORKFLOW_REGISTRY[workflow_name]()
        workflow_config = workflow.prompt_for_config(config.get_workflow_config(workflow_name))
        config.update_workflow_config(workflow_name, workflow_config)
        
        common_config.launch_type = workflow_name
        config.update_common_config(common_config)
    
    console.print("[green]✅ 配置完成![/green]")


def _handle_global_config(show: bool, set_field: Optional[str], init_global: bool):
    """处理全局配置操作"""
    from agentkit.toolkit.config import (
        get_global_config,
        save_global_config,
        global_config_exists
    )
    
    if init_global:
        # 初始化全局配置文件
        _init_global_config()
    elif show:
        # 显示全局配置
        _show_global_config()
    elif set_field:
        # 设置字段
        _set_global_field(set_field)
    else:
        # 提示用户如何使用
        if not global_config_exists():
            console.print("[yellow]⚠️  全局配置文件不存在[/yellow]")
            console.print(f"\n快速创建：")
            console.print(f"  agentkit config --global --init")
            console.print(f"\n或手动创建：")
            console.print(f"  mkdir -p ~/.agentkit")
            console.print(f"  vim ~/.agentkit/config.yaml")
        else:
            console.print("[cyan]💡 全局配置文件已存在[/cyan]")
            console.print(f"\n[bold]使用方式：[/bold]")
            console.print(f"  • 查看配置: [yellow]agentkit config --global --show[/yellow]")
            console.print(f"  • 设置字段: [yellow]agentkit config --global --set <字段>=<值>[/yellow]")
            console.print(f"  • 编辑文件: [yellow]vim ~/.agentkit/config.yaml[/yellow]")
            
            console.print(f"\n[bold]支持的字段：[/bold]")
            console.print(f"  [dim]Volcengine:[/dim]")
            console.print(f"    • [green]volcengine.access_key[/green]   - Access Key")
            console.print(f"    • [green]volcengine.secret_key[/green]   - Secret Key")
            console.print(f"    • [green]volcengine.region[/green]       - 区域")
            console.print(f"  [dim]CR:[/dim]")
            console.print(f"    • [green]cr.instance_name[/green]        - CR 实例名称")
            console.print(f"    • [green]cr.namespace_name[/green]       - CR 命名空间")
            console.print(f"  [dim]TOS:[/dim]")
            console.print(f"    • [green]tos.bucket[/green]              - Bucket 名称")
            console.print(f"    • [green]tos.prefix[/green]              - 对象前缀")
            console.print(f"    • [green]tos.region[/green]              - 区域")


def _init_global_config():
    """初始化全局配置文件（创建模板）"""
    from agentkit.toolkit.config import GlobalConfig, save_global_config, global_config_exists
    from pathlib import Path
    
    config_path = Path.home() / ".agentkit" / "config.yaml"
    
    # 检查文件是否已存在
    if global_config_exists():
        console.print(f"[yellow]⚠️  全局配置文件已存在: {config_path}[/yellow]")
        console.print(f"\n如需重新创建，请先删除现有文件：")
        console.print(f"  rm ~/.agentkit/config.yaml")
        console.print(f"\n或直接编辑现有文件：")
        console.print(f"  vim ~/.agentkit/config.yaml")
        return
    
    # 创建默认配置
    config = GlobalConfig()
    # 保留默认值
    config.volcengine.region = "cn-beijing"
    config.tos.prefix = "agentkit-builds"
    config.tos.region = "cn-beijing"
    
    # 保存配置文件
    try:
        save_global_config(config)
        console.print(f"[green]✅ 全局配置文件已创建: {config_path}[/green]")
        console.print(f"\n[bold cyan]📝 配置模板已生成，包含以下配置项：[/bold cyan]\n")
        
        console.print("[bold]🔐 Volcengine 凭证[/bold]")
        console.print("  access_key: ''     # 火山引擎 Access Key")
        console.print("  secret_key: ''     # 火山引擎 Secret Key")
        console.print("  region: cn-beijing # 默认区域")
        
        console.print("\n[bold]📦 CR 配置[/bold]")
        console.print("  instance_name: ''  # CR 实例名称")
        console.print("  namespace_name: '' # CR 命名空间")
        
        console.print("\n[bold]🗂️  TOS 配置[/bold]")
        console.print("  bucket: ''         # TOS Bucket 名称")
        console.print("  prefix: agentkit-builds")
        console.print("  region: cn-beijing")
        
        console.print(f"\n[dim]💡 提示：[/dim]")
        console.print(f"  • 编辑配置: vim {config_path}")
        console.print(f"  • 查看配置: agentkit config --global --show")
        
    except Exception as e:
        console.print(f"[red]❌ 创建配置文件失败: {e}[/red]")
        raise typer.Exit(code=1)


def _show_global_config():
    """显示全局配置"""
    from agentkit.toolkit.config import get_global_config, global_config_exists
    
    if not global_config_exists():
        console.print("[yellow]⚠️  全局配置文件不存在: ~/.agentkit/config.yaml[/yellow]")
        console.print("\n快速创建：")
        console.print("  agentkit config --global --init")
        return
    
    config = get_global_config()
    
    console.print("\n[bold cyan]📋 全局配置[/bold cyan] [dim](~/.agentkit/config.yaml)[/dim]\n")
    
    # 显示 Volcengine 凭证
    console.print("[bold]🔐 Volcengine 凭证[/bold]")
    if config.volcengine.access_key:
        masked_ak = config.volcengine.access_key[:5] + "***" if len(config.volcengine.access_key) > 5 else "***"
        console.print(f"  Access Key: [yellow]{masked_ak}[/yellow] (已设置)")
    else:
        console.print("  Access Key: [dim](未设置)[/dim]")
    
    if config.volcengine.secret_key:
        console.print(f"  Secret Key: [yellow]***[/yellow] (已设置)")
    else:
        console.print("  Secret Key: [dim](未设置)[/dim]")
    
    console.print(f"  Region:     [yellow]{config.volcengine.region}[/yellow]")
    
    # 显示 CR 配置
    console.print("\n[bold]📦 CR 配置[/bold]")
    console.print(f"  Instance:   [yellow]{config.cr.instance_name or '[dim](未设置)[/dim]'}[/yellow]")
    console.print(f"  Namespace:  [yellow]{config.cr.namespace_name or '[dim](未设置)[/dim]'}[/yellow]")
    
    # 显示 TOS 配置
    console.print("\n[bold]🗂️  TOS 配置[/bold]")
    console.print(f"  Bucket:     [yellow]{config.tos.bucket or '[dim](未设置)[/dim]'}[/yellow]")
    console.print(f"  Prefix:     [yellow]{config.tos.prefix}[/yellow]")
    console.print(f"  Region:     [yellow]{config.tos.region}[/yellow]")
    
    console.print()


def _set_global_field(field_value: str):
    """设置全局配置字段"""
    from agentkit.toolkit.config import get_global_config, save_global_config
    
    # 解析 key=value
    if '=' not in field_value:
        console.print(f"[red]❌ 错误格式，应为: key=value[/red]")
        console.print(f"\n示例：")
        console.print(f"  agentkit config --global --set cr.instance_name=my-cr")
        console.print(f"  agentkit config --global --set tos.bucket=my-bucket")
        raise typer.Exit(code=1)
    
    key, value = field_value.split('=', 1)
    parts = key.split('.')
    
    if len(parts) != 2:
        console.print(f"[red]❌ 错误的字段格式: {key}[/red]")
        console.print(f"\n支持的字段：")
        console.print(f"  • volcengine.access_key")
        console.print(f"  • volcengine.secret_key")
        console.print(f"  • volcengine.region")
        console.print(f"  • cr.instance_name")
        console.print(f"  • cr.namespace_name")
        console.print(f"  • tos.bucket")
        console.print(f"  • tos.prefix")
        console.print(f"  • tos.region")
        raise typer.Exit(code=1)
    
    section, field = parts
    
    # 加载配置
    config = get_global_config()
    
    # 设置字段值
    try:
        if section == 'volcengine':
            if not hasattr(config.volcengine, field):
                raise AttributeError(f"volcengine 没有字段: {field}")
            setattr(config.volcengine, field, value)
        elif section == 'cr':
            if not hasattr(config.cr, field):
                raise AttributeError(f"cr 没有字段: {field}")
            setattr(config.cr, field, value)
        elif section == 'tos':
            if not hasattr(config.tos, field):
                raise AttributeError(f"tos 没有字段: {field}")
            setattr(config.tos, field, value)
        else:
            console.print(f"[red]❌ 未知的配置段: {section}[/red]")
            console.print(f"\n支持的配置段: volcengine, cr, tos")
            raise typer.Exit(code=1)
        
        # 保存配置
        save_global_config(config)
        
        # 显示成功消息
        if value:
            console.print(f"[green]✅ 已设置:[/green] {key} = [yellow]{value}[/yellow]")
        else:
            console.print(f"[green]✅ 已清空:[/green] {key}")
        
        console.print(f"\n[dim]配置已保存到: ~/.agentkit/config.yaml[/dim]")
        
    except AttributeError as e:
        console.print(f"[red]❌ {e}[/red]")
        raise typer.Exit(code=1)
