# Copyright (c) 2025 Beijing Volcano Engine Technology Co., Ltd. and/or its affiliates.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"""AgentKit CLI - Invoke command implementation."""

from pathlib import Path
from typing import Optional, Any, List
import json
import typer
from rich.console import Console
import random
from agentkit.toolkit.config import get_config

# Note: 不要在文件开头导很重的包，不然会导致命令很卡(import包很慢)

console = Console()

def build_standard_payload(message: Optional[str], payload: Optional[str]) -> dict:
    if message:
        return {"prompt": message}
    else:
        try:
            parsed = json.loads(payload) if isinstance(payload, str) else payload
            console.print(f"[blue]Using custom payload: {parsed}[/blue]")
            return parsed
        except json.JSONDecodeError as e:
            console.print(f"[red]Error: Invalid JSON payload: {e}[/red]")
            raise typer.Exit(1)

def build_a2a_payload(message: Optional[str], payload: Optional[str], headers: dict) -> dict:
    parsed = None
    if payload:
        try:
            parsed = json.loads(payload) if isinstance(payload, str) else payload
        except json.JSONDecodeError:
            parsed = None

    if isinstance(parsed, dict) and parsed.get("jsonrpc"):
        console.print("[blue]Using provided JSON-RPC payload for A2A[/blue]")
        return parsed

    if message:
        text = message
    elif parsed is not None:
        text = json.dumps(parsed, ensure_ascii=False)
    else:
        text = payload if payload else ""

    a2a = {
        "jsonrpc": "2.0",
        "method": "message/stream",
        "params": {
            "message": {
                "role": "user",
                "parts": [
                    {"kind": "text", "text": text}
                ]
            },
            "metadata": headers
        },
        "id": random.randint(1, 999999)
    }
    return a2a

def invoke_command(
    config_file: Path = typer.Option("agentkit.yaml", help="Configuration file"),
    message: str = typer.Argument(None, help="Simple message to send to agent"),
    payload: str = typer.Option(
        None, "--payload", "-p", help="JSON payload to send (advanced option)"
    ),
    headers: str = typer.Option(
        None, "--headers", "-h", help="JSON headers for request (advanced option)"
    ),
    apikey: str = typer.Option(
        None, "--apikey", "-ak", help="API key for authentication"
    ),
) -> Any:
    """Send a test request to deployed Agent.
    
    Examples:
        # Simple message
        agentkit invoke "今天天气如何？"
        
        # Custom payload
        agentkit invoke --payload '{"prompt": "杭州天气?"}'
        
        # With custom headers
        agentkit invoke --payload '{"prompt": "杭州天气?"}' --headers '{"user_id": "test123"}'
    """
    from agentkit.toolkit import sdk

    console.print("[cyan]Invoking agent...[/cyan]")
    
    # 验证参数：message和payload不能同时提供
    if message and payload:
        console.print("[red]Error: Cannot specify both message and payload. Use either message or --payload.[/red]")
        raise typer.Exit(1)
    
    # 验证参数：必须提供message或payload
    if not message and not payload:
        console.print("[red]Error: Must provide either a message or --payload option.[/red]")
        raise typer.Exit(1)

    config = get_config(config_path=config_file)
    common_config = config.get_common_config()
    
    final_payload = build_standard_payload(message, payload)
    agent_type = getattr(common_config, "agent_type", "") or getattr(common_config, "template_type", "")
    is_a2a = isinstance(agent_type, str) and "a2a" in agent_type.lower()

    # 如果是 A2A Agent，则用 A2A 构造函数重建 payload
    if is_a2a:
        console.print("[cyan]Detected A2A agent type - constructing A2A JSON-RPC envelope[/cyan]")
        final_payload = build_a2a_payload(message, payload, final_headers)

    # 处理headers
    final_headers = {"user_id": "agentkit_user", "session_id": "agentkit_sample_session"}
    if headers:
        try:
            final_headers = json.loads(headers) if isinstance(headers, str) else headers
            console.print(f"[blue]Using custom headers: {final_headers}[/blue]")
        except json.JSONDecodeError as e:
            console.print(f"[red]Error: Invalid JSON headers: {e}[/red]")
            raise typer.Exit(1)
    else:
        console.print(f"[blue]Using default headers: {final_headers}[/blue]")
    
    # Call SDK
    result = sdk.invoke(
        payload=final_payload,
        config_file=str(config_file),
        headers=final_headers,
        apikey=apikey
    )
    
    if not result.success:
        console.print(f"[red]❌ Invocation failed: {result.error}[/red]")
        raise typer.Exit(1)
    
    console.print("[green]✅ Invocation successful[/green]")

    # Get response
    response = result.response

    # 处理流式响应（生成器）
    if result.is_streaming:
        console.print("[cyan]📡 Streaming response detected...[/cyan]\n")
        result_list = []
        complete_text = []
        
        for event in result.stream():
            result_list.append(event)
            
            # 如果是字符串且以 "data: " 开头，尝试解析（fallback处理）
            if isinstance(event, str):
                if event.strip().startswith("data: "):
                    try:
                        json_str = event.strip()[6:].strip()  # 移除 "data: " 前缀
                        event = json.loads(json_str)
                    except json.JSONDecodeError:
                        # 解析失败，跳过此事件
                        continue
                else:
                    # 不是 SSE 格式的字符串，跳过
                    continue
            
            # 处理 A2A JSON-RPC
            if isinstance(event, dict) and event.get("jsonrpc") and "result" in event:
                event = event["result"]

            if isinstance(event, dict):
                parts = []
                if isinstance(event.get("parts"), list):
                    parts = event.get("parts", [])
                elif isinstance(event.get("message"), dict):
                    parts = event["message"].get("parts", [])
                elif isinstance(event.get("content"), dict):
                    parts = event["content"].get("parts", [])

                if parts:
                    for p in parts:
                        text = None
                        if isinstance(p, dict) and "text" in p:
                            text = p["text"]
                        elif isinstance(p, str):
                            text = p
                        if text:
                            complete_text.append(text)
                            # 增量打印（保持不换行）
                            console.print(text, end="", style="green")

                # 显示事件中的错误信息（如果有）
                if "error" in event:
                    console.print(f"\n[red]Error: {event['error']}[/red]")

                # 处理状态更新（例如 final 标志或 completed 状态）
                if event.get("final") is True:
                    console.print("\n[cyan]Stream marked final[/cyan]")
                    break

                status = event.get("status")
                if isinstance(status, dict) and status.get("state") == "completed":
                    console.print("\n[cyan]Status indicates completed[/cyan]")
                    break

        # # 显示完整响应
        # if complete_text:
        #     console.print("\n\n[cyan]━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━[/cyan]")
        #     console.print(f"[cyan]📝 Complete response:[/cyan] {''.join(complete_text)}")
        console.print("")   # line break
        
        return str(result_list)
    
    # 处理非流式响应
    console.print("[cyan]📝 Response:[/cyan]")
    if isinstance(response, dict):
        console.print(json.dumps(response, indent=2, ensure_ascii=False))
    else:
        console.print(response)
    
    return str(response)