# Copyright (c) 2025 Beijing Volcano Engine Technology Co., Ltd. and/or its affiliates.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"""AgentKit CLI - Status command implementation."""

from pathlib import Path
import typer
from rich.console import Console
from agentkit.toolkit.config import get_config

# Note: 不要在文件开头导很重的包，不然会导致命令很卡(import包很慢)

console = Console()


def status_command(
    config_file: Path = typer.Option("agentkit.yaml", help="Configuration file"),
    verbose: bool = typer.Option(False, "--verbose", "-v", help="Show detailed information"),
):
    """Show current status of the agent runtime."""
    from agentkit.toolkit import sdk
    from rich.table import Table
    from rich.panel import Panel
    
    try:
        # Call SDK
        result = sdk.status(config_file=str(config_file))
        
        if not result.success:
            console.print(f"[red]❌ Status query failed: {result.error}[/red]")
            raise typer.Exit(1)
        
        # 状态显示
        status_color = {
            'running': 'green',
            'stopped': 'yellow',
            'not_deployed': 'dim',
            'unknown': 'dim'
        }.get(result.status.value, 'white')
        
        status_icon = {
            'running': '✅',
            'stopped': '⏸️',
            'not_deployed': '⚫',
            'unknown': '❓'
        }.get(result.status.value, '●')
        
        console.print(f"\n[bold {status_color}]{status_icon} Service Status: {result.status.value.upper()}[/bold {status_color}]\n")
        
        # 基本信息表格
        table = Table(show_header=False, box=None, padding=(0, 2))
        table.add_column("Key", style="cyan")
        table.add_column("Value", style="white")
        
        # 根据模式显示不同信息
        if result.container_id:  # Local mode
            details = result.details or {}
            container_info = details.get('container', {})
            image_info = details.get('image', {})
            
            if result.endpoint_url:
                table.add_row("🌐 Endpoint", result.endpoint_url)
            if result.container_id:
                table.add_row("📦 Container ID", result.container_id[:12])
            if container_info.get('name'):
                table.add_row("📛 Container Name", container_info['name'])
            if result.uptime:
                table.add_row("⏱️ Uptime", result.uptime)
            
            # 端口映射
            ports = container_info.get('ports', {})
            if ports:
                port_mappings = []
                for container_port, host_bindings in ports.items():
                    if host_bindings:
                        for binding in host_bindings:
                            host_ip = binding.get('HostIp', '0.0.0.0')
                            host_port = binding.get('HostPort', '')
                            if host_ip == '0.0.0.0':
                                port_mappings.append(f"{host_port}->{container_port}")
                            else:
                                port_mappings.append(f"{host_ip}:{host_port}->{container_port}")
                if port_mappings:
                    table.add_row("🔌 Ports", ", ".join(port_mappings))
            
            # 镜像信息
            if image_info.get('name'):
                table.add_row("💿 Image", image_info['name'])
            if image_info.get('id'):
                table.add_row("🏷️  Image ID", image_info['id'])
            if image_info.get('size'):
                size_mb = image_info['size'] / (1024 * 1024)
                table.add_row("📊 Size", f"{size_mb:.1f} MB")
                
        elif result.service_id:  # Cloud mode
            if result.endpoint_url:
                table.add_row("🌐 Endpoint", result.endpoint_url)
            if result.service_id:
                table.add_row("☁️  Service ID", result.service_id)
            if result.uptime:
                table.add_row("⏱️  Uptime", result.uptime)
            
            # 从 details 提取其他信息
            details = result.details or {}
            if details.get('runtime_name'):
                table.add_row("📛 Runtime Name", details['runtime_name'])
            if details.get('image_url'):
                table.add_row("💿 Image", details['image_url'])
            
            # 显示数据面健康检查状态
            ping_status = details.get('ping_status')
            if ping_status is not None:
                if ping_status is True:
                    health_status = "[green]✔️ Healthy[/green]"
                elif ping_status is False:
                    health_status = "[red]❌ Unhealthy[/red]"
                else:
                    health_status = "[dim]❓ Unknown[/dim]"
                table.add_row("💚 Health Check", health_status)
        
        console.print(table)
        
        # Verbose 模式显示详细信息
        if verbose and result.details:
            console.print(f"\n[dim]ℹ️  Detailed Information:[/dim]")
            import json
            console.print(Panel(
                json.dumps(result.details, indent=2, ensure_ascii=False),
                title="Details",
                border_style="dim"
            ))
        
        console.print()  # 空行

    except Exception as e:
        console.print(f"[red]❌ Status query failed: {e}[/red]")
        raise typer.Exit(1)
