# Copyright (c) 2025 Beijing Volcano Engine Technology Co., Ltd. and/or its affiliates.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

from dataclasses import dataclass, field
from typing import Optional, Union, Dict
from .dataclass_utils import AutoSerializableMixin


@dataclass
class DockerBuildConfig(AutoSerializableMixin):
    """Docker构建配置
    
    用于自定义Docker镜像构建过程，支持以下功能：
    1. 自定义基础镜像（base_image）
    2. 执行自定义构建脚本（build_script）
    
    Examples:
        Python项目使用自定义镜像:
        >>> docker_build:
        >>>   base_image: "python:3.11-slim"
        >>>   build_script: "scripts/setup.sh"
        
        Golang项目使用多阶段构建镜像:
        >>> docker_build:
        >>>   base_image:
        >>>     builder: "golang:1.23-alpine"
        >>>     runtime: "alpine:latest"
        >>>   build_script: "scripts/install_certs.sh"
    """
    
    base_image: Optional[Union[str, Dict[str, str]]] = field(
        default=None,
        metadata={
            "description": "自定义基础镜像",
            "detail": (
                "Python项目使用字符串指定单个镜像，例如 'python:3.12-alpine'。"
                "Golang项目可使用dict指定builder和runtime镜像，例如 "
                "{'builder': 'golang:1.23-alpine', 'runtime': 'alpine:latest'}"
            )
        }
    )
    
    build_script: Optional[str] = field(
        default=None,
        metadata={
            "description": "自定义构建脚本路径（相对于项目根目录）",
            "detail": (
                "脚本将在Docker构建过程中执行，可用于安装系统依赖、"
                "编译C扩展、设置环境等。路径相对于项目根目录，"
                "例如 'scripts/setup.sh' 或 'docker/install_deps.sh'"
            )
        }
    )
    
    regenerate_dockerfile: bool = field(
        default=False,
        metadata={
            "description": "强制重新生成 Dockerfile",
            "detail": (
                "当设置为 True 时，即使 Dockerfile 存在也会强制重新生成。"
                "通常通过 CLI 参数 --regenerate-dockerfile 设置。"
            )
        }
    )
