# Copyright (c) 2025 Beijing Volcano Engine Technology Co., Ltd. and/or its affiliates.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"""
Field Mapper - Unified field name mapping for different workflows.

This module provides a centralized way to handle field name differences
between different workflow implementations (local, cloud, hybrid).
"""

from typing import Dict, Any, Optional, List


class FieldMapper:
    """
    Maps different workflow field names to standardized field names.
    
    This class handles the inconsistency in field naming across different
    workflow implementations. Each workflow may use different field names
    for the same logical data (e.g., 'runtime_endpoint' vs 'endpoint_url').
    """
    
    # Standard field mapping configuration
    # Format: {standard_name: [list of possible field names in order of priority]}
    FIELD_MAPPINGS = {
        # Build-related fields
        "image_name": [
            "cr_image_full_url",      # VE workflow
            "full_image_name",        # Local workflow
            "image_name",             # Generic
            "image_url",              # Alternative
        ],
        "image_id": [
            "image_id",
            "image_digest",
            "digest",
        ],
        "image_tag": [
            "image_tag",
            "tag",
        ],
        "build_timestamp": [
            "build_timestamp",
            "timestamp",
            "created_at",
        ],
        
        # Deploy-related fields
        "endpoint_url": [
            "runtime_endpoint",       # VE workflow
            "endpoint",               # VE workflow short
            "endpoint_url",           # Generic
            "service_endpoint",       # Alternative
            "url",                    # Short form
        ],
        "service_id": [
            "runtime_id",             # VE workflow
            "service_id",             # Generic
            "deployment_id",          # Alternative
        ],
        "container_id": [
            "container_id",
        ],
        "container_name": [
            "container_name",
            "runtime_name",           # VE workflow may use this
        ],
        "deploy_timestamp": [
            "deploy_timestamp",
            "timestamp",
            "created_at",
        ],
        
        # Status-related fields
        "status": [
            "status",
            "state",
        ],
        "uptime": [
            "uptime",
            "running_time",
        ],
    }
    
    # Status value mapping - maps workflow-specific status values to standard values
    STATUS_MAPPINGS = {
        "ready": "running",
        "running": "running",
        "stopped": "stopped",
        "starting": "starting",
        "stopping": "stopping",
        "error": "error",
        "failed": "error",
        "unknown": "unknown",
    }
    
    @classmethod
    def get_field(
        cls,
        config: Dict[str, Any],
        standard_name: str,
        default: Any = None
    ) -> Any:
        """
        Get a field value using standard field name.
        
        Tries multiple possible field names for the given standard name
        and returns the first non-None value found.
        
        Args:
            config: Configuration dictionary to search.
            standard_name: Standard field name to look up.
            default: Default value if field not found.
            
        Returns:
            Field value or default if not found.
            
        Example:
            >>> config = {"runtime_endpoint": "http://example.com"}
            >>> FieldMapper.get_field(config, "endpoint_url")
            "http://example.com"
        """
        possible_names = cls.FIELD_MAPPINGS.get(standard_name, [standard_name])
        
        for field_name in possible_names:
            value = config.get(field_name)
            if value is not None:
                # Filter out template variables like {{timestamp}}
                if isinstance(value, str) and "{{" in value and "}}" in value:
                    continue
                return value
        
        return default
    
    @classmethod
    def normalize_status(cls, status: str) -> str:
        """
        Normalize status value to standard status.
        
        Args:
            status: Status value from workflow.
            
        Returns:
            Normalized status value.
            
        Example:
            >>> FieldMapper.normalize_status("Ready")
            "running"
        """
        return cls.STATUS_MAPPINGS.get(status.lower(), status.lower())
    
    @classmethod
    def extract_fields(
        cls,
        config: Dict[str, Any],
        fields: List[str]
    ) -> Dict[str, Any]:
        """
        Extract multiple fields from config using standard names.
        
        Args:
            config: Configuration dictionary.
            fields: List of standard field names to extract.
            
        Returns:
            Dictionary with extracted values.
            
        Example:
            >>> config = {"runtime_endpoint": "http://...", "runtime_id": "123"}
            >>> FieldMapper.extract_fields(config, ["endpoint_url", "service_id"])
            {"endpoint_url": "http://...", "service_id": "123"}
        """
        result = {}
        for field in fields:
            value = cls.get_field(config, field)
            if value is not None:
                result[field] = value
        return result
    
    @classmethod
    def extract_image_tag_from_name(cls, image_name: Optional[str]) -> Optional[str]:
        """
        Extract tag from full image name.
        
        Args:
            image_name: Full image name (e.g., "registry/repo:tag").
            
        Returns:
            Tag part or None.
            
        Example:
            >>> FieldMapper.extract_image_tag_from_name("myimage:latest")
            "latest"
            >>> FieldMapper.extract_image_tag_from_name("registry/image:20251105024919")
            "20251105024919"
        """
        if not image_name or ":" not in image_name:
            return None
        
        tag = image_name.split(":")[-1]
        
        # Filter out template variables
        if "{{" in tag:
            return None
        
        # Filter out port numbers (only digits and small numbers)
        # Port numbers are typically < 65536, timestamps are much larger
        if tag.isdigit() and len(tag) <= 5:
            return None
        
        return tag


# Convenience functions for common operations
def get_image_info(config: Dict[str, Any]) -> Dict[str, Optional[str]]:
    """
    Extract all image-related information from config.
    
    Args:
        config: Workflow configuration.
        
    Returns:
        Dictionary with image_name, image_id, image_tag.
    """
    image_name = FieldMapper.get_field(config, "image_name")
    image_tag = FieldMapper.get_field(config, "image_tag")
    
    # If tag is a template variable or not found, try to extract from image_name
    if not image_tag and image_name:
        image_tag = FieldMapper.extract_image_tag_from_name(image_name)
    
    return {
        "image_name": image_name,
        "image_id": FieldMapper.get_field(config, "image_id"),
        "image_tag": image_tag,
        "build_timestamp": FieldMapper.get_field(config, "build_timestamp"),
    }


def get_deploy_info(config: Dict[str, Any]) -> Dict[str, Optional[str]]:
    """
    Extract all deploy-related information from config.
    
    Args:
        config: Workflow configuration.
        
    Returns:
        Dictionary with endpoint_url, service_id, etc.
    """
    return FieldMapper.extract_fields(
        config,
        ["endpoint_url", "service_id", "container_id", "container_name", "deploy_timestamp"]
    )


def get_status_info(status_data: Dict[str, Any]) -> Dict[str, Any]:
    """
    Extract and normalize status information.
    
    Args:
        status_data: Status data from workflow.
        
    Returns:
        Dictionary with normalized status information.
    """
    status = FieldMapper.get_field(status_data, "status", "unknown")
    normalized_status = FieldMapper.normalize_status(str(status))
    
    result = {
        "status": normalized_status,
        "endpoint_url": FieldMapper.get_field(status_data, "endpoint_url"),
        "service_id": FieldMapper.get_field(status_data, "service_id"),
        "container_id": FieldMapper.get_field(status_data, "container_id"),
        "uptime": FieldMapper.get_field(status_data, "uptime"),
    }
    
    return {k: v for k, v in result.items() if v is not None}
