# Copyright (c) 2025 Beijing Volcano Engine Technology Co., Ltd. and/or its affiliates.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"""Build operation result model."""

from dataclasses import dataclass, field
from typing import Optional, List, Dict, Any
from datetime import datetime


@dataclass
class BuildResult:
    """Result of a build operation."""
    
    success: bool
    """Whether the build was successful."""
    
    image_name: Optional[str] = None
    """Full image name (e.g., 'registry/namespace/repo:tag')."""
    
    image_id: Optional[str] = None
    """Image ID or digest."""
    
    image_tag: Optional[str] = None
    """Image tag."""
    
    build_timestamp: Optional[datetime] = None
    """When the build completed."""
    
    build_logs: Optional[List[str]] = None
    """Build logs if available."""
    
    error: Optional[str] = None
    """Error message if build failed."""
    
    error_type: Optional[str] = None
    """Error type classification (e.g., 'ValueError', 'WorkflowErrorType.BUILD_FAILURE')."""
    
    metadata: Dict[str, Any] = field(default_factory=dict)
    """Additional metadata (workflow-specific)."""
    
    def __bool__(self) -> bool:
        """Allow using result in boolean context: if result: ..."""
        return self.success
    
    def __str__(self) -> str:
        """String representation."""
        if self.success:
            return f"BuildResult(success=True, image={self.image_name})"
        else:
            return f"BuildResult(success=False, error={self.error})"
