# Copyright (c) 2025 Beijing Volcano Engine Technology Co., Ltd. and/or its affiliates.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"""Deploy operation result model."""

from dataclasses import dataclass, field
from typing import Optional, List, Dict, Any
from datetime import datetime


@dataclass
class DeployResult:
    """Result of a deploy operation."""
    
    success: bool
    """Whether the deployment was successful."""
    
    container_id: Optional[str] = None
    """Container ID (for local deployments)."""
    
    container_name: Optional[str] = None
    """Container name."""
    
    service_id: Optional[str] = None
    """Service ID (for cloud deployments)."""
    
    endpoint_url: Optional[str] = None
    """Deployed service endpoint URL."""
    
    deploy_timestamp: Optional[datetime] = None
    """When the deployment completed."""
    
    deploy_logs: Optional[List[str]] = None
    """Deployment logs if available."""
    
    error: Optional[str] = None
    """Error message if deployment failed."""
    
    error_type: Optional[str] = None
    """Error type classification (e.g., 'ValueError', 'WorkflowErrorType.DEPLOY_FAILURE')."""
    
    metadata: Dict[str, Any] = field(default_factory=dict)
    """Additional metadata (workflow-specific)."""
    
    def __bool__(self) -> bool:
        """Allow using result in boolean context: if result: ..."""
        return self.success
    
    def __str__(self) -> str:
        """String representation."""
        if self.success:
            endpoint = self.endpoint_url or self.container_id or self.service_id
            return f"DeployResult(success=True, endpoint={endpoint})"
        else:
            return f"DeployResult(success=False, error={self.error})"
