# Copyright (c) 2025 Beijing Volcano Engine Technology Co., Ltd. and/or its affiliates.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"""Init operation result model."""

from dataclasses import dataclass, field
from typing import Optional, List, Dict, Any


@dataclass
class InitResult:
    """Result of a project initialization operation."""
    
    success: bool
    """Whether the initialization was successful."""
    
    project_name: Optional[str] = None
    """Name of the initialized project."""
    
    template: Optional[str] = None
    """Template used for initialization."""
    
    project_path: Optional[str] = None
    """Path to the initialized project."""
    
    created_files: List[str] = field(default_factory=list)
    """List of files created during initialization."""
    
    error: Optional[str] = None
    """Error message if initialization failed."""
    
    error_type: Optional[str] = None
    """Error type classification."""
    
    metadata: Dict[str, Any] = field(default_factory=dict)
    """Additional metadata."""
    
    def __bool__(self) -> bool:
        """Allow using result in boolean context: if result: ..."""
        return self.success
    
    def __str__(self) -> str:
        """String representation."""
        if self.success:
            return f"InitResult(success=True, project={self.project_name}, files={len(self.created_files)})"
        else:
            return f"InitResult(success=False, error={self.error})"
    
    # ========== Convenience Methods for SDK Users ==========
    
    @property
    def config_file(self) -> Optional[str]:
        """Get path to the generated configuration file.
        
        Returns:
            Path to agentkit.yaml if initialization succeeded, None otherwise.
        
        Example:
            >>> init_result = AgentKitClient.init_project(...)
            >>> config_path = init_result.config_file
            >>> print(config_path)  # "./my_agent/agentkit.yaml"
        """
        if self.success and self.project_path:
            return f"{self.project_path}/agentkit.yaml"
        return None
    
    def load_config(self) -> 'AgentConfig':
        """Load the configuration as an AgentConfig object.
        
        This is a convenience method that loads the generated agentkit.yaml
        file as an AgentConfig instance, ready for reading or modification.
        
        Returns:
            AgentConfig instance for the initialized project.
        
        Raises:
            ValueError: If initialization failed or no config file exists.
        
        Example:
            >>> init_result = AgentKitClient.init_project(
            ...     project_name="my_agent",
            ...     template="basic_stream"
            ... )
            >>> 
            >>> # Load config and modify
            >>> config = init_result.load_config()
            >>> config.launch_type = "hybrid"
            >>> config.save()
            >>> 
            >>> # Create client
            >>> client = AgentKitClient(config)
        """
        if not self.success or not self.config_file:
            raise ValueError(
                "Cannot load config: initialization failed or no config file available. "
                f"Initialization success: {self.success}, config_file: {self.config_file}"
            )
        
        from agentkit.toolkit.sdk.config import AgentConfig
        return AgentConfig.load(self.config_file)
    
    def create_client(
        self,
        launch_type: Optional[str] = None,
        **config_overrides
    ) -> 'AgentKitClient':
        """Create an AgentKitClient for this project.
        
        This is a convenience method that creates a client configured for
        the initialized project. Optionally override configuration values.
        
        Args:
            launch_type: Override launch type ('local', 'cloud', 'hybrid').
            **config_overrides: Additional configuration overrides.
        
        Returns:
            AgentKitClient instance ready to use.
        
        Raises:
            ValueError: If initialization failed.
        
        Example:
            >>> init_result = AgentKitClient.init_project(...)
            >>> 
            >>> # Create client with default config
            >>> client = init_result.create_client()
            >>> 
            >>> # Create client with overrides
            >>> client = init_result.create_client(launch_type="hybrid")
            >>> 
            >>> # Use the client
            >>> client.launch()
            >>> client.invoke({"prompt": "Hello"})
        """
        if not self.success:
            raise ValueError(
                f"Cannot create client: initialization failed. Error: {self.error}"
            )
        
        from agentkit.toolkit.sdk.client import AgentKitClient
        
        # Build config_dict for overrides
        config_dict = None
        if launch_type or config_overrides:
            config_dict = {}
            
            if launch_type:
                config_dict["common"] = {"launch_type": launch_type}
            
            # Merge additional overrides
            if config_overrides:
                for key, value in config_overrides.items():
                    if key in config_dict:
                        # Merge nested dicts
                        if isinstance(config_dict[key], dict) and isinstance(value, dict):
                            config_dict[key].update(value)
                        else:
                            config_dict[key] = value
                    else:
                        config_dict[key] = value
        
        return AgentKitClient(
            config_file=self.config_file,
            config_dict=config_dict
        )
