# Copyright (c) 2025 Beijing Volcano Engine Technology Co., Ltd. and/or its affiliates.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"""Invoke operation result model."""

from dataclasses import dataclass, field
from typing import Optional, Any, Dict, Iterator


@dataclass
class InvokeResult:
    """Result of an invoke operation."""
    
    success: bool
    """Whether the invocation was successful."""
    
    response: Optional[Any] = None
    """Response data (dict for non-streaming, generator for streaming)."""
    
    is_streaming: bool = False
    """Whether the response is a streaming generator."""
    
    error: Optional[str] = None
    """Error message if invocation failed."""
    
    error_type: Optional[str] = None
    """Error type classification (e.g., 'ValueError', 'WorkflowErrorType.INVOKE_FAILURE')."""
    
    metadata: Dict[str, Any] = field(default_factory=dict)
    """Additional metadata."""
    
    def __bool__(self) -> bool:
        """Allow using result in boolean context: if result: ..."""
        return self.success
    
    def stream(self) -> Iterator[Any]:
        """
        Iterate over streaming response.
        
        Returns:
            Iterator yielding response events.
            
        Raises:
            ValueError: If response is not streaming.
        """
        if not self.is_streaming:
            raise ValueError("Response is not streaming")
        
        if self.response is None:
            return iter([])
        
        return iter(self.response)
    
    def get_response(self) -> Any:
        """
        Get the full response.
        For streaming responses, this will consume the entire stream.
        
        Returns:
            The response data.
        """
        if self.is_streaming and hasattr(self.response, '__iter__'):
            # Consume and collect streaming response
            return list(self.response)
        return self.response
    
    def __str__(self) -> str:
        """String representation."""
        if self.success:
            resp_type = "streaming" if self.is_streaming else "complete"
            return f"InvokeResult(success=True, type={resp_type})"
        else:
            return f"InvokeResult(success=False, error={self.error})"
