# Copyright (c) 2025 Beijing Volcano Engine Technology Co., Ltd. and/or its affiliates.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"""Lifecycle operation result model."""

from dataclasses import dataclass, field
from typing import Optional, Dict, Any


@dataclass
class LifecycleResult:
    """Result of a lifecycle operation (launch, destroy, etc.)."""
    
    success: bool
    """Whether the operation was successful."""
    
    operation: str = ""
    """Operation type (launch, destroy, stop, etc.)."""
    
    message: Optional[str] = None
    """Success or informational message."""
    
    error: Optional[str] = None
    """Error message if operation failed."""
    
    details: Dict[str, Any] = field(default_factory=dict)
    """Additional operation details."""
    
    def __bool__(self) -> bool:
        """Allow using result in boolean context: if result: ..."""
        return self.success
    
    def __str__(self) -> str:
        """String representation."""
        if self.success:
            return f"LifecycleResult(operation={self.operation}, success=True)"
        else:
            return f"LifecycleResult(operation={self.operation}, success=False, error={self.error})"
