# Copyright (c) 2025 Beijing Volcano Engine Technology Co., Ltd. and/or its affiliates.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"""Build service - Core business logic for building agent images."""

from typing import Optional, Dict, Any

from ..models.build_result import BuildResult
from ..field_mapper import get_image_info
from .base_service import BaseService


class BuildService(BaseService):
    """Service for building agent images."""
    
    def build(
        self,
        config_dict: Optional[Dict[str, Any]] = None,
        config_file: Optional[str] = None,
        platform: str = "auto",
        regenerate_dockerfile: bool = False
    ) -> BuildResult:
        """
        Build agent image.
        
        Args:
            config_dict: Configuration dictionary (highest priority).
            config_file: Configuration file path.
            platform: Build platform (auto/local/cloud).
            regenerate_dockerfile: Force regenerate Dockerfile.
            
        Returns:
            BuildResult: Build operation result.
        """
        try:
            # Load and validate configuration
            config = self._load_config(config_dict, config_file)
            self._validate_config(config)
            
            # Get workflow
            common_config = config.get_common_config()
            workflow_name = common_config.launch_type
            workflow_class = self._get_workflow(workflow_name)
            
            # Get workflow config
            workflow_config = config.get_workflow_config(workflow_name)
            
            # Set regenerate_dockerfile on docker_build_config (runtime parameter, not persisted)
            config.set_docker_build_runtime_param("regenerate_dockerfile", regenerate_dockerfile)
            
            # Execute build
            self.logger.info(f"Starting build with workflow '{workflow_name}'")
            
            # Create workflow instance with progress reporting
            from ...workflows.progress import ProgressReporter, create_cli_progress_listener
            progress_reporter = ProgressReporter()
            progress_reporter.add_listener(create_cli_progress_listener())
            
            # Instantiate workflow with dependencies
            if callable(workflow_class):
                workflow_instance = workflow_class(
                    config_manager=config,
                    progress_reporter=progress_reporter,
                    logger=self.logger
                )
            else:
                # Old-style: already instantiated
                workflow_instance = workflow_class
                workflow_instance.progress = progress_reporter
                workflow_instance.config_manager = config
                workflow_instance.logger = self.logger
            
            # Call workflow and get BuildInfo
            build_info = workflow_instance.build(workflow_config)
            
            # Convert BuildInfo to BuildResult
            return BuildResult(
                success=build_info.success,
                image_name=build_info.image_name,
                image_id=build_info.image_id,
                image_tag=build_info.image_tag,
                build_timestamp=build_info.build_timestamp,
                error=build_info.error,
                error_type=str(build_info.error_type) if build_info.error_type else None,
                metadata={
                    "workflow": workflow_name,
                    "details": build_info.details
                }
            )
                
        except Exception as e:
            error_info = self._handle_exception("Build", e)
            return BuildResult(**error_info)
