# Copyright (c) 2025 Beijing Volcano Engine Technology Co., Ltd. and/or its affiliates.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"""Lifecycle service - Core business logic for agent lifecycle management."""

import logging
from typing import Optional, Dict, Any

from ..models.lifecycle_result import LifecycleResult
from .build_service import BuildService
from .deploy_service import DeployService
from .base_service import BaseService


logger = logging.getLogger(__name__)


class LifecycleService(BaseService):
    """Service for agent lifecycle management (launch, destroy, etc.)."""
    
    def __init__(self):
        """Initialize lifecycle service."""
        super().__init__()
        self.build_service = BuildService()
        self.deploy_service = DeployService()
    
    def launch(
        self,
        config_dict: Optional[Dict[str, Any]] = None,
        config_file: Optional[str] = None,
        platform: str = "auto"
    ) -> LifecycleResult:
        """
        Launch agent (build + deploy in one operation).
        
        Args:
            config_dict: Configuration dictionary (highest priority).
            config_file: Configuration file path.
            platform: Build platform (auto/local/cloud).
            
        Returns:
            LifecycleResult: Launch operation result.
        """
        try:
            self.logger.info("Starting launch operation (build + deploy)")
            
            # Step 1: Build
            build_result = self.build_service.build(
                config_dict=config_dict,
                config_file=config_file,
                platform=platform
            )
            
            if not build_result.success:
                return LifecycleResult(
                    success=False,
                    operation="launch",
                    error=f"Build failed: {build_result.error}",
                    details={"build_result": build_result}
                )
            
            self.logger.info("Build completed, starting deployment")
            
            # Step 2: Deploy
            deploy_result = self.deploy_service.deploy(
                config_dict=config_dict,
                config_file=config_file
            )
            
            if not deploy_result.success:
                return LifecycleResult(
                    success=False,
                    operation="launch",
                    error=f"Deploy failed: {deploy_result.error}",
                    details={
                        "build_result": build_result,
                        "deploy_result": deploy_result
                    }
                )
            
            # Success
            return LifecycleResult(
                success=True,
                operation="launch",
                message=f"Successfully launched agent: {deploy_result.endpoint_url or deploy_result.container_id}",
                details={
                    "build_result": build_result,
                    "deploy_result": deploy_result
                }
            )
                
        except Exception as e:
            self.logger.error(f"Launch error: {e}", exc_info=True)
            return LifecycleResult(
                success=False,
                operation="launch",
                error=str(e)
            )
    
    def destroy(
        self,
        config_dict: Optional[Dict[str, Any]] = None,
        config_file: Optional[str] = None,
        force: bool = False
    ) -> LifecycleResult:
        """
        Destroy agent runtime and resources.
        
        Args:
            config_dict: Configuration dictionary (highest priority).
            config_file: Configuration file path.
            force: Force destroy without confirmation.
            
        Returns:
            LifecycleResult: Destroy operation result.
        """
        try:
            # Load configuration
            config = self._load_config(config_dict, config_file)
            common_config = config.get_common_config()
            
            # Get workflow
            workflow_name = common_config.launch_type
            workflow = self._get_workflow(workflow_name)
            
            # Execute destroy
            self.logger.info(f"Destroying runtime with workflow '{workflow_name}'")
            
            # Get workflow config
            workflow_config = config.get_workflow_config(workflow_name)
            
            # Instantiate workflow if it's a class
            if callable(workflow):
                workflow_instance = workflow(config_manager=config, logger=self.logger)
            else:
                workflow_instance = workflow
                workflow_instance.config_manager = config
                workflow_instance.logger = self.logger
            
            # Call destroy with unified signature
            workflow_instance.destroy(workflow_config, force=force)
            
            return LifecycleResult(
                success=True,
                operation="destroy",
                message=f"{workflow_name} runtime destroyed successfully",
                details={"workflow": workflow_name}
            )
                
        except Exception as e:
            self.logger.error(f"Destroy error: {e}", exc_info=True)
            return LifecycleResult(
                success=False,
                operation="destroy",
                error=str(e)
            )
    
    def stop(
        self,
        config_dict: Optional[Dict[str, Any]] = None,
        config_file: Optional[str] = None
    ) -> LifecycleResult:
        """
        Stop agent runtime (without destroying resources).
        
        Args:
            config_dict: Configuration dictionary (highest priority).
            config_file: Configuration file path.
            
        Returns:
            LifecycleResult: Stop operation result.
        """
        try:
            # Load configuration
            config = self._load_config(config_dict, config_file)
            common_config = config.get_common_config()
            
            # Get workflow
            workflow_name = common_config.launch_type
            workflow = self._get_workflow(workflow_name)
            
            # Execute stop
            self.logger.info(f"Stopping runtime with workflow '{workflow_name}'")
            
            # Instantiate workflow if it's a class
            if callable(workflow):
                workflow_instance = workflow(config_manager=config, logger=self.logger)
            else:
                workflow_instance = workflow
                workflow_instance.config_manager = config
                workflow_instance.logger = self.logger
            
            # Call stop
            workflow_instance.stop()
            
            return LifecycleResult(
                success=True,
                operation="stop",
                message=f"{workflow_name} runtime stopped successfully",
                details={"workflow": workflow_name}
            )
                
        except Exception as e:
            self.logger.error(f"Stop error: {e}", exc_info=True)
            return LifecycleResult(
                success=False,
                operation="stop",
                error=str(e)
            )
