# Copyright (c) 2025 Beijing Volcano Engine Technology Co., Ltd. and/or its affiliates.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"""
Docker builder implementation
Provides local Docker environment build functionality
"""

import logging
import os
from pathlib import Path
from typing import Dict, Any, Optional, Tuple, List
from dataclasses import dataclass, field
from datetime import datetime
from agentkit.toolkit.config import CommonConfig
from agentkit.toolkit.config.dataclass_utils import AutoSerializableMixin
from .base import Builder
from ..container import DockerManager, DockerfileRenderer
from agentkit.toolkit.config import DockerBuildConfig
import shutil

logger = logging.getLogger(__name__)


@dataclass
class LocalDockerBuilderConfig(AutoSerializableMixin):
    """Docker builder configuration"""
    common_config: Optional[CommonConfig] = field(default=None, metadata={"system": True, "description": "Common configuration"})
    image_name: str = field(default="", metadata={"description": "Image name"})
    image_tag: str = field(default="latest", metadata={"description": "Image tag"})
    dockerfile_path: str = field(default=".", metadata={"description": "Dockerfile directory path"})
    dockerfile_name: str = field(default="Dockerfile", metadata={"description": "Dockerfile filename"})
    template_dir: Optional[str] = field(default=None, metadata={"description": "Dockerfile template directory"})
    template_name: str = field(default="Dockerfile.j2", metadata={"description": "Dockerfile template filename"})

@dataclass
class LocalDockerBuilderResult(AutoSerializableMixin):
    """Docker builder result"""
    success: bool = field(default=False, metadata={"description": "Build success status"})
    image_id: Optional[str] = field(default=None, metadata={"description": "Built image ID"})
    build_logs: Optional[List[str]] = field(default=None, metadata={"description": "Build logs"})
    build_timestamp: Optional[str] = field(default=None, metadata={"description": "Build timestamp"})
    full_image_name: Optional[str] = field(default=None, metadata={"description": "Full image name"})

class LocalDockerBuilder(Builder):
    """Docker builder implementation"""
    
    def __init__(self):
        super().__init__()
        self.docker_manager = DockerManager()
        self.dockerfile_renderer = None
    
    
    def build(self, config: Dict[str, Any]) -> Tuple[bool, Dict[str, Any]]:
        """Build Docker image"""
        docker_config = LocalDockerBuilderConfig.from_dict(config)
        
        common_config = None
        if docker_config.common_config is not None:
            common_config = CommonConfig.from_dict(docker_config.common_config)
        
        # 获取Docker构建配置（可选）
        docker_build_config = None
        if "docker_build_config" in config:
            docker_build_config = DockerBuildConfig.from_dict(config["docker_build_config"])
        
        # 获取 regenerate_dockerfile 参数
        force_regenerate = False
        if docker_build_config:
            force_regenerate = docker_build_config.regenerate_dockerfile
        
        if common_config is None:
            return False, LocalDockerBuilderResult(success=False, build_logs=["Missing common configuration"]).to_dict()
        
        # Check if Docker is available before attempting to build
        docker_available, docker_message = self.docker_manager.is_docker_available()
        if not docker_available:
            logger.error(f"Docker availability check failed")
            # Split multi-line error message into list for better display
            error_lines = docker_message.split('\n')
            return False, LocalDockerBuilderResult(
                success=False, 
                build_logs=error_lines
            ).to_dict()
        
        try:
            if common_config.language == "Python":
                template_dir = os.path.join(os.path.dirname(__file__), "..", "..", "resources", "templates", "python")
            elif common_config.language == "Golang":
                template_dir = os.path.join(os.path.dirname(__file__), "..", "..", "resources", "templates", "golang")
            else:
                return False, LocalDockerBuilderResult(success=False, build_logs=[f"Unsupported language: {common_config.language}"]).to_dict()

            try:
                from agentkit.toolkit.integrations.container import DockerfileRenderer, DockerManager
            except ImportError:
                return False, LocalDockerBuilderResult(success=False, build_logs=["Missing Docker dependencies"]).to_dict()

            try:
                renderer = DockerfileRenderer(template_dir)
            except Exception:
                return False, LocalDockerBuilderResult(success=False, build_logs=["Missing Dockerfile renderer"]).to_dict()

            context = {
                "language_version": common_config.language_version,
            }
            
            # 注入Docker构建配置参数
            if docker_build_config:
                # 处理基础镜像
                if docker_build_config.base_image:
                    if common_config.language == "Golang" and isinstance(docker_build_config.base_image, dict):
                        # Golang多阶段构建：支持builder和runtime镜像
                        context["base_image_builder"] = docker_build_config.base_image.get("builder")
                        context["base_image_runtime"] = docker_build_config.base_image.get("runtime")
                    else:
                        # Python或Golang使用单个镜像
                        context["base_image"] = docker_build_config.base_image
                
                # 处理构建脚本
                if docker_build_config.build_script:
                    build_script_path = self.workdir / docker_build_config.build_script
                    if build_script_path.exists():
                        context["build_script"] = docker_build_config.build_script
                    else:
                        logger.warning(f"Build script not found: {docker_build_config.build_script}")

            if common_config.language == "Python":
                context["agent_module_path"] = os.path.splitext(common_config.entry_point)[0]
                if common_config.dependencies_file:
                    # 确保dependencies_file存在，使用相对于构建上下文的路径
                    dependencies_file_path = self.workdir / common_config.dependencies_file
                    if not dependencies_file_path.exists():
                        dependencies_file_path.write_text("")
                    # 在Docker构建上下文中使用相对路径
                    context["dependencies_file"] = common_config.dependencies_file
            
            if common_config.language == "Golang":
                entry_path = (self.workdir / common_config.entry_point).resolve()
                if not entry_path.exists():
                    candidate = self.workdir / common_config.entry_point
                    candidate = candidate if candidate.exists() else self.workdir
                    found = None
                    for p in [candidate] + list(candidate.parents):
                        if (Path(p) / "go.mod").exists():
                            found = Path(p)
                            break
                    if found:
                        entry_path = found.resolve()
                    else:
                        return False, LocalDockerBuilderResult(success=False, build_logs=[f"Project path not found: {common_config.entry_point}"]).to_dict()

                src_dest = self.workdir / "src"
                src_dest.mkdir(parents=True, exist_ok=True)

                if entry_path.is_file() and entry_path.suffix == ".sh":
                    project_root = entry_path.parent
                    entry_relative_path = str((Path("src") / project_root.name / entry_path.name).as_posix())
                elif entry_path.is_dir():
                    project_root = entry_path
                    entry_relative_path = str((Path("src") / project_root.name).as_posix())
                else:
                    return False, LocalDockerBuilderResult(
                        success=False,
                        build_logs=[f"Unsupported Go entry: single-file compilation is not supported. Provide project directory or build.sh"]
                    ).to_dict()

                binary_name = (common_config.agent_name or project_root.name)

                try:
                    proj_res = project_root.resolve()
                    src_res = src_dest.resolve()
                except Exception:
                    proj_res = project_root
                    src_res = src_dest

                target_subdir = src_dest / project_root.name
                target_subdir.mkdir(parents=True, exist_ok=True)
                for child in project_root.iterdir():
                    try:
                        if child.resolve() == src_res:
                            continue
                    except Exception:
                        if child == src_dest:
                            continue
                    dest = target_subdir / child.name
                    if child.is_dir():
                        shutil.copytree(child, dest, dirs_exist_ok=True, symlinks=True)
                    else:
                        dest.parent.mkdir(parents=True, exist_ok=True)
                        shutil.copy2(child, dest)

                context.update({
                    "entry_relative_path": entry_relative_path,
                    "binary_name": binary_name,
                    "agent_module_path": f"/usr/local/bin/{binary_name}"
                })

            # === 使用 DockerfileManager 管理 Dockerfile ===
            from agentkit.toolkit.integrations.dockerfile import DockerfileManager
            
            # 准备配置哈希字典（用于检测配置变化）
            config_hash_dict = {
                'language': common_config.language,
                'language_version': common_config.language_version,
                'entry_point': common_config.entry_point,
                'dependencies_file': common_config.dependencies_file,
            }
            if docker_build_config:
                config_hash_dict['docker_build'] = {
                    'base_image': docker_build_config.base_image,
                    'build_script': docker_build_config.build_script,
                }
            
            # 定义内容生成函数（闭包捕获上下文）
            def generate_dockerfile_content() -> str:
                """生成 Dockerfile 内容（不含元数据头）"""
                from io import StringIO
                output = StringIO()
                
                # 使用 renderer 渲染到字符串
                template = renderer.env.get_template(docker_config.template_name)
                rendered = template.render(**context)
                return rendered
            
            # 使用 DockerfileManager 准备 Dockerfile
            dockerfile_manager = DockerfileManager(self.workdir, self.logger)
            generated, dockerfile_path = dockerfile_manager.prepare_dockerfile(
                config_hash_dict=config_hash_dict,
                content_generator=generate_dockerfile_content,
                force_regenerate=force_regenerate
            )
            
            # 确保 .dockerignore 存在（如果需要）
            dockerignore_path = self.workdir / ".dockerignore"
            if not dockerignore_path.exists():
                renderer.create_dockerignore(str(dockerignore_path))
            image_name = f"{docker_config.image_name or 'agentkit-app'}"
            image_tag = f"{docker_config.image_tag or 'latest'}"
            
            success, build_logs, image_id = self.docker_manager.build_image(
                dockerfile_path=str(self.workdir),
                image_name=image_name,
                image_tag=image_tag,
                build_args={}
            )
            
            if success:
                return True, LocalDockerBuilderResult(
                    success=True,
                    image_id=image_id,
                    full_image_name=f"{image_name}:{image_tag}",
                    build_timestamp=datetime.now().isoformat(),
                    build_logs=build_logs,
                ).to_dict()
            else:
                return False, LocalDockerBuilderResult(
                    success=False,
                    build_logs=build_logs,
                    build_timestamp=datetime.now().isoformat()
                ).to_dict()
                
        except Exception as e:
            return False, LocalDockerBuilderResult(
                success=False,
                build_logs=[str(e)],
                build_timestamp=datetime.now().isoformat()
            ).to_dict()
    
    def check_artifact_exists(self, config: Dict[str, Any]) -> bool:
        """Check if build artifact exists"""
        try:
            exists, image_info, actual_image_id = self.docker_manager.check_image_exists(
                config['full_image_name'], None
            )
            return exists
        except Exception as e:
            self.logger.error(f"Error checking image existence: {str(e)}")
            return False
    
    def remove_artifact(self, config: Dict[str, Any]) -> bool:
        """Remove Docker image"""
        try:
            return self.docker_manager.remove_image(config['full_image_name'], force=True)
        except Exception as e:
            self.logger.error(f"Error removing image: {str(e)}")
            return False