# Copyright (c) 2025 Beijing Volcano Engine Technology Co., Ltd. and/or its affiliates.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import os
from datetime import datetime
from typing import Any, Dict, List, Tuple
from pathlib import Path

from agentkit.toolkit.workflows import Workflow
from agentkit.toolkit.workflows.models import BuildInfo, DeployInfo, StatusInfo, InvokeInfo, WorkflowErrorType
from agentkit.toolkit.workflows.progress import ProgressLevel
from agentkit.toolkit.config import get_config, CommonConfig, LocalDockerConfig


class LocalWorkflow(Workflow):
    """Local Docker workflow implementation"""
    
    def __init__(self, config_manager=None, progress_reporter=None, logger=None):
        super().__init__(config_manager, progress_reporter, logger)

    def prompt_for_config(self, current_config: Dict[str, Any] = None) -> Dict[str, Any]:
        """Generate interactive configuration based on dataclass"""
        from agentkit.toolkit.config.auto_prompt import generate_config_from_dataclass
        
        if current_config is None:
            current_config = {}

        return generate_config_from_dataclass(LocalDockerConfig, current_config)

    def build(self, config: Dict[str, Any]) -> BuildInfo:
        """Build the agent image using LocalDockerBuilder."""
        try:
            from agentkit.toolkit.integrations.builder.local_docker_builder import (
                LocalDockerBuilder, LocalDockerBuilderConfig, LocalDockerBuilderResult
            )
        except ImportError as e:
            self.logger.error(f"Docker dependencies missing: {e}")
            self.progress.report("依赖错误: Docker未安装", 0, level=ProgressLevel.ERROR)
            return BuildInfo(
                success=False,
                error="Docker dependencies missing, install agentkit[docker] extras",
                error_type=WorkflowErrorType.DEPENDENCY_ERROR
            )
            
        try:
            self.progress.report("开始构建镜像", 0, level=ProgressLevel.INFO)
            
            agent_config = self.config_manager or get_config()
            common_config = agent_config.get_common_config()
            docker_config = LocalDockerConfig.from_dict(config)
            
            self.progress.report("准备构建配置", 20, level=ProgressLevel.INFO)
            
            # 构建基础配置
            builder_config = LocalDockerBuilderConfig(
                common_config=common_config,
                image_name=common_config.agent_name or "agentkit-app",
                image_tag=docker_config.image_tag
            ).to_dict()
            
            # 添加Docker构建配置（如果存在）
            docker_build_config = agent_config.get_docker_build_config()
            builder_config["docker_build_config"] = docker_build_config.to_dict()

            self.progress.report("执行构建", 40, level=ProgressLevel.INFO)
            builder = LocalDockerBuilder()
            success, build_result = builder.build(builder_config)
            
            if success:
                result = LocalDockerBuilderResult.from_dict(build_result)
                docker_config.full_image_name = result.full_image_name
                docker_config.image_id = result.image_id
                docker_config.build_timestamp = result.build_timestamp
                agent_config.update_workflow_config("local", docker_config.to_persist_dict())
                
                self.progress.report("构建完成", 100, level=ProgressLevel.SUCCESS)
                
                # Extract tag from image name
                image_tag = docker_config.image_tag
                
                return BuildInfo(
                    success=True,
                    image_name=result.full_image_name,
                    image_id=result.image_id,
                    image_tag=image_tag,
                    build_timestamp=result.build_timestamp,
                    details={"docker_config": docker_config.to_dict()}
                )
            else:
                result = LocalDockerBuilderResult.from_dict(build_result)
                error_msg = "Build failed"
                if result.build_logs:
                    if isinstance(result.build_logs, list):
                        error_msg = "\n".join([log for log in result.build_logs if log.strip()])
                    else:
                        error_msg = str(result.build_logs)
                
                self.logger.error(f"Build failed: {error_msg}")
                self.progress.report("构建失败", 0, level=ProgressLevel.ERROR)
                
                return BuildInfo(
                    success=False,
                    error=error_msg,
                    error_type=WorkflowErrorType.BUILD_FAILURE,
                    details={"build_logs": result.build_logs if result.build_logs else []}
                )
                
        except Exception as e:
            self.logger.error(f"Build error: {e}", exc_info=True)
            self.progress.report(f"构建错误: {e}", 0, level=ProgressLevel.ERROR)
            return BuildInfo(
                success=False,
                error=str(e),
                error_type=WorkflowErrorType.UNKNOWN_ERROR,
                details={"exception": type(e).__name__}
            )

    def deploy(self, config: Dict[str, Any]) -> DeployInfo:
        """Deploy agent image to local Docker container"""
        try:
            from agentkit.toolkit.integrations.runner.local_docker_runner import (
                LocalDockerRunner, LocalDockerRunnerConfig, LocalDockerDeployResult
            )
        except ImportError as e:
            self.logger.error(f"Docker dependencies missing: {e}")
            self.progress.report("依赖错误: Docker未安装", 0, level=ProgressLevel.ERROR)
            return DeployInfo(
                success=False,
                error="Docker dependencies missing, install agentkit[docker] extras",
                error_type=WorkflowErrorType.DEPENDENCY_ERROR
            )
            
        try:
            self.progress.report("开始部署容器", 0, level=ProgressLevel.INFO)
            
            docker_config = LocalDockerConfig.from_dict(config)
            agent_config = self.config_manager or get_config()
            common_config = agent_config.get_common_config()
            
            runner_config = self._build_runner_config(docker_config, common_config)
            runner = LocalDockerRunner()
            
            self.progress.report("部署容器", 50, level=ProgressLevel.INFO)
            success, deploy_result = runner.deploy(runner_config)
            
            result = LocalDockerDeployResult.from_dict(deploy_result)
            
            if success:
                docker_config.container_id = result.container_id
                docker_config.container_name = result.container_name
                docker_config.deploy_timestamp = result.deploy_timestamp
                agent_config.update_workflow_config("local", docker_config.to_persist_dict())
                
                self.progress.report("部署完成", 100, level=ProgressLevel.SUCCESS)
                
                return DeployInfo(
                    success=True,
                    container_id=result.container_id,
                    container_name=result.container_name,
                    endpoint_url=f"http://localhost:{docker_config.invoke_port}",
                    deploy_timestamp=result.deploy_timestamp,
                    details={"docker_config": docker_config.to_dict()}
                )
            else:
                error_msg = result.error_message or "Deployment failed"
                self.logger.error(f"Deploy failed: {error_msg}")
                self.progress.report("部署失败", 0, level=ProgressLevel.ERROR)
                
                return DeployInfo(
                    success=False,
                    error=error_msg,
                    error_type=WorkflowErrorType.DEPLOY_FAILURE
                )
                
        except Exception as e:
            self.logger.error(f"Deploy error: {e}", exc_info=True)
            self.progress.report(f"部署错误: {e}", 0, level=ProgressLevel.ERROR)
            return DeployInfo(
                success=False,
                error=str(e),
                error_type=WorkflowErrorType.UNKNOWN_ERROR,
                details={"exception": type(e).__name__}
            )
        
    def invoke(self, config: Dict[str, Any] = None, args: Dict[str, Any] = None) -> InvokeInfo:
        """Invoke the workflow with given configuration and arguments."""
        try:
            from agentkit.toolkit.integrations.runner.local_docker_runner import LocalDockerRunner, LocalDockerRunnerConfig
        except ImportError as e:
            self.logger.error(f"Docker dependencies missing: {e}")
            self.progress.report("依赖错误: Docker未安装", 0, level=ProgressLevel.ERROR)
            return InvokeInfo(
                success=False,
                error="缺少Docker相关依赖，请安装agentkit[docker] extras",
                error_type=WorkflowErrorType.DEPENDENCY_ERROR
            )
            
        try:
            agent_config = get_config()
            common_config = agent_config.get_common_config()
            docker_config = LocalDockerConfig.from_dict(config)
            
            payload = args.get("payload") if args else None
            headers = args.get("headers") if args else None
            
            runner_config = self._build_runner_config(docker_config, common_config)
            runner = LocalDockerRunner()
            success, response_data = runner.invoke(runner_config, payload, headers)
            
            if success:
                # 检测是否是流式响应（生成器）
                is_streaming = hasattr(response_data, '__iter__') and not isinstance(response_data, (dict, str, list, bytes))
                
                return InvokeInfo(
                    success=True,
                    response=response_data,
                    is_streaming=is_streaming
                )
            else:
                error_msg = str(response_data)
                self.logger.error(f"Invocation failed: {error_msg}")
                self.progress.report(f"调用失败: {error_msg}", 0, level=ProgressLevel.ERROR)
                return InvokeInfo(
                    success=False,
                    error=error_msg,
                    error_type=WorkflowErrorType.INVOKE_FAILURE
                )
                
        except Exception as e:
            self.logger.error(f"Invoke error: {e}", exc_info=True)
            self.progress.report(f"调用错误: {e}", 0, level=ProgressLevel.ERROR)
            return InvokeInfo(
                success=False,
                error=str(e),
                error_type=WorkflowErrorType.UNKNOWN_ERROR,
                details={"exception": type(e).__name__}
            )

    def status(self, config: Dict[str, Any] = None) -> StatusInfo:
        """Get local Docker deployment status"""
        try:
            from agentkit.toolkit.integrations.runner.local_docker_runner import LocalDockerRunner, LocalDockerRunnerConfig
        except ImportError as e:
            self.logger.error(f"Docker dependencies missing: {e}")
            self.progress.report("依赖错误: Docker未安装", 0, level=ProgressLevel.ERROR)
            return StatusInfo(
                success=False,
                status="unknown",
                error="缺少Docker相关依赖，请安装agentkit[docker] extras",
                error_type=WorkflowErrorType.DEPENDENCY_ERROR,
                details={'docker_available': False}
            )
        
        try:
            agent_config = get_config()
            if config is None:
                config = agent_config.get_workflow_config("local")
                
            docker_config = LocalDockerConfig.from_dict(config)
            common_config = agent_config.get_common_config()
            runner_config = self._build_runner_config(docker_config, common_config)
            
            runner = LocalDockerRunner()
            status_dict = runner.status(runner_config)
            
            # 转换 runner 返回的 dict 为 StatusInfo
            # LocalDockerRunner.status 返回结构: {deploy: {status, container_id}, build: {...}, ...}
            deploy_info = status_dict.get('deploy', {})
            build_info = status_dict.get('build', {})
            
            # 提取状态
            container_status = deploy_info.get('status', '')
            container_exists = deploy_info.get('exists', False)
            
            # 映射到标准状态
            if container_status == 'running':
                status = 'running'
            elif container_exists:
                status = 'stopped'
            else:
                status = 'not_deployed'
            
            # 构建 endpoint URL
            endpoint_url = None
            ports = deploy_info.get('ports', {})
            if container_status == 'running' and ports:
                # 提取第一个端口映射
                for container_port, host_bindings in ports.items():
                    if host_bindings and isinstance(host_bindings, list) and len(host_bindings) > 0:
                        host_port = host_bindings[0].get('HostPort', '')
                        if host_port:
                            endpoint_url = f"http://localhost:{host_port}"
                            break
            
            # 计算运行时长
            uptime = None
            if container_status == 'running':
                created_str = deploy_info.get('created', '')
                if created_str:
                    try:
                        from datetime import datetime
                        created_time = datetime.fromisoformat(created_str.replace('Z', '+00:00'))
                        now = datetime.now(created_time.tzinfo)
                        uptime_delta = now - created_time
                        days = uptime_delta.days
                        hours, remainder = divmod(uptime_delta.seconds, 3600)
                        minutes, _ = divmod(remainder, 60)
                        if days > 0:
                            uptime = f"{days}d {hours}h {minutes}m"
                        elif hours > 0:
                            uptime = f"{hours}h {minutes}m"
                        else:
                            uptime = f"{minutes}m"
                    except Exception:
                        pass
            
            return StatusInfo(
                success=True,
                status=status,
                container_id=deploy_info.get('container_id'),
                endpoint_url=endpoint_url,
                uptime=uptime,
                details={
                    'container': {
                        'name': status_dict.get('project_name'),
                        'status': container_status,
                        'ports': ports,
                        'created': deploy_info.get('created')
                    },
                    'image': {
                        'name': status_dict.get('image_name'),
                        'id': build_info.get('image_id'),
                        'tags': build_info.get('tags', []),
                        'size': build_info.get('size', 0),
                        'created': build_info.get('created')
                    },
                    'system': status_dict.get('system', {})
                }
            )
            
        except Exception as e:
            self.logger.error(f"Status query error: {e}", exc_info=True)
            self.progress.report(f"状态查询错误: {e}", 0, level=ProgressLevel.ERROR)
            return StatusInfo(
                success=False,
                status="unknown",
                error=str(e),
                error_type=WorkflowErrorType.UNKNOWN_ERROR,
                details={"exception": type(e).__name__}
            )

    def _build_runner_config(self, docker_config: LocalDockerConfig, common_config: CommonConfig) -> Dict[str, Any]:
        """Build LocalDockerRunner configuration object"""
        from agentkit.toolkit.integrations.runner.local_docker_runner import LocalDockerRunnerConfig
        from agentkit.toolkit.config import merge_runtime_envs
        
        # 合并应用级和 Workflow 级环境变量
        merged_envs = merge_runtime_envs(common_config, docker_config.to_dict())
        
        return LocalDockerRunnerConfig(
            common_config=common_config,
            invoke_port=docker_config.invoke_port,
            full_image_name=docker_config.full_image_name,
            image_name=common_config.agent_name or "agentkit-app",
            image_tag=docker_config.image_tag,
            container_name=docker_config.container_name,
            container_id=docker_config.container_id,
            image_id=docker_config.image_id,
            environment=merged_envs,
            ports=getattr(docker_config, 'ports', None),
            volumes=getattr(docker_config, 'volumes', None),
            restart_policy=getattr(docker_config, 'restart_policy', None),
            memory_limit=getattr(docker_config, 'memory_limit', None),
            cpu_limit=getattr(docker_config, 'cpu_limit', None)
        ).to_dict()

    def stop(self, config: Dict[str, Any] = None) -> bool:
        """Stop the workflow.
        
        Args:
            config: 工作流配置（可选）
        
        Returns:
            bool: 是否成功停止
        
        Note:
            Local 模式下通常不需要停止操作，直接返回 True
        """
        self.logger.info("Local workflow stop - no action needed")
        return True
    
    def destroy(self, config: Dict[str, Any] = None, force: bool = False) -> bool:
        """Stop and destroy workflow resources"""
        try:
            from agentkit.toolkit.config import get_config
            try:
                from agentkit.toolkit.integrations.runner.local_docker_runner import LocalDockerRunner, LocalDockerRunnerConfig
            except ImportError as e:
                self.logger.error(f"Docker dependencies missing: {e}")
                return False
            
            agent_config = self.config_manager or get_config()
            if config is None:
                config = agent_config.get_workflow_config("local")
            docker_config = LocalDockerConfig.from_dict(config)
            common_config = agent_config.get_common_config()
            
            runner_config = self._build_runner_config(docker_config, common_config)
            runner = LocalDockerRunner()
            runner.destroy(runner_config)
            
            # Clear configuration state
            docker_config.container_id = None
            docker_config.deploy_timestamp = None
            docker_config.image_id = None
            docker_config.build_timestamp = None
            docker_config.full_image_name = None
            
            agent_config.update_workflow_config("local", docker_config.to_persist_dict())
            
            # Clean up Dockerfile
            try:
                dockerfile_path = Path.cwd() / "Dockerfile"
                if dockerfile_path.exists():
                    dockerfile_path.unlink()
            except Exception:
                pass
            
            self.logger.info("Workflow destroyed successfully")
            return True
                
        except Exception as e:
            self.logger.error(f"Destruction error: {e}", exc_info=True)
            if not force:
                raise
            return False