"""SDK to interact with the Fewsats Marketplace API"""

# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/01_client.ipynb.

# %% auto 0
__all__ = ['FewsatsMarketplaceClient']

# %% ../nbs/01_client.ipynb 4
class FewsatsMarketplaceClient:
    """
    Client for the Fewsats Marketplace API.
    
    This client provides methods to interact with the Fewsats Marketplace API,
    including product search, cart management, and checkout functionality.
    """
    
    def __init__(self, base_url: str = "http://localhost:8000/api/v1", private_key: str = ''):
        """
        Initialize the Fewsats Marketplace client.
        
        Args:
            base_url: The base URL of the Fewsats Marketplace API.
            customer_id: The unique identifier for the customer. If not provided,
                         a random UUID will be generated.
        """
        self.base_url = base_url

        cfg = get_cfg()

        if private_key: self.pk, self.pub = from_pk_hex(private_key) # if provided use the private key
        elif cfg.priv: self.pk, self.pub = from_pk_hex(cfg.priv) # if not provided use the private key from the config file
        else: 
            self.pk, self.pub = generate_keys()
            save_cfg({'priv': priv_key_hex(self.pk)})

        self.mongo_id = self.pub[:24] # agora uses mongoDB so the related user-objectId needs to be 24 characters

        self.client = httpx.Client(timeout=30.0)
        self.client.headers.update({"customuserid": self.mongo_id})

    def _make_request(self, method: str, endpoint: str, **kwargs) -> httpx.Response:
        """
        Make a request to the Fewsats Marketplace API.
        
        Args:
            method: The HTTP method to use.
            endpoint: The API endpoint to call.
            **kwargs: Additional arguments to pass to httpx.
            
        Returns:
            The response from the API.
        """
        url = f"{self.base_url}/{endpoint}"
        response = self.client.request(method, url, **kwargs)
        
        
        return response
        
    def search_products(self, query: str, count: int = 20, page: int = 1, 
                         price_min: int = 0, price_max: Optional[int] = None, 
                         sort: Optional[str] = None, order: Optional[str] = None) -> Dict:
        """
        Search for products.
        
        Args:
            query: The search query.
            count: The number of products to return per page.
            page: The page number.
            price_min: The minimum price.
            price_max: The maximum price.
            sort: The sort field.
            order: The sort order.
            
        Returns:
            The search results.
        """
        params = {
            "q": query,
            "count": count,
            "page": page,
            "price_min": price_min
        }
        
        if price_max is not None:
            params["price_max"] = price_max
        if sort is not None:
            params["sort"] = sort
        if order is not None:
            params["order"] = order
            
        return self._make_request("GET", "search", params=params)


# %% ../nbs/01_client.ipynb 6
@patch    
def get_product_detail(self: FewsatsMarketplaceClient, slug: str) -> Dict:
    """
    Get details for a specific product.
    
    Args:
        slug: The product slug.
        
    Returns:
        The product details.
    """
    params = {"slug": slug}
    return self._make_request("GET", "product-detail", params=params)


# %% ../nbs/01_client.ipynb 8
@patch
def get_cart(self: FewsatsMarketplaceClient) -> Dict:
    """
    Get the current user's cart.
    
    Returns:
        The cart details.
    """
    return self._make_request("GET", "cart")


# %% ../nbs/01_client.ipynb 10
@patch
def add_to_cart(self: FewsatsMarketplaceClient, product_id: str, variant_id: str, title: str, 
                description: str, amount: int, quantity: int = 1,
                currency: str = "USD") -> Dict:
    """
    Add an item to the user's cart.
    
    Args:
        product_id: The product ID.
        variant_id: The product variant ID.
        title: The product title.
        description: The product description.
        amount: The product price in cents.
        quantity: The quantity to add.
        currency: The currency code.
        
    Returns:
        The updated cart.
    """
    item = {
        "product_id": product_id,
        "variant_id": variant_id,
        "title": title,
        "description": description,
        "amount": amount,
        "quantity": quantity,
        "currency": currency
    }
    
    return self._make_request("POST", "cart/items", json=item)


# %% ../nbs/01_client.ipynb 12
@patch
def update_cart_item(self: FewsatsMarketplaceClient, product_id: str, variant_id: str, quantity: int) -> Dict:
    """
    Update the quantity of an item in the cart.
    
    Args:
        product_id: The product ID.
        variant_id: The product variant ID.
        quantity: The new quantity.
        
    Returns:
        The updated cart.
    """
    params = {
        "product_id": product_id,
        "variant_id": variant_id,
        "quantity": quantity
    }
    
    return self._make_request("PUT", "cart/items", params=params)
    


# %% ../nbs/01_client.ipynb 13
@patch
def clear_cart(self: FewsatsMarketplaceClient) -> Dict:
    """
    Clear all items from the cart.
    
    Returns:
        The response from the API.
    """
    return self._make_request("DELETE", "cart")
    


# %% ../nbs/01_client.ipynb 14
@patch
def buy_now(self: FewsatsMarketplaceClient, product_id: str, variant_id: str, shipping_address: Dict,
            user: Dict, quantity: int = 1) -> Dict:
    """
    Purchase a product directly.
    
    Args:
        product_id: The product ID.
        variant_id: The product variant ID.
        quantity: The quantity to purchase.
        shipping_address: The shipping address. If not provided, a dummy address will be used.
        user: The user information. If not provided, dummy user info will be used.
        
    Example:
        shipping_address = {
            "addressFirst": "123 Main St",
            "city": "New York",
            "state": "NY",
            "country": "US",
            "addressName": "Home",
            "zipCode": "10001"
        }
        
        user = {
            "firstname": "John",
            "lastname": "Doe",
            "email": "john@example.com",
            "_id": "user123"
        }
        
        fm.buy_now(product_id, variant_id, shipping_address, user)
        
    Returns:
        The payment information.
    """

        
    request_data = {
        "currency": "USD",
        "product_id": product_id,
        "variant_id": variant_id,
        "quantity": quantity,
        "shipping_address": shipping_address,
        "user": user
    }
    
    return self._make_request("POST", "buy-now", json=request_data)

# %% ../nbs/01_client.ipynb 17
@patch
def get_order(self: FewsatsMarketplaceClient, order_id: int) -> Dict:
    """
    Get details for a specific order.
    
    Args:
        order_id: The order ID.
        
    Returns:
        The order details.
    """
    return self._make_request("GET", f"orders/{order_id}")
    

@patch
def get_user_orders(self: FewsatsMarketplaceClient) -> List[Dict]:
    """
    Get all orders for the current user.
    
    Returns:
        A list of orders.
    """
    return self._make_request("GET", f"users/{self.customer_id}/orders")
    
