"""SDK to interact with the Agora-Fewsats Marketplace API"""

# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/00_client.ipynb.

# %% auto 0
__all__ = ['Agora']

# %% ../nbs/00_client.ipynb 3
import httpx
from typing import Dict, List
from .config import *
from .crypto import *

from fastcore.utils import *


# %% ../nbs/00_client.ipynb 5
class Agora:
    """
    Client for the Agora-Fewsats Marketplace API.
    
    This client provides methods to interact with the Agora-Fewsats Marketplace API,
    including product search, cart management, and checkout functionality.
    """
    
    def __init__(self, base_url: str = "https://agora-backend.replit.app/api/v1", private_key: str = ''):
        """
        Initialize the Agora Marketplace client.
        
        Args:
            base_url: The base URL of the Agora-Fewsats Backend API.
            private_key: The private key of the Agora-Fewsats Backend API. If not provided,
                         a random UUID will be generated.
        """
        self.base_url = base_url

        cfg = get_cfg()

        if private_key: self.pk, self.pub = from_pk_hex(private_key) # if provided use the private key
        elif cfg.priv: self.pk, self.pub = from_pk_hex(cfg.priv) # if not provided use the private key from the config file
        else: 
            self.pk, self.pub = generate_keys()
            save_cfg({'priv': priv_key_hex(self.pk)})

        self.customer_user_id = self.pub[:24] # agora uses mongoDB so the related user-objectId needs to be 24 characters

        self.client = httpx.Client(timeout=30.0)
        self.client.headers.update({"customuserid": self.customer_user_id})

    def _make_request(self, method: str, endpoint: str, **kwargs) -> httpx.Response:
        """
        Make a request to the Agora-Fewsats Backend API.
        
        Args:
            method: The HTTP method to use.
            endpoint: The API endpoint to call.
            **kwargs: Additional arguments to pass to httpx.
            
        Returns:
            The response from the API.
        """
        url = f"{self.base_url}/{endpoint}"
        response = self.client.request(method, url, **kwargs)
        
        
        return response

# %% ../nbs/00_client.ipynb 8
@patch
def search_products(self: Agora, query: str, count: int = 20, page: int = 1, 
                        price_min: int = 0, price_max: int = None, 
                        sort: str = None, order: str = None) -> Dict:
    """
    Search for products.
    
    Args:
        query: The search query.
        count: The number of products to return per page.
        page: The page number.
        price_min: The minimum price.
        price_max: The maximum price.
        sort: The sort field.
        order: The sort order.
        
    Returns:
        The search results.
    """
    params = {
        "q": query,
        "count": count,
        "page": page,
        "price_min": price_min
    }
    
    if price_max is not None:
        params["price_max"] = price_max
    if sort is not None:
        params["sort"] = sort
    if order is not None:
        params["order"] = order
    
    print(params)
    return self._make_request("GET", "search", params=params)


# %% ../nbs/00_client.ipynb 12
@patch    
def get_product_detail(self: Agora, slug: str) -> Dict:
    """
    Get details for a specific product.
    
    Args:
        slug: The product slug.
        
    Returns:
        The product details.
    """
    params = {"slug": slug}
    return self._make_request("GET", "product-detail", params=params)


# %% ../nbs/00_client.ipynb 15
@patch
def get_cart(self: Agora) -> Dict:
    """
    Get the current user's cart.
    
    Returns:
        The cart details.
    """
    return self._make_request("GET", "cart")


# %% ../nbs/00_client.ipynb 18
@patch
def add_to_cart(self: Agora, slug: str, product_id: str, variant_id: str = None, quantity: int = 1) -> Dict:
    """
    Add an item to the user's cart. Somem products do not have variants, in such cases use the product_id as variant_id too.
    
    Args:
        slug: The product slug.
        product_id: The product ID.
        variant_id: The product variant ID.
        quantity: The quantity to add.
        
    Returns:
        The updated cart.
    """
    item = {
        "slug": slug,
        "product_id": product_id,
        "variant_id": variant_id if variant_id else product_id,
        "quantity": quantity,
    }
    
    return self._make_request("POST", "cart/items", json=item)


# %% ../nbs/00_client.ipynb 21
@patch
def update_cart_item(self: Agora, slug: str, product_id: str, variant_id: str, quantity: int) -> Dict:
    """
    Update the quantity of an item in the cart. Some products do not have variants, in such cases use the product_id as variant_id too.
    
    Args:
        slug: The product slug.
        product_id: The product ID.
        variant_id: The product variant ID.
        quantity: The new quantity.
        
    Returns:
        The updated cart.
    """
    params = {
        "slug": slug,
        "product_id": product_id,
        "variant_id": variant_id,
        "quantity": quantity
    }
    
    return self._make_request("PUT", "cart/items", params=params)
    


# %% ../nbs/00_client.ipynb 23
@patch
def clear_cart(self: Agora) -> Dict:
    """
    Clear all items from the cart.
    
    Returns:
        The response from the API.
    """
    return self._make_request("DELETE", "cart")

# %% ../nbs/00_client.ipynb 26
@patch
def buy_now(self: Agora, slug: str, product_id: str, variant_id: str, shipping_address: Dict,
            user: Dict, quantity: int = 1) -> Dict:
    """
    Purchase a product directly. Some products do not have variants, in such cases use the product_id as variant_id too.
    
    Args:
        slug: The product slug.
        product_id: The product ID.
        variant_id: The product variant ID.
        quantity: The quantity to purchase.
        shipping_address: The shipping address.
        user: The user information.
        
    Example:
        shipping_address = {
            "addressFirst": "123 Main St",
            "city": "New York",
            "state": "NY",
            "country": "US",
            "addressName": "Home",
            "zipCode": "10001"
        }
        
        user = {
            "firstname": "John",
            "lastname": "Doe",
            "email": "john@example.com",
        }
        
        a.buy_now(product_id, variant_id, shipping_address, user)
        
    Returns:
        The payment information.
    """

        
    request_data = {
        "slug": slug,
        "product_id": product_id,
        "variant_id": variant_id,
        "quantity": quantity,
        "shipping_address": shipping_address,
        "user": user
    }
    
    return self._make_request("POST", "buy-now", json=request_data)

# %% ../nbs/00_client.ipynb 32
@patch
def get_user_orders(self: Agora) -> List[Dict]:
    """
    Get all orders for the current user.
    
    Returns:
        A list of orders.
    """
    return self._make_request("GET", f"users/orders")

# %% ../nbs/00_client.ipynb 35
@patch
def get_order(self: Agora, external_id: str) -> Dict:
    """
    Get details for a specific order.
    
    Args:
        external_id: The external ID of the order.
        
    Returns:
        The order details.
    """
    return self._make_request("GET", f"orders/{external_id}")

# %% ../nbs/00_client.ipynb 38
@patch
def get_user_info(self: Agora) -> Dict:
    """
    Get the current user's profile and shipping addresses.
    
    Returns:
        Dict containing user profile info (firstname, lastname, email) and list of shipping addresses
    """
    return self._make_request("GET", "user/info")
