import os
from pathlib import Path

from aiohttp import web
# {% if database.is_pg_sqlalchemy %}
from aiopg.sa import create_engine
from sqlalchemy.engine.url import URL
# {% endif %}

# {% if template_engine.is_jinja %}
import aiohttp_jinja2
from aiohttp_jinja2 import APP_KEY as JINJA2_APP_KEY
import jinja2
# {% endif %}
# {% if session.is_secure %}
import base64
import aiohttp_session
from aiohttp_session.cookie_storage import EncryptedCookieStorage
# {% endif %}

import trafaret as t
from trafaret_config import read_and_validate

# {% if example.is_message_board %}
from .views import index, messages, message_data
# {% else %}
from .views import index
# {% endif %}


THIS_DIR = Path(__file__).parent
BASE_DIR = THIS_DIR.parent
SETTINGS_FILE = BASE_DIR / 'settings.yml'  # type: Path

DEV_DICT = t.Dict()
DEV_DICT.allow_extra('*')
ENV_PREFIX = 'APP_'

SETTINGS_STRUCTURE = t.Dict({
    # the "dev" dictionary contains information used by aiohttp-devtools to serve your app locally
    # you may wish to use it yourself,
    # eg. you might use dev.static_path in a management script to deploy static assets
    'dev': DEV_DICT,
    # {% if database.is_none and example.is_message_board %}
    'message_file':  t.String() >> (lambda f: BASE_DIR / f),
    # {% elif database.is_pg_sqlalchemy %}
    'database': t.Dict({
        'name': t.String,
        'password': t.String,
        'user': t.String,
        'host': t.String,
        'port': t.Int(gte=0) >> str,
    }),
    # {% endif %}
    # {% if session.is_secure %}
    'cookie_secret_key': t.String,
    # {% endif %}
})


def substitute_environ(s_dict: dict, prefix: str) -> dict:
    """
    Substitute environment variables into a settings dict.

    Names are searched hierarchically with underscores representing levels, environment variables must be
    capitalised.

    For example lets say we have ` {'foo': 'bar', 'subdict': {'value': 123}}` with prefix 'APP_',
    the environment variable "APP_FOO = spam" would replace "bar" and "APP_SUBDICT_VALUE = 3"
    would be converted to int and replace 123 in the dict.

    :param: s_dict: dict to replace values in
    :param: prefix: required prefix for environment variables to
    :return: modified dict
    """
    for key, value in s_dict.items():
        if isinstance(value, dict):
            s_dict[key] = substitute_environ(value, prefix + key + '_')
        elif isinstance(value, list):
            # doesn't make sense, we can't do anything here
            pass
        else:
            env_var = os.getenv((prefix + key).upper(), None)
            if env_var is not None:
                # basic attempt to convert the new value to match the original type
                if isinstance(value, int):
                    s_dict[key] = int(env_var)
                else:
                    # are there any other types we might need to cope with here?
                    s_dict[key] = env_var
    return s_dict


def load_settings() -> dict:
    """
    Read settings.yml and, validation its content.
    :return: settings dict
    """
    settings_file = SETTINGS_FILE.resolve()
    settings = read_and_validate(str(settings_file), SETTINGS_STRUCTURE)
    settings = substitute_environ(settings, ENV_PREFIX)
    return settings

# {% if template_engine.is_jinja %}


@jinja2.contextfilter
def reverse_url(context, name, **parts):
    """
    jinja2 filter for generating urls,
    see http://aiohttp.readthedocs.io/en/stable/web.html#reverse-url-constructing-using-named-resources

    Usage:
    {%- raw %}

      {{ 'the-view-name'|url }} might become "/path/to/view"

    or with parts and a query

      {{ 'item-details'|url(id=123, query={'active': 'true'}) }} might become "/items/1?active=true
    {%- endraw %}

    see app/templates.index.jinja for usage.

    :param context: see http://jinja.pocoo.org/docs/dev/api/#jinja2.contextfilter
    :param name: the name of the route
    :param parts: url parts to be passed to route.url(), if parts includes "query" it's removed and passed seperately
    :return: url as generated by app.route[<name>].url(parts=parts, query=query)
    """
    app = context['app']

    kwargs = {}
    if 'query' in parts:
        kwargs['query'] = parts.pop('query')
    if parts:
        kwargs['parts'] = parts
    return app.router[name].url(**kwargs)


@jinja2.contextfilter
def static_url(context, static_file_path):
    """
    jinja2 filter for generating urls for static files. NOTE: heed the warning in create_app about "static_root_url"
    as this filter uses app['static_root_url'].

    Usage:

    {%- raw %}
      {{ 'styles.css'|static }} might become "http://mycdn.example.com/styles.css"
    {%- endraw %}

    see app/templates.index.jinja for usage.

    :param context: see http://jinja.pocoo.org/docs/dev/api/#jinja2.contextfilter
    :param static_file_path: path to static file under static route
    :return: roughly just "<static_root_url>/<static_file_path>"
    """
    app = context['app']
    try:
        static_url = app['static_root_url']
    except KeyError:
        raise RuntimeError('app does not define a static root url "static_root_url"')
    return '{}/{}'.format(static_url.rstrip('/'), static_file_path.lstrip('/'))
# {% endif %}


# {% if database.is_pg_sqlalchemy %}
def pg_dsn(db_settings: dict) -> str:
    """
    :param db_settings: dict of connection settings, see SETTINGS_STRUCTURE for definition
    :return: DSN url suitable for sqlalchemy and aiopg.
    """
    return str(URL(
        database=db_settings['name'],
        password=db_settings['password'],
        host=db_settings['host'],
        port=db_settings['port'],
        username=db_settings['user'],
        drivername='postgres',
    ))


async def startup(app: web.Application):
    app['pg_engine'] = await create_engine(pg_dsn(app['database']), loop=app.loop)


async def cleanup(app: web.Application):
    app['pg_engine'].close()
    await app['pg_engine'].wait_closed()
# {% endif %}


def setup_routes(app):
    app.router.add_get('/', index, name='index')
    # {% if example.is_message_board %}
    app.router.add_route('*', '/messages', messages, name='messages')
    app.router.add_get('/messages/data', message_data, name='message-data')
    # {% endif %}


def create_app(loop):
    app = web.Application(loop=loop)
    app['name'] = '{{ name }}'
    app.update(load_settings())
    # {% if template_engine.is_jinja %}

    jinja2_loader = jinja2.FileSystemLoader(str(THIS_DIR / 'templates'))
    aiohttp_jinja2.setup(app, loader=jinja2_loader, app_key=JINJA2_APP_KEY)
    app[JINJA2_APP_KEY].filters.update(
        url=reverse_url,
        static=static_url,
    )
    # {% endif %}
    # {% if database.is_pg_sqlalchemy %}

    app.on_startup.append(startup)
    app.on_cleanup.append(cleanup)
    # {% endif %}
    # {% if session.is_secure %}

    secret_key = base64.urlsafe_b64decode(app['cookie_secret_key'])
    aiohttp_session.setup(app, EncryptedCookieStorage(secret_key))
    # {% endif %}

    setup_routes(app)
    return app
