from __future__ import annotations

import asyncio
from typing import Optional

import grpc
from grpc import aio as grpc_aio

from ameide_core_proto.agents.v1 import agents_pb2_grpc as agents_grpc
from ameide_core_proto.agents_runtime.v1 import agents_runtime_service_pb2_grpc as agents_runtime_grpc
from ameide_core_proto.governance.v1 import governance_service_pb2_grpc as governance_grpc
from ameide_core_proto.graph.v1 import graph_service_pb2_grpc as graph_grpc
from ameide_core_proto.inference.v1 import inference_service_pb2_grpc as inference_grpc
from ameide_core_proto.platform.v1 import invitations_pb2_grpc as invitations_grpc
from ameide_core_proto.platform.v1 import organizations_pb2_grpc as organizations_grpc
from ameide_core_proto.platform.v1 import roles_pb2_grpc as roles_grpc
from ameide_core_proto.platform.v1 import teams_pb2_grpc as teams_grpc
from ameide_core_proto.platform.v1 import tenants_pb2_grpc as tenants_grpc
from ameide_core_proto.platform.v1 import users_pb2_grpc as users_grpc
from ameide_core_proto.threads.v1 import threads_service_pb2_grpc as threads_grpc
from ameide_core_proto.transformation.v1 import transformation_service_pb2_grpc as transformation_grpc
from ameide_core_proto.workflows_runtime.v1 import workflows_service_pb2_grpc as workflows_grpc

from .config import SDKOptions
from .interceptors import auth_interceptor, metadata_interceptor, timeout_interceptor, tracing_interceptor
from .retry import retry_interceptor


class AmeideClient:
    """High level client exposing strongly typed stubs for platform services."""

    def __init__(self, options: Optional[SDKOptions] = None) -> None:
        self._options = options or SDKOptions()
        self._async = bool(self._options.use_async_channel)
        self._channel = self._create_channel()
        self.agents = self._wrap_stub(agents_grpc.AgentsServiceStub)
        self.organizations = self._wrap_stub(organizations_grpc.OrganizationServiceStub)
        self.organization_roles = self._wrap_stub(roles_grpc.OrganizationRoleServiceStub)
        self.teams = self._wrap_stub(teams_grpc.TeamServiceStub)
        self.users = self._wrap_stub(users_grpc.UserServiceStub)
        self.invitations = self._wrap_stub(invitations_grpc.InvitationServiceStub)
        self.tenants = self._wrap_stub(tenants_grpc.TenantServiceStub)
        self.graph = self._wrap_stub(graph_grpc.GraphServiceStub)
        self.transformation = self._wrap_stub(transformation_grpc.TransformationServiceStub)
        self.inference = self._wrap_stub(inference_grpc.InferenceServiceStub)
        self.agents_runtime = self._wrap_stub(agents_runtime_grpc.AgentsRuntimeServiceStub)
        self.governance = self._wrap_stub(governance_grpc.GovernanceServiceStub)
        self.threads = self._wrap_stub(threads_grpc.ThreadsServiceStub)
        self.workflows = self._wrap_stub(workflows_grpc.WorkflowServiceStub)

    def close(self) -> None:
        if self._async:
            try:
                loop = asyncio.get_running_loop()
            except RuntimeError:
                asyncio.run(self._channel.close())
            else:
                loop.create_task(self._channel.close())
        else:
            self._channel.close()

    async def aclose(self) -> None:
        if self._async:
            await self._channel.close()
        else:
            self._channel.close()

    def _create_channel(self):
        opts = self._options
        channel_opts = [
            ("grpc.max_send_message_length", 16 * 1024 * 1024),
            ("grpc.max_receive_message_length", 16 * 1024 * 1024),
        ]

        if opts.channel is not None:
            base_channel = opts.channel
        elif opts.channel_factory is not None:
            base_channel = opts.channel_factory(opts)
        elif self._async:
            if opts.secure:
                base_channel = grpc_aio.secure_channel(opts.endpoint, grpc.ssl_channel_credentials(), options=channel_opts)
            else:
                base_channel = grpc_aio.insecure_channel(opts.endpoint, options=channel_opts)
        elif opts.secure:
            base_channel = grpc.secure_channel(opts.endpoint, grpc.ssl_channel_credentials(), options=channel_opts)
        else:
            base_channel = grpc.insecure_channel(opts.endpoint, options=channel_opts)

        if self._async and not isinstance(base_channel, grpc_aio.Channel):
            raise TypeError("SDKOptions.use_async_channel requires an async gRPC channel")
        if not self._async and isinstance(base_channel, grpc_aio.Channel):
            raise TypeError("Synchronous AmeideClient cannot consume grpc.aio.Channel instances")

        if self._async:
            return base_channel

        built_in = [
            metadata_interceptor(opts),
            auth_interceptor(opts.auth),
            timeout_interceptor(opts.timeout),
            retry_interceptor(opts.retry),
            tracing_interceptor(opts.telemetry),
        ]
        interceptors = [interceptor for interceptor in built_in if interceptor]
        if opts.interceptors:
            interceptors.extend(opts.interceptors)

        if interceptors:
            return grpc.intercept_channel(base_channel, *interceptors)
        return base_channel

    def _wrap_stub(self, factory):
        return factory(self._channel)
