# Ameide Python SDK

`ameide-sdk-python` ships the unified `AmeideClient` with defaults and service coverage defined in `sdk/README.md` and `sdk/manifest/ameide-client.json`.

## Install
- Buf BSR (primary registry): configure Buf auth in `~/.netrc` (or set `PIP_NETRC=1` + `NETRC=...`) and install from the Buf Python index:
  ```bash
  cat >~/.netrc <<'NETRC'
  machine buf.build
    login user
    password ${BUF_TOKEN:?set BUF_TOKEN to a Buf API token}
  NETRC

  PIP_EXTRA_INDEX_URL=https://buf.build/gen/python \
    pip install ameide-sdk-python==<version>
  ```
- GitHub Packages mirror:
  ```bash
  pip install \
    --extra-index-url https://<USER>:<PAT>@pypi.pkg.github.com/ameideio/simple/ \
    --index-url https://pypi.org/simple \
    ameide-sdk-python==<version>
  ```
- GHCR (OCI) mirror for air-gapped installs:
  ```bash
  printf '%s' "${PAT_WITH_READ_PACKAGES}" | oras login ghcr.io --username "${GITHUB_USER}" --password-stdin
  VERSION=<version>
  oras pull ghcr.io/ameideio/ameide/ameide-sdk-python:${VERSION}
  wheel_path=$(find . -name 'ameide-sdk-python-*.whl' -print -quit)
  pip install "${wheel_path}"
  ```

### Tokens & handling
- PAT scope: `read:packages` (add `write:packages` only when publishing).
- Local dev: keep the PAT in `.env` and feed it to pip (`PIP_INDEX_URL`/`PIP_EXTRA_INDEX_URL`) or inline in the mirror URL; never commit credentials.
- CI: store the PAT in repo/org secrets (e.g., `PKG_GITHUB_COM_TOKEN`) and inject at runtime for mirror installs; PyPI publishes use `PYPI_TOKEN`.

## Usage
```python
from ameide_sdk.client import AmeideClient

client = AmeideClient(
    endpoint="api.ameide.io:443",
    tenant_id="...",
    user_id="...",
    auth=lambda: "...token...",
)
```
The client supports async/sync auth providers, tracing, retries, timeouts, tenant/user metadata, request-id injection, and transport overrides via `channel` or `channel_factory`.
Optional helper factory: `create_element_service(client.graph)` yields convenience methods for elements/relationships without mutating the core client surface.

## Integration settings
Integration tests read `INTEGRATION_GRPC_ADDRESS` (or `NEXT_CORE_GRPC_URL`), plus `INTEGRATION_TENANT_ID`, `INTEGRATION_USER_ID`, `INTEGRATION_ORGANIZATION_ID`, and `INTEGRATION_BEARER_TOKEN`; enable with `RUN_INTEGRATION_TESTS=1`. See `sdk/README.md` for the full option contract.

## Testing
- Sync and run with uv so the `test` extra gets installed from `pyproject.toml`/`uv.lock`:
  ```bash
  cd packages/ameide_sdk_python
  uv run --extra test pytest __tests__/unit __tests__/integration
  ```

## Notes
- Import stubs directly from `ameide_core_proto.*` (exposed by the SDK package).
- The wheel bundles the generated proto/gRPC code so no additional Buf wheels or registry credentials are required beyond the registry hosting `ameide-sdk-python`.
- Align release/process documentation with `backlog/388-ameideio-sdks.md` and keep AmeideClient behavior in sync with `sdk/README.md`.

## Development (committed stubs, synced from BSR)
- Sync stubs from the BSR when working from source: `./packages/ameide_sdk_python/scripts/sync_from_bsr.sh [buf.build/ameideio/ameide:ref]` (requires `buf` access to the module).
- CI runs the same generation step before packing/tests; the generated files are tracked in `src/ameide_core_proto/**` and bundled into the wheel/sdist.

## Versioning
- Source version stays `0.0.0` in Git. CI copies the package to a temp dir, stamps `pyproject.toml` with the tag-derived version, and runs `uv build`/`uv publish`. No committed `VERSION` file or env override participates in version resolution.
- Local builds keep `0.0.0` (for ad-hoc/dev use only). The published artifacts always carry the tag-derived version.
