from __future__ import annotations

from collections.abc import Awaitable, Callable, Mapping, MutableMapping, Sequence
from dataclasses import dataclass, field

import grpc


DEFAULT_ENDPOINT = "api.ameide.io:443"


TokenProvider = Callable[[], str | None | Awaitable[str | None]]
MetadataProvider = Mapping[str, str] | MutableMapping[str, str]


@dataclass
class RetryConfig:
    max_attempts: int = 3
    initial_backoff: float = 0.25  # seconds
    max_backoff: float = 5.0
    backoff_multiplier: float = 2.0
    retryable_codes: Sequence[str] = (
        "UNAVAILABLE",
        "DEADLINE_EXCEEDED",
        "RESOURCE_EXHAUSTED",
        "ABORTED",
    )


@dataclass
class TelemetryConfig:
    tracer: object | None = None
    span_name: str | None = None
    attributes: Mapping[str, str] | MutableMapping[str, str] = field(default_factory=dict)


@dataclass
class SDKOptions:
    endpoint: str = DEFAULT_ENDPOINT
    secure: bool = True
    auth: TokenProvider | None = None
    use_async_channel: bool = False
    tenant_id: str | None = None
    user_id: str | None = None
    metadata: MetadataProvider = field(default_factory=dict)
    timeout: float = 5.0
    retry: RetryConfig = field(default_factory=RetryConfig)
    request_id_provider: Callable[[], str] | None = None
    interceptors: Sequence[object] = field(default_factory=tuple)
    telemetry: TelemetryConfig | None = None
    channel: grpc.Channel | None = None
    channel_factory: Callable[[SDKOptions], grpc.Channel] | None = None
