# Copyright (C) 2023 - 2025 ANSYS, Inc. and/or its affiliates.
# SPDX-License-Identifier: MIT
#
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.

"""Provides additional functions to the SCADE DLL for snesors and context management."""

from pathlib import Path

import ansys.scade.python_wrapper.pydata as data


def generate_c(model: data.Model, c_pathname: Path, banner: str = '') -> None:
    """Generate the C wrapping file for the DLL."""
    with c_pathname.open('w') as f:
        if banner:
            f.write('/* generated by %s */\n' % banner)
            f.write('\n')
        f.write('#include <stdlib.h>\n')
        f.write('\n')
        for op in model.operators:
            f.write('#include "%s"\n' % op.header)
        if model.sensors:
            f.write('#include "%s_sensors.h"\n' % model.prefix)
            f.write('\n')
            for sensor in model.sensors:
                f.write('/* %s */\n' % sensor.path)
                f.write('%s %s;\n' % (sensor.c_type, sensor.c_name))

        for op in model.operators:
            f.write('\n')
            f.write('/* %s */\n' % op.path)

            # [Allocate &] initialize context
            f.write('__declspec(dllexport) void* py_alloc_%s()\n' % op.c_name)
            f.write('{\n')
            if op.context is None:
                f.write('    return 0;\n')
            else:
                # if there is a context, the operator must have a reset function
                assert op.reset
                if not op.reset.parameters:
                    # global context: nothing to allocate
                    if op.init is not None:
                        f.write('    %s();\n' % op.init.c_name)
                    f.write('    return 0;\n')
                else:
                    # regular context to allocate
                    f.write(
                        '    {0} *ctx = ({0}*)malloc(sizeof({0}));\n'.format(op.context.c_type)
                    )
                    if op.init is not None:
                        f.write('    %s(ctx);\n' % op.init.c_name)
                    f.write('    return (void*)ctx;\n')
            f.write('}\n')

            # Free allocated context
            f.write('__declspec(dllexport) void py_free_%s(void *ctx)\n' % op.c_name)
            f.write('{\n')
            f.write('    if (ctx != 0)\n')
            f.write('        free(ctx);\n')
            f.write('}\n')

            if op.context and op.context.ios:
                f.write('\n')
                f.write('__declspec(dllexport) size_t py_offsets_%s[] = {\n' % op.c_name)
                for io in op.context.ios:
                    name = io.c_name if io.c_name else io.m_name
                    f.write('    offsetof(%s, %s),\n' % (op.context.c_type, name))
                f.write('};\n')

        f.write('\n')
        f.write('/* end of file */\n')
