"""Licensed to the Apache Software Foundation (ASF) under one or more contributor
license agreements.  See the NOTICE file distributed with this work for additional
information regarding copyright ownership. The ASF licenses this file to You under the
Apache License, Version 2.0 (the "License"); you may not use this file except in
compliance with the License.  You may obtain a copy of the License at.

http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
"""

import copy
from typing import Final, Optional, List, Dict, Any
# from ..util.validator_utils import ValidatorUtils
from .util.validator_utils import ValidatorUtils
import threading

class Field:
    """This contains the list of pluggable validators to run on a field and any message
    information and variables to perform the validations and generate error messages.
    Instances of this class are configured with a <field> xml element.

    Taken from apache.commons.validator.Field;

    Attributes:
        serializable (bool): Indicates if the object is serializable.
        cloneable (bool): Indicates if the object can be cloned.
        
    """

    __DEFAULT_ARG: Final[str] = "src.apache_commons_validator_python.field_new.Field.DEFAULT"
    #: This is the value that will be used as a key if the Arg name field has no value.

    TOKEN_INDEXED: Final[str] = "[]"
    #: This indicates an indexed property is being referenced.

    _TOKEN_START: Final[str] = "${"
    #: The start of a token.

    _TOKEN_END: Final[str] = "}"
    #: The end of a token.

    _TOKEN_VAR: Final[str] = "var:"
    #: A Variable token.

    serializable = True
    #: The class in the original translation was serializable

    cloneable = True
    #: The class in the original translation was cloneable

    def __init__(self):
        """
        Field constructor.
        """
        self._property: Optional[str] = None
        #: The Field's property name.

        self._indexed_property: Optional[str] = None
        #: The Field's indexed property name.

        self._indexed_list_property: Optional[str] = None
        #: The Field's indexed list property name.

        self._key: Optional[str] = None
        #: The Field's unique key.

        self._depends: Optional[str] = None
        #: a comma separated list of validator's this field depends on.

        self._page: int = 0
        #: The page number

        self._client_validation: bool = True
        #: The flag that indicates whether scripting should be generated by the client for client-side validation.

        self._field_order: Optional[int] = None
        #: The order of the Field in the Form.

        self.__dependency_list: List[str] = []
        #: Internal representation of self._depends String as a List.

        self.dependency_lock: threading.Lock = threading.Lock()
        #: Lock for __dependency_list

        self._h_vars: Dict[str, "Var"] = {}
        #: use get_var_map for modifiable Dict and .vars for unmodifiable map

        self._h_msgs: Dict[str, "Msg"] = {}
        #: use get_msg_map for modifiable DIct and .msgs for unmodifiable map

        self._args: List[Dict[str, "Arg"]] = []
        #: Holds Dicts of arguments. args[0] returns the Dict for the first replacement argument. Starts with a 0 length array so that it will only grow to the size of the highest argument position.
    
    def add_arg(self, arg: "Arg") -> None:
        """Add an Arg to the replacement argument list.

        Args:
            arg (Arg): Validation message's argument.
        """
        # TODO this first if check can go away after arg0, etc. are removed from dtd
        if arg is None or arg.key is None:
            return

        self.__determine_arg_position(arg)
        self.__ensure_args_capacity(arg)

        arg_map: Dict[str, "Arg"] = self._args[arg.position]
        if arg_map is None:
            arg_map = {}
            self._args[arg.position] = arg_map

        name: Final[str] = arg.name
        if name is not None:
            arg_map[name] = arg
        else:
            arg_map[self.__DEFAULT_ARG] = arg

    def add_msg(self, msg: "Msg") -> None:
        """Add a Msg to the Field.

        Args:
            msg (Msg): A validation message.
        """
        self.get_msg_map()[msg.name] = msg

    def add_var(self, arg0, arg1=None, arg2=None) -> None:
        """Add a Var to the Field.
        
        Example Method Call 1:
            v = Var(name, value, js_type)
            add_var(arg0=v)

        Example Method Call 2:
            add_var(arg0=name, arg1=value, arg2=jsType)

        Args:
            arg0 (str | Var): Either the Validator Argument or the name of the validation.
            arg1 (str | None): The Argument's value
            arg2 (str | None): The JavaScript type

        Returns: 
            None
        """
        # from ..var_new import Var
        from .var_new import Var
        
        if isinstance(arg0, str) and arg1 is not None and arg2 is not None:
            v = Var(name=arg0, value=arg1, js_type=arg2)
            self.get_var_map()[v.name] = v
        elif isinstance(arg0, Var) and arg1 is None and arg2 is None:
            self.get_var_map()[arg0.name] = arg0

    # TODO: implement __copy__ and __deepcopy__ to do the following overriding
    def clone(self) -> "Field":
        """Creates and returns a copy of this field.

        Returns:
            A copy of the Field.
        """
        field = copy.deepcopy(self)

        temp_map: Dict[str, "Arg"] = {}
        field._args = temp_map
        for i, _ in enumerate(self._args):
            if self._args[i] is None:
                pass
            else:
                arg_map: Final[Dict[str, "Arg"]] = copy.deepcopy(self._args[i])
                for validator_name, arg in arg_map.items():
                    arg_map[validator_name] = arg.clone()
                field._args[i] = arg_map

        field._args = [
            copy.deepcopy(arg_map) if arg_map else None for arg_map in self._args
        ]

        field._h_msgs = ValidatorUtils.copy_map(self.get_msg_map())
        field._h_vars = ValidatorUtils.copy_map(self.get_var_map())
        return field

    def __determine_arg_position(self, arg: "Arg") -> None:
        """Calculate the position of the Arg and store the position as an 
        attribute of arg.

        Args:
            arg (Arg): the arg to determine the position of

        Returns: 
            None
        """
        position: Final[int] = arg.position

        if position >= 0:
            return

        if self._args is None or len(self._args) == 0:
            arg.position = 0
            return

        if arg.name is None:
            key_name = self.__DEFAULT_ARG
        else:
            key_name = arg.name

        last_position: int = -1
        last_default: int = -1
        for i, _ in enumerate(self._args):
            if self._args[i] is not None and key_name in self._args[i]:
                last_position = i
            if self._args[i] is not None and self.__DEFAULT_ARG in self._args[i]:
                last_default = i

        if last_position < 0:
            last_position = last_default

        last_position += 1
        arg.position = last_position  # allocate the next position

    def __ensure_args_capacity(self, arg: "Arg") -> None:
        """Ensures that the args array can hold the given arg. Resizes the array as
        necessary.

        Args:
            arg (Arg): Determine if the args array is long enough to store this
                        arg's position.

        Returns: 
            None
        """
        if arg.position >= len(self._args):
            self._args.extend([None] * (arg.position + 1 - len(self._args)))

    def generate_key(self) -> None:
        """Generates correct key value.
        
        Returns: 
            None
        """
        if self.is_indexed():
            self._key = (
                self._indexed_list_property + self.TOKEN_INDEXED + "." + self._property
            )
        else:
            self._key = self._property

    def get_arg(self, position: int, key: Optional[str]) -> Optional["Arg"]:
        """Gets the Arg object at the given position. If the key finds a None value,
        then the default value then the default value will be retrieved.

        Args:
            position (int): The Arg number to find.
            key (str | None): The name the Arg is stored under. If not found, the default Arg for the given position (if any) will be retrieved.

        Returns:
            The Arg with the given name and position or None if not found.
        """
        if position >= len(self._args) or self._args[position] is None:
            return None

        if key is None:
            return self._args[position].get(self.__DEFAULT_ARG)
        else:
            return self._args[position].get(key)

    def get_args(self, key: str) -> List[Optional["Arg"]]:
        """Retrieves the Args for the given validator name.

        Args:
            key (str): The validator's args to retrieve.

        Returns:
            A List of Args sorted by the Args' positions.
        """
        arg_list: Final[List["Arg"]] = []

        for i in range(len(self._args)):
            arg_list.append(self.get_arg(i, key))

        return arg_list

    @property
    def dependency_list(self) -> List[str]:
        """Gets an copy (so it is unmodifiable) list of the dependencies in the same
        order they were defined in the parameter passed to depends.

        Returns:
            dependency_list (List[Field]): A lit of the Field's dependencies.
        """
        return self.__dependency_list.copy()

    @property
    def depends(self) -> str | None:
        """Gets the validation rules for this field as a comma separated list.
        (translation of getDepends())

        Returns:
            A comma separated list of validator names.
        """
        return self._depends

    @property
    def field_order(self) -> int | None:
        """Gets the position of the {@code Field} in the validation list. (translation
        of getFieldOrder())

        Returns:
            field_order (Optional[int]): the field position
        """
        return self._field_order

    @property
    def indexed_list_property(self) -> str | None:
        """Gets the indexed property name of the field. (translation of
        getIndexedListProperty())

        Returns:
            indexed_list_property (Optional[str]): the field's indexed list property name.
        """
        return self._indexed_list_property

    @property
    def indexed_property(self) -> str | None:
        """Gets the indexed property name of the field."""
        return self._indexed_property

    @property
    def indexed_property_bean(self, bean) -> List[Any]:
        """Returns an indexed property from the object we're validating.

        Args:
            bean (object): The bean to extract the indexed values from.

        Raises: ValidatorException If there's an error looking up the property
                or, the property found is not indexed.
        """
        # TODO: not implemented
        print("not implemented")

    def __get_indexed_property_size(self, bean) -> Optional[int]:
        """Returns the size of an indexed property from the object we're validating.

        Method Arugments:
            bean (object): The bean to extract the indexed values from.

        Raises:
            ValidatorException If there's an error looking up the property or, the property found is not indexed.
        """
        # TODO: not implemented
        print("not implemented")

    @property
    def key(self) -> str:
        """Gets a unique key based on the property and indexedProperty fields.
        (translation of getKey())

        Returns:
            key (str): a unique key for the field.
        """
        if self._key is None:
            self.generate_key()
        return self._key

    def get_message(self, key: str) -> Optional["Msg"]:
        """Retrieve a message object.

        Args:
     
            key (str): Validation key

        Returns:
            A validation message for a specified validator.
        """
        return self.get_msg_map()[key]

    @property
    def msgs(self) -> Dict[str, "Msg"]:
        """The Field's messages are returned as a copied dictionary so it doesn't modify
        the original.

        Returns:
            Dict[str, "Msg"] of validation messages for the field.
        """
        # TODO double check to make sure this doesn't modify original map
        return self._h_msgs.copy()

    def get_msg(self, key: str) -> Optional[str]:
        """Retrieve a message value.

        Args:
            key (str): Validation key

        Returns:
            A validation message for a specified validator.
        """
        msg: Final["Msg"] = self.get_message(key)
        if msg is None:
            return None
        else:
            return msg.key

    def get_msg_map(self) -> Dict[str, "Msg"]:
        """Returns a Dict of String Msg names to Msg objects."""
        return self._h_msgs

    @property
    def page(self) -> int | None:
        """Gets the page value that the Field is associated with for validation."""
        return self._page

    @property
    def field_property(self) -> str | None:
        """Gets the property name of the field."""
        return self._property

    def get_var(self, main_key: str) -> Optional["Var"]:
        """Retrieve a variable.

        Args:
            main_key (str): the Variable's key

        Returns:
            the Variable
        """
        return self.get_var_map()[main_key]

    def get_var_map(self) -> Dict[str, "Var"]:
        """Returns a Map of String Var names to Var objects."""
        return self._h_vars

    @property
    def vars(self) -> Dict[str, "Var"]:
        """The Field's variables are returned as a copied dictionary so it doesn't
        modify the original.

        Returns:
            Dict[str, "Msg"] of Variable's for a field.
        """
        # TODO double check to make sure this doesn't modify original map
        return self._h_vars.copy()

    def get_var_value(self, main_key: str) -> Optional[str]:
        """Retrieve a variable's value.

        Args:
            main_key (str): the Variable's key

        Returns:
            the Variable's value
        """
        value: Optional[str] = None

        v: "Var" = self.get_var_map()[main_key]
        if v is not None:
            value = v.value
        return value

    def __handle_missing_action(self, name: str):
        """Called when a validator name is used in a depends clause but there is no know
        ValidatorAction configured for that name.

        Args:
            name (str): the name of the validator in the depends list

        Throws: ValidatorException
        """
        from ..validator_exception_new import ValidatorException
        raise ValidatorException(
            f"No ValidatorAction named {name} found for field {self.field_property}"
        )

    @property
    def client_validation(self) -> bool:
        """Determines whether client-side scripting should be generated for this field.
        The default is true.

        Returns true for scripting; otherwise false.
        """
        return self._client_validation

    def is_dependency(self, validator_name: str) -> bool:
        """Checks if the validator is listed as a dependency.

        Args:
            validator_name (str): the name of the validator to check

        Returns: whether the field is dependent on a validator
        """
        return validator_name in self.dependency_list

    def is_indexed(self) -> bool:
        """If there is a value specified for the __indexed_property field then true will
        be returned. otherwise it will be false.

        Returns: whether the Field is indexed.
        """
        return self._indexed_list_property is not None and self._indexed_list_property

    def process(
        self, global_constants: Dict[str, str], constants: Dict[str, str]
    ) -> None:
        """Replace the constants with values in fields and process the depends field to
        create the dependency dict.

        Args:
            global_constants (Dict[str, str])
            constants (Dict[str, str])
        """
        # from ..util.validator_utils import ValidatorUtils
        from .util.validator_utils import ValidatorUtils
        self.generate_key()

        for key1 in constants:
            key2:str = self._TOKEN_START + key1 + self._TOKEN_END
            replace_value:str = constants[key1]

            self.field_property = ValidatorUtils.replace(
                self.field_property, key2, replace_value
            )

            self.__process_vars(key2, replace_value)
            self.__process_message_components(key2, replace_value)

        for key1 in global_constants:
            key2:str = self._TOKEN_START + key1 + self._TOKEN_END
            replace_value:str = global_constants[key1]

            self.field_property = ValidatorUtils.replace(
                self.field_property, key2, replace_value
            )

            self.__process_vars(key2, replace_value)
            self.__process_message_components(key2, replace_value)

        for key1 in self.get_var_map():
            key2:str = (
                self._TOKEN_START + self._TOKEN_VAR + key1 + self._TOKEN_END
            )
            var: Final["Var"] = self.get_var(key1)
            replace_value:str = var.value

            self.__process_message_components(key2, replace_value)

    def __process_arg(self, key: str, replace_value: str) -> None:
        """Replace the arg key value with the key/value pairs passed in.

        Args:
            key (str): the arg key to replace the value of 
            replace_value (str): the new value
        """
        # from ..util.validator_utils import ValidatorUtils
        from .util.validator_utils import ValidatorUtils

        for arg_map in self._args:
            if arg_map is None:
                pass
            else:
                for arg in arg_map.values():
                    if arg is not None:
                        arg.key = ValidatorUtils.replace(arg.key, key, replace_value)

    def __process_message_components(self, key: str, replace_value: str) -> None:
        """Replace the msgs key value with the key/value pairs passed in.

        Args:
            key (str): the msgs key to replace the value of 
            replace_value (str): the new value
        """
        # from ..util.validator_utils import ValidatorUtils
        from .util.validator_utils import ValidatorUtils

        var_key: Final[str] = self._TOKEN_START + self._TOKEN_VAR
        # process messages
        if key is not None and not key.startswith(var_key):
            for msg in self.get_msg_map().values():
                msg.key = ValidatorUtils.replace(msg.key, key, replace_value)

        self.__process_arg(key, replace_value)

    def __process_vars(self, key: str, replace_value: str) -> None:
        """Replace the vars value with the key/value pairs passed in.

        Args:
            key (str): the vars key to replace the value of 
            replace_value (str): the new value
        """
        # from ..util.validator_utils import ValidatorUtils
        from .util.validator_utils import ValidatorUtils

        for var_key in self.get_var_map():
            var: Final["Var"] = self.get_var(var_key)
            var.value = ValidatorUtils.replace(var.value, key, replace_value)

    def __run_dependent_validators(
        self,
        va: 'ValidatorAction',
        results: 'ValidatorResults',
        actions: Dict[str, 'ValidatorAction'],
        params: Dict[str, object],
        pos: int,
    ) -> bool:
        """Calls all of the validators that this validator depends on. TODO
        ValidatorAction should know how to run its own dependencies.

        Args:
            va (ValidatorAction): run dependent validators for this action
            results (ValidatorResults)
            actions (Dict[str, ValidatorAction])
            params (Dict[str, object])
            pos (int)

        Returns:
            True if all dependent validations passed.

        Throws ValidatorException If there's an error running a validator
        """
        dependent_validators: List[str] = va.get_dependencies()

        if len(dependent_validators) == 0:
            return True

        for depend in dependent_validators:
            action: Final["ValidatorAction"] = actions[depend]
            if action is None:
                self.__handle_missing_action(depend)

            if not self.__validate_for_rule(action, results, actions, params, pos):
                return False

        return True

    @client_validation.setter
    def client_validation(self, client_va: bool) -> None:
        """Sets the flag that determines whether client-side scripting should be
        generated for this field. (translation of setClientValidation())

        Args:
            client_va (bool): the scripting flag
        """
        self._client_validation = client_va

    @depends.setter
    def depends(self, depends: str) -> None:
        """Sets the validation rules for this field as a comma separated list/
        (translation of setDepends())

        Args:
            depends (str): a comma separated list of validator names
        """
        self._depends = depends

        self.__dependency_list.clear()
        self.__dependency_list = depends.split(",")

    @field_order.setter
    def field_order(self, field_ord: int) -> None:
        """Sets the position of the Field in the validation list. (translation of
        setFieldOrder())

        Args:
            field_ord (int): the field position
        """
        self._field_order = field_ord

    @indexed_list_property.setter
    def indexed_list_property(self, indexed_list_prop: str) -> None:
        """Sets the indexed property name of the field.

        Args:
            indexed_list_prop (str): the field's indexed list property name.
        """
        self._indexed_list_property = indexed_list_prop

    @indexed_property.setter
    def indexed_property(self, indexed_property: str) -> None:
        """Sets the indexed property name of the field.

        Args:
            indexed_property (str): the field's indexed property name
        """
        self._indexed_property = indexed_property

    @key.setter
    def key(self, key: str) -> None:
        """Sets a unique key for the field. This can be used to change the key
        temporarily to have a unique key for an indexed field.

        Args:
            key (str): a unique key for the field
        """
        self._key = key

    @page.setter
    def page(self, page: int) -> None:
        """Sets the page value that the Field is associated with for validation.

        Args:
          page (int): the page number
        """
        self._page = page

    @field_property.setter
    def field_property(self, prop: str) -> None:
        """Sets the property name of the field.

        Args:
          property (str): the field's property name
        """
        self._property = prop

    def __str__(self):
        # TODO update this print everything
        return f"Field: key={self.key}, field_property={self.field_property}, depends={self.depends}"

    def validate(
        self, params: Dict[str, object], actions: Dict[str, "ValidatorAction"]
    ) -> "ValidatorResults":
        """Run the configured validations on this field.  Run all validations in the
        depends clause over each item in turn, returning when the first one fails.

        Args:
            params (Dict[str, object]): A dict of parameter class names to parameter values to pass into validation methods.
            actions A dict of validator names to ValidatorAction objects.

        Returns:
            A ValidatorResults object containing validation messages for this field.

        Throws:
            ValidatorException If an error occurs during validation.
        """
        # from ..validator_results_new import ValidatorResults
        from .validator_results_new import ValidatorResults
        if self.depends is None:
            return ValidatorResults()

        all_results: Final["ValidatorResults"] = ValidatorResults()
        number_of_fields_to_validate = (
            self.__get_indexed_property_size(bean) if self.is_indexed() else 1
        )

        for field_number in range(number_of_fields_to_validate):
            results: Final["ValidatorResults"] = ValidatorResults()

            with self.dependency_lock: # in java implementation there is a synchronized{dependencyList} here\
                for depend in self.dependency_list:
                    action: Final["ValidatorAction"] = actions[depend]
                    if action is None:
                        self.__handle_missing_action(depend)

                    good: Final[bool] = self.__validate_for_rule(
                        action, results, actions, params, field_number
                    )

                    if not good:
                        all_results.merge(results)
                        return all_results

                all_results.merge(results)
        return all_results

    def __validate_for_rule(
        self,
        va: "ValidatorAction",
        results: "ValidatorResults",
        actions: Dict[str, "ValidatorAction"],
        params: Dict[str, object],
        pos: int,
    ) -> bool:
        """Executes the given ValidatorAction and all ValidatorActions that it depends
        on.

        Returns:
            True if the validation succeeded.
        """
        result: Final["ValidatorResult"] = results.get_validator_result(self.key)
        if result is not None and result.contains_action(va.name):
            return result.is_valid(va.name)

        if not self.__run_dependent_validators(va, results, actions, params, pos):
            return False

        # from ..validator_results_new import ValidatorResults
        from .validator_results_new import ValidatorResults
        result_bool = va.execute_validation_method(self, params)
        results.add(self, va.name, result_bool)
        return result_bool
