"""
    Licensed to the Apache Software Foundation (ASF) under one or more
    contributor license agreements.  See the NOTICE file distributed with
    this work for additional information regarding copyright ownership.
    The ASF licenses this file to You under the Apache License, Version 2.0
    (the "License"); you may not use this file except in compliance with
    the License.  You may obtain a copy of the License at

        http://www.apache.org/licenses/LICENSE-2.0

    Unless required by applicable law or agreed to in writing, software
    distributed under the License is distributed on an "AS IS" BASIS,
    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
    See the License for the specific language governing permissions and
    limitations under the License.
"""

from src.apache_commons_validator_python.routines.inet_address_validator import InetAddressValidator

class TestInetAddressValidator:
    
    def test_broken_inet_address(self):
        """
        Test obviously broken IPs.
        """
        validator = InetAddressValidator()
        assert validator.is_valid("124.14.32.abc") is False, f"IP with characters should be invalid"
        assert validator.is_valid("124.14.32.01") is False, f"IP with leading zeroes should be invalid"
        assert validator.is_valid("23.64.12") is False, f"IP with three groups should be invalid"
        assert validator.is_valid("26.34.23.77.234") is False, f"IP with five groups should be invalid"
        assert validator.is_valid("") is False, f"IP empty string should be invalid"

    def test_inet_addresses_by_class(self):
        """
        Test valid and invalid IPs from each address class.
        """
        validator = InetAddressValidator()
        assert validator.is_valid("24.25.231.12") is True, f"Class A IP should be valid"
        assert validator.is_valid("2.41.32.324") is False, f"Illegal class A IP should be invalid"

        assert validator.is_valid("135.14.44.12") is True, f"Class B IP should be valid"
        assert validator.is_valid("154.123.441.123") is False, f"Illegal class B IP should be invalid"

        assert validator.is_valid("213.25.224.32") is True, f"Class C IP should be valid"
        assert validator.is_valid("201.543.23.11") is False, f"Illegal class C IP should be invalid"

        assert validator.is_valid("229.35.159.6") is True, f"Class D IP should be valid"
        assert validator.is_valid("231.54.11.987") is False, f"Illegal class D IP should be invalid"

        assert validator.is_valid("248.85.24.92") is True, f"Class E IP should be valid"
        assert validator.is_valid("250.21.323.48") is False, f"Illegal class E IP should be invalid"
    
    def test_inet_addresses_from_the_wild(self):
        """
        Test IPs that point to real, well-known hosts (without actually looking them up)
        """
        validator = InetAddressValidator()
        assert validator.is_valid("140.211.11.130") is True, f"www.apache.org IP should be valid"
        assert validator.is_valid("72.14.253.103") is True, f"www.l.google.com IP should be valid"
        assert validator.is_valid("199.232.41.5") is True, f"fsf.org IP should be valid"
        assert validator.is_valid("216.35.123.87") is True, f"appscs.ign.com IP should be valid"
    
    def test_ipv6(self):
        """
        Test IPv6 addresses.

        These tests were ported from "https://download.dartware.com/thirdparty/test-ipv6-regex.pl"
        """
        validator = InetAddressValidator()
        assert validator.is_valid_inet6_address('') is False, "IPV6 empty string should be invalid"
        assert validator.is_valid_inet6_address("::1") is True, "IPV6 ::1 should be valid"
        assert validator.is_valid_inet6_address("::") is True, "IPV6 :: should be valid"
        assert validator.is_valid_inet6_address("0:0:0:0:0:0:0:1") is True, "IPV6 0:0:0:0:0:0:0:1 should be valid"
        assert validator.is_valid_inet6_address("0:0:0:0:0:0:0:0") is True, "IPV6 0:0:0:0:0:0:0:0 should be valid"
        assert validator.is_valid_inet6_address("2001:DB8:0:0:8:800:200C:417A") is True, "IPV6 2001:DB8:0:0:8:800:200C:417A should be valid"
        assert validator.is_valid_inet6_address("FF01:0:0:0:0:0:0:101") is True, "IPV6 FF01:0:0:0:0:0:0:101 should be valid"
        assert validator.is_valid_inet6_address("2001:DB8::8:800:200C:417A") is True, "IPV6 2001:DB8::8:800:200C:417A should be valid"
        assert validator.is_valid_inet6_address("FF01::101") is True, "IPV6 FF01::101 should be valid"
        assert validator.is_valid_inet6_address("2001:DB8:0:0:8:800:200C:417A:221") is False, "IPV6 2001:DB8:0:0:8:800:200C:417A:221 should be invalid"
        assert validator.is_valid_inet6_address("FF01::101::2") is False, "IPV6 FF01::101::2 should be invalid"
        assert validator.is_valid_inet6_address("fe80::217:f2ff:fe07:ed62") is True, "IPV6 fe80::217:f2ff:fe07:ed62 should be valid"
        assert validator.is_valid_inet6_address("2001:0000:1234:0000:0000:C1C0:ABCD:0876") is True, "IPV6 2001:0000:1234:0000:0000:C1C0:ABCD:0876 should be valid"
        assert validator.is_valid_inet6_address("3ffe:0b00:0000:0000:0001:0000:0000:000a") is True, "IPV6 3ffe:0b00:0000:0000:0001:0000:0000:000a should be valid"
        assert validator.is_valid_inet6_address("FF02:0000:0000:0000:0000:0000:0000:0001") is True, "IPV6 FF02:0000:0000:0000:0000:0000:0000:0001 should be valid"
        assert validator.is_valid_inet6_address("0000:0000:0000:0000:0000:0000:0000:0001") is True, "IPV6 0000:0000:0000:0000:0000:0000:0000:0001 should be valid"
        assert validator.is_valid_inet6_address("0000:0000:0000:0000:0000:0000:0000:0000") is True, "IPV6 0000:0000:0000:0000:0000:0000:0000:0000 should be valid"
        assert validator.is_valid_inet6_address("02001:0000:1234:0000:0000:C1C0:ABCD:0876") is False, "IPV6 02001:0000:1234:0000:0000:C1C0:ABCD:0876 should be invalid"
        assert validator.is_valid_inet6_address("2001:0000:1234:0000:00001:C1C0:ABCD:0876") is False, "IPV6 2001:0000:1234:0000:00001:C1C0:ABCD:0876 should be invalid"
        assert validator.is_valid_inet6_address("2001:0000:1234:0000:0000:C1C0:ABCD:0876 0") is False, "IPV6 2001:0000:1234:0000:0000:C1C0:ABCD:0876 0 should be invalid"
        assert validator.is_valid_inet6_address("2001:0000:1234: 0000:0000:C1C0:ABCD:0876") is False, "IPV6 2001:0000:1234: 0000:0000:C1C0:ABCD:0876 should be invalid"
        assert validator.is_valid_inet6_address("3ffe:0b00:0000:0001:0000:0000:000a") is False, "IPV6 3ffe:0b00:0000:0001:0000:0000:000a should be invalid"
        assert validator.is_valid_inet6_address("FF02:0000:0000:0000:0000:0000:0000:0000:0001") is False, "IPV6 FF02:0000:0000:0000:0000:0000:0000:0000:0001 should be invalid"
        assert validator.is_valid_inet6_address("3ffe:b00::1::a") is False, "IPV6 3ffe:b00::1::a should be invalid"
        assert validator.is_valid_inet6_address("::1111:2222:3333:4444:5555:6666::") is False, "IPV6 ::1111:2222:3333:4444:5555:6666:: should be invalid"
        assert validator.is_valid_inet6_address("2::10") is True, "IPV6 2::10 should be valid"
        assert validator.is_valid_inet6_address("ff02::1") is True, "IPV6 ff02::1 should be valid"
        assert validator.is_valid_inet6_address("fe80::") is True, "IPV6 fe80:: should be valid"
        assert validator.is_valid_inet6_address("2002::") is True, "IPV6 2002:: should be valid"
        assert validator.is_valid_inet6_address("2001:db8::") is True, "IPV6 2001:db8:: should be valid"
        assert validator.is_valid_inet6_address("2001:0db8:1234::") is True, "IPV6 2001:0db8:1234:: should be valid"
        assert validator.is_valid_inet6_address("::ffff:0:0") is True, "IPV6 ::ffff:0:0 should be valid"
        assert validator.is_valid_inet6_address("1:2:3:4:5:6:7:8") is True, "IPV6 1:2:3:4:5:6:7:8 should be valid"
        assert validator.is_valid_inet6_address("1:2:3:4:5:6::8") is True, "IPV6 1:2:3:4:5:6::8 should be valid"
        assert validator.is_valid_inet6_address("1:2:3:4:5::8") is True, "IPV6 1:2:3:4:5::8 should be valid"
        assert validator.is_valid_inet6_address("1:2:3:4::8") is True, "IPV6 1:2:3:4::8 should be valid"
        assert validator.is_valid_inet6_address("1:2:3::8") is True, "IPV6 1:2:3::8 should be valid"
        assert validator.is_valid_inet6_address("1:2::8") is True, "IPV6 1:2::8 should be valid"
        assert validator.is_valid_inet6_address("1::8") is True, "IPV6 1::8 should be valid"
        assert validator.is_valid_inet6_address("1::2:3:4:5:6:7") is True, "IPV6 1::2:3:4:5:6:7 should be valid"
        assert validator.is_valid_inet6_address("1::2:3:4:5:6") is True, "IPV6 1::2:3:4:5:6 should be valid"
        assert validator.is_valid_inet6_address("1::2:3:4:5") is True, "IPV6 1::2:3:4:5 should be valid"
        assert validator.is_valid_inet6_address("1::2:3:4") is True, "IPV6 1::2:3:4 should be valid"
        assert validator.is_valid_inet6_address("1::2:3") is True, "IPV6 1::2:3 should be valid"
        assert validator.is_valid_inet6_address("::2:3:4:5:6:7:8") is True, "IPV6 ::2:3:4:5:6:7:8 should be valid"
        assert validator.is_valid_inet6_address("::2:3:4:5:6:7") is True, "IPV6 ::2:3:4:5:6:7 should be valid"
        assert validator.is_valid_inet6_address("::2:3:4:5:6") is True, "IPV6 ::2:3:4:5:6 should be valid"
        assert validator.is_valid_inet6_address("::2:3:4:5") is True, "IPV6 ::2:3:4:5 should be valid"
        assert validator.is_valid_inet6_address("::2:3:4") is True, "IPV6 ::2:3:4 should be valid"
        assert validator.is_valid_inet6_address("::2:3") is True, "IPV6 ::2:3 should be valid"
        assert validator.is_valid_inet6_address("::8") is True, "IPV6 ::8 should be valid"
        assert validator.is_valid_inet6_address("1:2:3:4:5:6::") is True, "IPV6 1:2:3:4:5:6:: should be valid"
        assert validator.is_valid_inet6_address("1:2:3:4:5::") is True, "IPV6 1:2:3:4:5:: should be valid"
        assert validator.is_valid_inet6_address("1:2:3:4::") is True, "IPV6 1:2:3:4:: should be valid"
        assert validator.is_valid_inet6_address("1:2:3::") is True, "IPV6 1:2:3:: should be valid"
        assert validator.is_valid_inet6_address("1:2::") is True, "IPV6 1:2:: should be valid"
        assert validator.is_valid_inet6_address("1::") is True, "IPV6 1:: should be valid"
        assert validator.is_valid_inet6_address("1:2:3:4:5::7:8") is True, "IPV6 1:2:3:4:5::7:8 should be valid"
        assert validator.is_valid_inet6_address("1:2:3::4:5::7:8") is False, "IPV6 1:2:3::4:5::7:8 should be invalid"
        assert validator.is_valid_inet6_address("12345::6:7:8") is False, "IPV6 12345::6:7:8 should be invalid"
        assert validator.is_valid_inet6_address("1:2:3:4::7:8") is True, "IPV6 1:2:3:4::7:8 should be valid"
        assert validator.is_valid_inet6_address("1:2:3::7:8") is True, "IPV6 1:2:3::7:8 should be valid"
        assert validator.is_valid_inet6_address("1:2::7:8") is True, "IPV6 1:2::7:8 should be valid"
        assert validator.is_valid_inet6_address("1::7:8") is True, "IPV6 1::7:8 should be valid"
        assert validator.is_valid_inet6_address("1:2:3:4:5:6:1.2.3.4") is True, "IPV6 1:2:3:4:5:6:1.2.3.4 should be valid"
        assert validator.is_valid_inet6_address("1:2:3:4:5::1.2.3.4") is True, "IPV6 1:2:3:4:5::1.2.3.4 should be valid"
        assert validator.is_valid_inet6_address("1:2:3:4::1.2.3.4") is True, "IPV6 1:2:3:4::1.2.3.4 should be valid"
        assert validator.is_valid_inet6_address("1:2:3::1.2.3.4") is True, "IPV6 1:2:3::1.2.3.4 should be valid"
        assert validator.is_valid_inet6_address("1:2::1.2.3.4") is True, "IPV6 1:2::1.2.3.4 should be valid"
        assert validator.is_valid_inet6_address("1::1.2.3.4") is True, "IPV6 1::1.2.3.4 should be valid"
        assert validator.is_valid_inet6_address("1:2:3:4::5:1.2.3.4") is True, "IPV6 1:2:3:4::5:1.2.3.4 should be valid"
        assert validator.is_valid_inet6_address("1:2:3::5:1.2.3.4") is True, "IPV6 1:2:3::5:1.2.3.4 should be valid"
        assert validator.is_valid_inet6_address("1:2::5:1.2.3.4") is True, "IPV6 1:2::5:1.2.3.4 should be valid"
        assert validator.is_valid_inet6_address("1::5:1.2.3.4") is True, "IPV6 1::5:1.2.3.4 should be valid"
        assert validator.is_valid_inet6_address("1::5:11.22.33.44") is True, "IPV6 1::5:11.22.33.44 should be valid"
        assert validator.is_valid_inet6_address("1::5:400.2.3.4") is False, "IPV6 1::5:400.2.3.4 should be invalid"
        assert validator.is_valid_inet6_address("1::5:260.2.3.4") is False, "IPV6 1::5:260.2.3.4 should be invalid"
        assert validator.is_valid_inet6_address("1::5:256.2.3.4") is False, "IPV6 1::5:256.2.3.4 should be invalid"
        assert validator.is_valid_inet6_address("1::5:1.256.3.4") is False, "IPV6 1::5:1.256.3.4 should be invalid"
        assert validator.is_valid_inet6_address("1::5:1.2.256.4") is False, "IPV6 1::5:1.2.256.4 should be invalid"
        assert validator.is_valid_inet6_address("1::5:1.2.3.256") is False, "IPV6 1::5:1.2.3.256 should be invalid"
        assert validator.is_valid_inet6_address("1::5:300.2.3.4") is False, "IPV6 1::5:300.2.3.4 should be invalid"
        assert validator.is_valid_inet6_address("1::5:1.300.3.4") is False, "IPV6 1::5:1.300.3.4 should be invalid"
        assert validator.is_valid_inet6_address("1::5:1.2.300.4") is False, "IPV6 1::5:1.2.300.4 should be invalid"
        assert validator.is_valid_inet6_address("1::5:1.2.3.300") is False, "IPV6 1::5:1.2.3.300 should be invalid"
        assert validator.is_valid_inet6_address("1::5:900.2.3.4") is False, "IPV6 1::5:900.2.3.4 should be invalid"
        assert validator.is_valid_inet6_address("1::5:1.900.3.4") is False, "IPV6 1::5:1.900.3.4 should be invalid"
        assert validator.is_valid_inet6_address("1::5:1.2.900.4") is False, "IPV6 1::5:1.2.900.4 should be invalid"
        assert validator.is_valid_inet6_address("1::5:1.2.3.900") is False, "IPV6 1::5:1.2.3.900 should be invalid"
        assert validator.is_valid_inet6_address("1::5:300.300.300.300") is False, "IPV6 1::5:300.300.300.300 should be invalid"
        assert validator.is_valid_inet6_address("1::5:3000.30.30.30") is False, "IPV6 1::5:3000.30.30.30 should be invalid"
        assert validator.is_valid_inet6_address("1::400.2.3.4") is False, "IPV6 1::400.2.3.4 should be invalid"
        assert validator.is_valid_inet6_address("1::260.2.3.4") is False, "IPV6 1::260.2.3.4 should be invalid"
        assert validator.is_valid_inet6_address("1::256.2.3.4") is False, "IPV6 1::256.2.3.4 should be invalid"
        assert validator.is_valid_inet6_address("1::1.256.3.4") is False, "IPV6 1::1.256.3.4 should be invalid"
        assert validator.is_valid_inet6_address("1::1.2.256.4") is False, "IPV6 1::1.2.256.4 should be invalid"
        assert validator.is_valid_inet6_address("1::1.2.3.256") is False, "IPV6 1::1.2.3.256 should be invalid"
        assert validator.is_valid_inet6_address("1::300.2.3.4") is False, "IPV6 1::300.2.3.4 should be invalid"
        assert validator.is_valid_inet6_address("1::1.300.3.4") is False, "IPV6 1::1.300.3.4 should be invalid"
        assert validator.is_valid_inet6_address("1::1.2.300.4") is False, "IPV6 1::1.2.300.4 should be invalid"
        assert validator.is_valid_inet6_address("1::1.2.3.300") is False, "IPV6 1::1.2.3.300 should be invalid"
        assert validator.is_valid_inet6_address("1::900.2.3.4") is False, "IPV6 1::900.2.3.4 should be invalid"
        assert validator.is_valid_inet6_address("1::1.900.3.4") is False, "IPV6 1::1.900.3.4 should be invalid"
        assert validator.is_valid_inet6_address("1::1.2.900.4") is False, "IPV6 1::1.2.900.4 should be invalid"
        assert validator.is_valid_inet6_address("1::1.2.3.900") is False, "IPV6 1::1.2.3.900 should be invalid"
        assert validator.is_valid_inet6_address("1::300.300.300.300") is False, "IPV6 1::300.300.300.300 should be invalid"
        assert validator.is_valid_inet6_address("1::3000.30.30.30") is False, "IPV6 1::3000.30.30.30 should be invalid"
        assert validator.is_valid_inet6_address("::400.2.3.4") is False, "IPV6 ::400.2.3.4 should be invalid"
        assert validator.is_valid_inet6_address("::260.2.3.4") is False, "IPV6 ::260.2.3.4 should be invalid"
        assert validator.is_valid_inet6_address("::256.2.3.4") is False, "IPV6 ::256.2.3.4 should be invalid"
        assert validator.is_valid_inet6_address("::1.256.3.4") is False, "IPV6 ::1.256.3.4 should be invalid"
        assert validator.is_valid_inet6_address("::1.2.256.4") is False, "IPV6 ::1.2.256.4 should be invalid"
        assert validator.is_valid_inet6_address("::1.2.3.256") is False, "IPV6 ::1.2.3..256 should be invalid"
        assert validator.is_valid_inet6_address("::300.2.3.4") is False, "IPV6 ::300.2.3.4 should be invalid"
        assert validator.is_valid_inet6_address("::1.300.3.4") is False, "IPV6 ::1.300.3.4 should be invalid"
        assert validator.is_valid_inet6_address("::1.2.300.4") is False, "IPV6 ::1.2.300.4 should be invalid"
        assert validator.is_valid_inet6_address("::1.2.3.300") is False, "IPV6 ::1.2.3.300 should be invalid"
        assert validator.is_valid_inet6_address("::900.2.3.4") is False, "IPV6 ::900.2.3.4 should be invalid"
        assert validator.is_valid_inet6_address("::1.900.3.4") is False, "IPV6 ::1.900.3.4 should be invalid"
        assert validator.is_valid_inet6_address("::1.2.900.4") is False, "IPV6 ::1.2.900.4 should be invalid"
        assert validator.is_valid_inet6_address("::1.2.3.900") is False, "IPV6 ::1.2.3.900 should be invalid"
        assert validator.is_valid_inet6_address("::300.300.300.300") is False, "IPV6 ::300.300.300.300 should be invalid"
        assert validator.is_valid_inet6_address("::3000.30.30.30") is False, "IPV6 ::3000.30.30.30 should be invalid"
        assert validator.is_valid_inet6_address("fe80::217:f2ff:254.7.237.98") is True, "IPV6 fe80::217:f2ff:254.7.237.98 should be valid"
        assert validator.is_valid_inet6_address("::ffff:192.168.1.26") is True, "IPV6 ::ffff:192.168.1.26 should be valid"
        assert validator.is_valid_inet6_address("2001:1:1:1:1:1:255Z255X255Y255") is False, "IPV6 2001:1:1:1:1:1:255Z255X255Y255 should be invalid"
        assert validator.is_valid_inet6_address("::ffff:192x168.1.26") is False, "IPV6 ::ffff:192x168.1.26 should be invalid"  # ditto
        assert validator.is_valid_inet6_address("::ffff:192.168.1.1") is True, "IPV6 ::ffff:192.168.1.1 should be valid"
        assert validator.is_valid_inet6_address("0:0:0:0:0:0:13.1.68.3") is True, "IPV6 0:0:0:0:0:0:13.1.68.3 should be valid"  # IPv4-compatible IPv6 address, full, deprecated
        assert validator.is_valid_inet6_address("0:0:0:0:0:FFFF:129.144.52.38") is True, "IPV6 0:0:0:0:0:FFFF:129.144.52.38 should be valid"  # IPv4-mapped IPv6 address, full
        assert validator.is_valid_inet6_address("::13.1.68.3") is True, "IPV6 ::13.1.68.3 should be valid"  # IPv4-compatible IPv6 address, compressed, deprecated
        assert validator.is_valid_inet6_address("::FFFF:129.144.52.38") is True, "IPV6 ::FFFF:129.144.52.38 should be valid"  # IPv4-mapped IPv6 address, compressed
        assert validator.is_valid_inet6_address("fe80:0:0:0:204:61ff:254.157.241.86") is True, "IPV6 fe80:0:0:0:204:61ff:254.157.241.86 should be valid"
        assert validator.is_valid_inet6_address("fe80::204:61ff:254.157.241.86") is True, "IPV6 fe80::204:61ff:254.157.241.86 should be valid"
        assert validator.is_valid_inet6_address("::ffff:12.34.56.78") is True, "IPV6 ::ffff:12.34.56.78 should be valid"
        assert validator.is_valid_inet6_address("::ffff:2.3.4") is False, "IPV6 ::ffff:2.3.4 should be invalid"
        assert validator.is_valid_inet6_address("::ffff:257.1.2.3") is False, "IPV6 ::ffff:257.1.2.3 should be invalid"
        assert validator.is_valid_inet6_address("1.2.3.4") is False, "IPV6 1.2.3.4 should be invalid"
        assert validator.is_valid_inet6_address("1.2.3.4:1111:2222:3333:4444::5555") is False, "IPV6 1.2.3.4:1111:2222:3333:4444::5555 should be invalid"
        assert validator.is_valid_inet6_address("1.2.3.4:1111:2222:3333::5555") is False, "IPV6 1.2.3.4:1111:2222:3333::5555 should be invalid"
        assert validator.is_valid_inet6_address("1.2.3.4:1111:2222::5555") is False, "IPV6 1.2.3.4:1111:2222::5555 should be invalid"
        assert validator.is_valid_inet6_address("1.2.3.4:1111::5555") is False, "IPV6 1.2.3.4:1111::5555 should be invalid"
        assert validator.is_valid_inet6_address("1.2.3.4::5555") is False, "IPV6 1.2.3.4::5555 should be invalid"
        assert validator.is_valid_inet6_address("1.2.3.4::") is False, "IPV6 1.2.3.4:: should be invalid"
        # Testing IPv4 addresses represented as dotted-quads
        # Leading zeroes in IPv4 addresses not allowed: some systems treat the leading "0" in ".086" as the start of an octal number
        # Update: The BNF in RFC-3986 explicitly defines the dec-octet (for IPv4 addresses) not to have a leading zero
        assert validator.is_valid_inet6_address("fe80:0000:0000:0000:0204:61ff:254.157.241.086") is False, "IPV6 fe80:0000:0000:0000:0204:61ff:254.157.241.086 should be invalid"
        assert validator.is_valid_inet6_address("::ffff:192.0.2.128") is True, "IPV6 ::ffff:192.0.2.128 should be valid"
        assert validator.is_valid_inet6_address("XXXX:XXXX:XXXX:XXXX:XXXX:XXXX:1.2.3.4") is False, "IPV6 XXXX:XXXX:XXXX:XXXX:XXXX:XXXX:1.2.3.4 should be invalid"
        assert validator.is_valid_inet6_address("1111:2222:3333:4444:5555:6666:00.00.00.00") is False, "IPV6 1111:2222:3333:4444:5555:6666:00.00.00.00 should be invalid"
        assert validator.is_valid_inet6_address("1111:2222:3333:4444:5555:6666:000.000.000.000") is False, "IPV6 1111:2222:3333:4444:5555:6666:000.000.000.000 should be invalid"
        assert validator.is_valid_inet6_address("1111:2222:3333:4444:5555:6666:256.256.256.256") is False, "IPV6 1111:2222:3333:4444:5555:6666:256.256.256.256 should be invalid"
        assert validator.is_valid_inet6_address("fe80:0000:0000:0000:0204:61ff:fe9d:f156") is True, "IPV6 fe80:0000:0000:0000:0204:61ff:fe9d:f156 should be valid"
        assert validator.is_valid_inet6_address("fe80:0:0:0:204:61ff:fe9d:f156") is True, "IPV6 fe80:0:0:0:204:61ff:fe9d:f156 should be valid"
        assert validator.is_valid_inet6_address("fe80::204:61ff:fe9d:f156") is True, "IPV6 fe80::204:61ff:fe9d:f156 should be valid"
        assert validator.is_valid_inet6_address(":") is False, "IPV6 : should be invalid"
        assert validator.is_valid_inet6_address("::ffff:c000:280") is True, "IPV6 ::ffff:c000:280 should be valid"
        assert validator.is_valid_inet6_address("1111:2222:3333:4444::5555:") is False, "IPV6 1111:2222:3333:4444::5555: should be invalid"
        assert validator.is_valid_inet6_address("1111:2222:3333::5555:") is False, "IPV6 1111:2222:3333::5555: should be invalid"
        assert validator.is_valid_inet6_address("1111:2222::5555:") is False, "IPV6 1111:2222::5555: should be invalid"
        assert validator.is_valid_inet6_address("1111::5555:") is False, "IPV6 1111::5555: should be invalid"
        assert validator.is_valid_inet6_address("::5555:") is False, "IPV6 ::5555: should be invalid"
        assert validator.is_valid_inet6_address(":::") is False, "IPV6 ::: should be invalid"
        assert validator.is_valid_inet6_address("1111:") is False, "IPV6 1111: should be invalid"
        assert validator.is_valid_inet6_address(":1111:2222:3333:4444::5555") is False, "IPV6 :1111:2222:3333:4444::5555 should be invalid"
        assert validator.is_valid_inet6_address(":1111:2222:3333::5555") is False, "IPV6 :1111:2222:3333::5555 should be invalid"
        assert validator.is_valid_inet6_address(":1111:2222::5555") is False, "IPV6 :1111:2222::5555 should be invalid"
        assert validator.is_valid_inet6_address(":1111::5555") is False, "IPV6 :1111::5555 should be invalid"
        assert validator.is_valid_inet6_address(":::5555") is False, "IPV6 :::5555 should be invalid"
        assert validator.is_valid_inet6_address("2001:0db8:85a3:0000:0000:8a2e:0370:7334") is True, "IPV6 2001:0db8:85a3:0000:0000:8a2e:0370:7334 should be valid"
        assert validator.is_valid_inet6_address("2001:db8:85a3:0:0:8a2e:370:7334") is True, "IPV6 2001:db8:85a3:0:0:8a2e:370:7334 should be valid"
        assert validator.is_valid_inet6_address("2001:db8:85a3::8a2e:370:7334") is True, "IPV6 2001:db8:85a3::8a2e:370:7334 should be valid"
        assert validator.is_valid_inet6_address("2001:0db8:0000:0000:0000:0000:1428:57ab") is True, "IPV6 2001:0db8:0000:0000:0000:0000:1428:57ab should be valid"
        assert validator.is_valid_inet6_address("2001:0db8:0000:0000:0000::1428:57ab") is True, "IPV6 2001:0db8:0000:0000:0000::1428:57ab should be valid"
        assert validator.is_valid_inet6_address("2001:0db8:0:0:0:0:1428:57ab") is True, "IPV6 2001:0db8:0:0:0:0:1428:57ab should be valid"
        assert validator.is_valid_inet6_address("2001:0db8:0:0::1428:57ab") is True, "IPV6 2001:0db8:0:0::1428:57ab should be valid"
        assert validator.is_valid_inet6_address("2001:0db8::1428:57ab") is True, "IPV6 2001:0db8::1428:57ab should be valid"
        assert validator.is_valid_inet6_address("2001:db8::1428:57ab") is True, "IPV6 2001:db8::1428:57ab should be valid"
        assert validator.is_valid_inet6_address("::ffff:0c22:384e") is True, "IPV6 ::ffff:0c22:384e should be valid"
        assert validator.is_valid_inet6_address("2001:0db8:1234:0000:0000:0000:0000:0000") is True, "IPV6 2001:0db8:1234:0000:0000:0000:0000:0000 should be valid"
        assert validator.is_valid_inet6_address("2001:0db8:1234:ffff:ffff:ffff:ffff:ffff") is True, "IPV6 2001:0db8:1234:ffff:ffff:ffff:ffff:ffff should be valid"
        assert validator.is_valid_inet6_address("2001:db8:a::123") is True, "IPV6 2001:db8:a::123 should be valid"
        assert validator.is_valid_inet6_address("123") is False, "IPV6 123 should be invalid"
        assert validator.is_valid_inet6_address("ldkfj") is False, "IPV6 ldkfj should be invalid"
        assert validator.is_valid_inet6_address("2001::FFD3::57ab") is False, "IPV6 2001::FFD3::57ab should be invalid"
        assert validator.is_valid_inet6_address("2001:db8:85a3::8a2e:37023:7334") is False, "IPV6 2001:db8:85a3::8a2e:37023:7334 should be invalid"
        assert validator.is_valid_inet6_address("2001:db8:85a3::8a2e:370k:7334") is False, "IPV6 2001:db8:85a3::8a2e:370k:7334 should be invalid"
        assert validator.is_valid_inet6_address("1:2:3:4:5:6:7:8:9") is False, "IPV6 1:2:3:4:5:6:7:8:9 should be invalid"
        assert validator.is_valid_inet6_address("1::2::3") is False, "IPV6 1::2::3 should be invalid"
        assert validator.is_valid_inet6_address("1:::3:4:5") is False, "IPV6 1:::3:4:5 should be invalid"
        assert validator.is_valid_inet6_address("1:2:3::4:5:6:7:8:9") is False, "IPV6 1:2:3::4:5:6:7:8:9 should be invalid"
        assert validator.is_valid_inet6_address("1111:2222:3333:4444:5555:6666:7777:8888") is True, "IPV6 1111:2222:3333:4444:5555:6666:7777:8888 should be valid"
        assert validator.is_valid_inet6_address("1111:2222:3333:4444:5555:6666:7777::") is True, "IPV6 1111:2222:3333:4444:5555:6666:7777:: should be valid"
        assert validator.is_valid_inet6_address("1111:2222:3333:4444:5555:6666::") is True, "IPV6 1111:2222:3333:4444:5555:6666:: should be valid"
        assert validator.is_valid_inet6_address("1111:2222:3333:4444:5555::") is True, "IPV6 1111:2222:3333:4444:5555:: should be valid"
        assert validator.is_valid_inet6_address("1111:2222:3333:4444::") is True, "IPV6 1111:2222:3333:4444:: should be valid"
        assert validator.is_valid_inet6_address("1111:2222:3333::") is True, "IPV6 1111:2222:3333:: should be valid"
        assert validator.is_valid_inet6_address("1111:2222::") is True, "IPV6 1111:2222:: should be valid"
        assert validator.is_valid_inet6_address("1111::") is True, "IPV6 1111:: should be valid"
        assert validator.is_valid_inet6_address("1111:2222:3333:4444:5555:6666::8888") is True, "IPV6 1111:2222:3333:4444:5555:6666::8888 should be valid"
        assert validator.is_valid_inet6_address("1111:2222:3333:4444:5555::8888") is True, "IPV6 1111:2222:3333:4444:5555::8888 should be valid"
        assert validator.is_valid_inet6_address("1111:2222:3333:4444::8888") is True, "IPV6 1111:2222:3333:4444::8888 should be valid"
        assert validator.is_valid_inet6_address("1111:2222:3333::8888") is True, "IPV6 1111:2222:3333::8888 should be valid"
        assert validator.is_valid_inet6_address("1111:2222::8888") is True, "IPV6 1111:2222::8888 should be valid"
        assert validator.is_valid_inet6_address("1111::8888") is True, "IPV6 1111::8888 should be valid"
        assert validator.is_valid_inet6_address("::8888") is True, "IPV6 ::8888 should be valid"
        assert validator.is_valid_inet6_address("1111:2222:3333:4444:5555::7777:8888") is True, "IPV6 1111:2222:3333:4444:5555::7777:8888 should be valid"
        assert validator.is_valid_inet6_address("1111:2222:3333:4444::7777:8888") is True, "IPV6 1111:2222:3333:4444::7777:8888 should be valid"
        assert validator.is_valid_inet6_address("1111:2222:3333::7777:8888") is True, "IPV6 1111:2222:3333::7777:8888 should be valid"
        assert validator.is_valid_inet6_address("1111:2222::7777:8888") is True, "IPV6 1111:2222::7777:8888 should be valid"
        assert validator.is_valid_inet6_address("1111::7777:8888") is True, "IPV6 1111::7777:8888 should be valid"
        assert validator.is_valid_inet6_address("::7777:8888") is True, "IPV6 ::7777:8888 should be valid"
        assert validator.is_valid_inet6_address("1111:2222:3333:4444::6666:7777:8888") is True, "IPV6 1111:2222:3333:4444::6666:7777:8888 should be valid"
        assert validator.is_valid_inet6_address("1111:2222:3333::6666:7777:8888") is True, "IPV6 1111:2222:3333::6666:7777:8888 should be valid"
        assert validator.is_valid_inet6_address("1111:2222::6666:7777:8888") is True, "IPV6 1111:2222::6666:7777:8888 should be valid"
        assert validator.is_valid_inet6_address("1111::6666:7777:8888") is True, "IPV6 1111::6666:7777:8888 should be valid"
        assert validator.is_valid_inet6_address("::6666:7777:8888") is True, "IPV6 ::6666:7777:8888 should be valid"
        assert validator.is_valid_inet6_address("1111:2222:3333::5555:6666:7777:8888") is True, "IPV6 1111:2222:3333::5555:6666:7777:8888 should be valid"
        assert validator.is_valid_inet6_address("1111:2222::5555:6666:7777:8888") is True, "IPV6 1111:2222::5555:6666:7777:8888 should be valid"
        assert validator.is_valid_inet6_address("1111::5555:6666:7777:8888") is True, "IPV6 1111::5555:6666:7777:8888 should be valid"
        assert validator.is_valid_inet6_address("::5555:6666:7777:8888") is True, "IPV6 ::5555:6666:7777:8888 should be valid"
        assert validator.is_valid_inet6_address("1111:2222::4444:5555:6666:7777:8888") is True, "IPV6 1111:2222::4444:5555:6666:7777:8888 should be valid"
        assert validator.is_valid_inet6_address("1111::4444:5555:6666:7777:8888") is True, "IPV6 1111::4444:5555:6666:7777:8888 should be valid"
        assert validator.is_valid_inet6_address("::4444:5555:6666:7777:8888") is True, "IPV6 ::4444:5555:6666:7777:8888 should be valid"
        assert validator.is_valid_inet6_address("1111::3333:4444:5555:6666:7777:8888") is True, "IPV6 1111::3333:4444:5555:6666:7777:8888 should be valid"
        assert validator.is_valid_inet6_address("::3333:4444:5555:6666:7777:8888") is True, "IPV6 ::3333:4444:5555:6666:7777:8888 should be valid"
        assert validator.is_valid_inet6_address("::2222:3333:4444:5555:6666:7777:8888") is True, "IPV6 ::2222:3333:4444:5555:6666:7777:8888 should be valid"
        assert validator.is_valid_inet6_address("1111:2222:3333:4444:5555:6666:123.123.123.123") is True, "IPV6 1111:2222:3333:4444:5555:6666:123.123.123.123 should be valid"
        assert validator.is_valid_inet6_address("1111:2222:3333:4444:5555::123.123.123.123") is True, "IPV6 1111:2222:3333:4444:5555::123.123.123.123 should be valid"
        assert validator.is_valid_inet6_address("1111:2222:3333:4444::123.123.123.123") is True, "IPV6 1111:2222:3333:4444::123.123.123.123 should be valid"
        assert validator.is_valid_inet6_address("1111:2222:3333::123.123.123.123") is True, "IPV6 1111:2222:3333::123.123.123.123 should be valid"
        assert validator.is_valid_inet6_address("1111:2222::123.123.123.123") is True, "IPV6 1111:2222::123.123.123.123 should be valid"
        assert validator.is_valid_inet6_address("1111::123.123.123.123") is True, "IPV6 1111::123.123.123.123 should be valid"
        assert validator.is_valid_inet6_address("::123.123.123.123") is True, "IPV6 ::123.123.123.123 should be valid"
        assert validator.is_valid_inet6_address("1111:2222:3333:4444::6666:123.123.123.123") is True, "IPV6 1111:2222:3333:4444::6666:123.123.123.123 should be valid"
        assert validator.is_valid_inet6_address("1111:2222:3333::6666:123.123.123.123") is True, "IPV6 1111:2222:3333::6666:123.123.123.123 should be valid"
        assert validator.is_valid_inet6_address("1111:2222::6666:123.123.123.123") is True, "IPV6 1111:2222::6666:123.123.123.123 should be valid"
        assert validator.is_valid_inet6_address("1111::6666:123.123.123.123") is True, "IPV6 1111::6666:123.123.123.123 should be valid"
        assert validator.is_valid_inet6_address("::6666:123.123.123.123") is True, "IPV6 ::6666:123.123.123.123 should be valid"
        assert validator.is_valid_inet6_address("1111:2222:3333::5555:6666:123.123.123.123") is True, "IPV6 1111:2222:3333::5555:6666:123.123.123.123 should be valid"
        assert validator.is_valid_inet6_address("1111:2222::5555:6666:123.123.123.123") is True, "IPV6 1111:2222::5555:6666:123.123.123.123 should be valid"
        assert validator.is_valid_inet6_address("1111::5555:6666:123.123.123.123") is True, "IPV6 1111::5555:6666:123.123.123.123 should be valid"
        assert validator.is_valid_inet6_address("::5555:6666:123.123.123.123") is True, "IPV6 ::5555:6666:123.123.123.123 should be valid"
        assert validator.is_valid_inet6_address("1111:2222::4444:5555:6666:123.123.123.123") is True, "IPV6 1111:2222::4444:5555:6666:123.123.123.123 should be valid"
        assert validator.is_valid_inet6_address("1111::4444:5555:6666:123.123.123.123") is True, "IPV6 1111::4444:5555:6666:123.123.123.123 should be valid"
        assert validator.is_valid_inet6_address("::4444:5555:6666:123.123.123.123") is True, "IPV6 ::4444:5555:6666:123.123.123.123 should be valid"
        assert validator.is_valid_inet6_address("1111::3333:4444:5555:6666:123.123.123.123") is True, "IPV6 1111::3333:4444:5555:6666:123.123.123.123 should be valid"
        assert validator.is_valid_inet6_address("::2222:3333:4444:5555:6666:123.123.123.123") is True, "IPV6 ::2222:3333:4444:5555:6666:123.123.123.123 should be valid"
        # Trying combinations of "0" and "::"
        # These are all syntactically correct, but are bad form
        # because "0" adjacent to "::" should be combined into "::"
        assert validator.is_valid_inet6_address("::0:0:0:0:0:0:0") is True, "IPV6 ::0:0:0:0:0:0:0 should be valid"
        assert validator.is_valid_inet6_address("::0:0:0:0:0:0") is True, "IPV6 ::0:0:0:0:0:0 should be valid"
        assert validator.is_valid_inet6_address("::0:0:0:0:0") is True, "IPV6 ::0:0:0:0:0 should be valid"
        assert validator.is_valid_inet6_address("::0:0:0:0") is True, "IPV6 ::0:0:0:0 should be valid"
        assert validator.is_valid_inet6_address("::0:0:0") is True, "IPV6 ::0:0:0 should be valid"
        assert validator.is_valid_inet6_address("::0:0") is True, "IPV6 ::0:0 should be valid"
        assert validator.is_valid_inet6_address("::0") is True, "IPV6 ::0 should be valid"
        assert validator.is_valid_inet6_address("0:0:0:0:0:0:0::") is True, "IPV6 0:0:0:0:0:0:0:: should be valid"
        assert validator.is_valid_inet6_address("0:0:0:0:0:0::") is True, "IPV6 0:0:0:0:0:0:: should be valid"
        assert validator.is_valid_inet6_address("0:0:0:0:0::") is True, "IPV6 0:0:0:0:0:: should be valid"
        assert validator.is_valid_inet6_address("0:0:0:0::") is True, "IPV6 0:0:0:0:: should be valid"
        assert validator.is_valid_inet6_address("0:0:0::") is True, "IPV6 0:0:0:: should be valid"
        assert validator.is_valid_inet6_address("0:0::") is True, "IPV6 0:0:: should be valid"
        assert validator.is_valid_inet6_address("0::") is True, "IPV6 0:: should be valid"
        # Invalid data
        assert validator.is_valid_inet6_address("XXXX:XXXX:XXXX:XXXX:XXXX:XXXX:XXXX:XXXX") is False, "IPV6 XXXX:XXXX:XXXX:XXXX:XXXX:XXXX:XXXX:XXXX should be invalid"
        # Too many components
        assert validator.is_valid_inet6_address("1111:2222:3333:4444:5555:6666:7777:8888:9999") is False, "IPV6 1111:2222:3333:4444:5555:6666:7777:8888:9999 should be invalid"
        assert validator.is_valid_inet6_address("1111:2222:3333:4444:5555:6666:7777:8888::") is False, "IPV6 1111:2222:3333:4444:5555:6666:7777:8888:: should be invalid"
        assert validator.is_valid_inet6_address("::2222:3333:4444:5555:6666:7777:8888:9999") is False, "IPV6 ::2222:3333:4444:5555:6666:7777:8888:9999 should be invalid"
        # Too few components
        assert validator.is_valid_inet6_address("1111:2222:3333:4444:5555:6666:7777") is False, "IPV6 1111:2222:3333:4444:5555:6666:7777 should be invalid"
        assert validator.is_valid_inet6_address("1111:2222:3333:4444:5555:6666") is False, "IPV6 1111:2222:3333:4444:5555:6666 should be invalid"
        assert validator.is_valid_inet6_address("1111:2222:3333:4444:5555") is False, "IPV6 1111:2222:3333:4444:5555 should be invalid"
        assert validator.is_valid_inet6_address("1111:2222:3333:4444") is False, "IPV6 1111:2222:3333:4444 should be invalid"
        assert validator.is_valid_inet6_address("1111:2222:3333") is False, "IPV6 1111:2222:3333 should be invalid"
        assert validator.is_valid_inet6_address("1111:2222") is False, "IPV6 1111:2222 should be invalid"
        assert validator.is_valid_inet6_address("1111") is False, "IPV6 1111 should be invalid"
        # Missing :
        assert validator.is_valid_inet6_address("11112222:3333:4444:5555:6666:7777:8888") is False, "IPV6 11112222:3333:4444:5555:6666:7777:8888 should be invalid"
        assert validator.is_valid_inet6_address("1111:22223333:4444:5555:6666:7777:8888") is False, "IPV6 1111:22223333:4444:5555:6666:7777:8888 should be invalid"
        assert validator.is_valid_inet6_address("1111:2222:33334444:5555:6666:7777:8888") is False, "IPV6 1111:2222:33334444:5555:6666:7777:8888 should be invalid"
        assert validator.is_valid_inet6_address("1111:2222:3333:44445555:6666:7777:8888") is False, "IPV6 1111:2222:3333:44445555:6666:7777:8888 should be invalid"
        assert validator.is_valid_inet6_address("1111:2222:3333:4444:55556666:7777:8888") is False, "IPV6 1111:2222:3333:4444:55556666:7777:8888 should be invalid"
        assert validator.is_valid_inet6_address("1111:2222:3333:4444:5555:66667777:8888") is False, "IPV6 1111:2222:3333:4444:5555:66667777:8888 should be invalid"
        assert validator.is_valid_inet6_address("1111:2222:3333:4444:5555:6666:77778888") is False, "IPV6 1111:2222:3333:4444:5555:6666:77778888 should be invalid"
        # Missing : intended for ::
        assert validator.is_valid_inet6_address("1111:2222:3333:4444:5555:6666:7777:8888:") is False, "IPV6 1111:2222:3333:4444:5555:6666:7777:8888: should be invalid"
        assert validator.is_valid_inet6_address("1111:2222:3333:4444:5555:6666:7777:") is False, "IPV6 1111:2222:3333:4444:5555:6666:7777: should be invalid"
        assert validator.is_valid_inet6_address("1111:2222:3333:4444:5555:6666:") is False, "IPV6 1111:2222:3333:4444:5555:6666: should be invalid"
        assert validator.is_valid_inet6_address("1111:2222:3333:4444:5555:") is False, "IPV6 1111:2222:3333:4444:5555: should be invalid"
        assert validator.is_valid_inet6_address("1111:2222:3333:4444:") is False, "IPV6 1111:2222:3333:4444: should be invalid"
        assert validator.is_valid_inet6_address("1111:2222:3333:") is False, "IPV6 1111:2222:3333: should be invalid"
        assert validator.is_valid_inet6_address("1111:2222:") is False, "IPV6 1111:2222: should be invalid"
        assert validator.is_valid_inet6_address(":8888") is False, "IPV6 :8888 should be invalid"
        assert validator.is_valid_inet6_address(":7777:8888") is False, "IPV6 :7777:8888 should be invalid"
        assert validator.is_valid_inet6_address(":6666:7777:8888") is False, "IPV6 :6666:7777:8888 should be invalid"
        assert validator.is_valid_inet6_address(":5555:6666:7777:8888") is False, "IPV6 :5555:6666:7777:8888 should be invalid"
        assert validator.is_valid_inet6_address(":4444:5555:6666:7777:8888") is False, "IPV6 :4444:5555:6666:7777:8888 should be invalid"
        assert validator.is_valid_inet6_address(":3333:4444:5555:6666:7777:8888") is False, "IPV6 :3333:4444:5555:6666:7777:8888 should be invalid"
        assert validator.is_valid_inet6_address(":2222:3333:4444:5555:6666:7777:8888") is False, "IPV6 :2222:3333:4444:5555:6666:7777:8888 should be invalid"
        assert validator.is_valid_inet6_address(":1111:2222:3333:4444:5555:6666:7777:8888") is False, "IPV6 :1111:2222:3333:4444:5555:6666:7777:8888 should be invalid"
        # :::
        assert validator.is_valid_inet6_address(":::2222:3333:4444:5555:6666:7777:8888") is False, "IPV6 :::2222:3333:4444:5555:6666:7777:8888 should be invalid"
        assert validator.is_valid_inet6_address("1111:::3333:4444:5555:6666:7777:8888") is False, "IPV6 1111:::3333:4444:5555:6666:7777:8888 should be invalid"
        assert validator.is_valid_inet6_address("1111:2222:::4444:5555:6666:7777:8888") is False, "IPV6 1111:2222:::4444:5555:6666:7777:8888 should be invalid"
        assert validator.is_valid_inet6_address("1111:2222:3333:::5555:6666:7777:8888") is False, "IPV6 1111:2222:3333:::5555:6666:7777:8888 should be invalid"
        assert validator.is_valid_inet6_address("1111:2222:3333:4444:::6666:7777:8888") is False, "IPV6 1111:2222:3333:4444:::6666:7777:8888 should be invalid"
        assert validator.is_valid_inet6_address("1111:2222:3333:4444:5555:::7777:8888") is False, "IPV6 1111:2222:3333:4444:5555:::7777:8888 should be invalid"
        assert validator.is_valid_inet6_address("1111:2222:3333:4444:5555:6666:::8888") is False, "IPV6 1111:2222:3333:4444:5555:6666:::8888 should be invalid"
        assert validator.is_valid_inet6_address("1111:2222:3333:4444:5555:6666:7777:::") is False, "IPV6 1111:2222:3333:4444:5555:6666:7777::: should be invalid"
        # Double ::
        assert validator.is_valid_inet6_address("::2222::4444:5555:6666:7777:8888") is False, "IPV6 ::2222::4444:5555:6666:7777:8888 should be invalid"
        assert validator.is_valid_inet6_address("::2222:3333::5555:6666:7777:8888") is False, "IPV6 ::2222:3333::5555:6666:7777:8888 should be invalid"
        assert validator.is_valid_inet6_address("::2222:3333:4444::6666:7777:8888") is False, "IPV6 ::2222:3333:4444::6666:7777:8888 should be invalid"
        assert validator.is_valid_inet6_address("::2222:3333:4444:5555::7777:8888") is False, "IPV6 ::2222:3333:4444:5555::7777:8888 should be invalid"
        assert validator.is_valid_inet6_address("::2222:3333:4444:5555:7777::8888") is False, "IPV6 ::2222:3333:4444:5555:7777::8888 should be invalid"
        assert validator.is_valid_inet6_address("::2222:3333:4444:5555:7777:8888::") is False, "IPV6 ::2222:3333:4444:5555:7777:8888:: should be invalid"
        assert validator.is_valid_inet6_address("1111::3333::5555:6666:7777:8888") is False, "IPV6 1111::3333::5555:6666:7777:8888 should be invalid"
        assert validator.is_valid_inet6_address("1111::3333:4444::6666:7777:8888") is False, "IPV6 1111::3333:4444::6666:7777:8888 should be invalid"
        assert validator.is_valid_inet6_address("1111::3333:4444:5555::7777:8888") is False, "IPV6 1111::3333:4444:5555::7777:8888 should be invalid"
        assert validator.is_valid_inet6_address("1111::3333:4444:5555:6666::8888") is False, "IPV6 1111::3333:4444:5555:6666::8888 should be invalid"
        assert validator.is_valid_inet6_address("1111::3333:4444:5555:6666:7777::") is False, "IPV6 1111::3333:4444:5555:6666:7777:: should be invalid"
        assert validator.is_valid_inet6_address("1111:2222::4444::6666:7777:8888") is False, "IPV6 1111:2222::4444::6666:7777:8888 should be invalid"
        assert validator.is_valid_inet6_address("1111:2222::4444:5555::7777:8888") is False, "IPV6 1111:2222::4444:5555::7777:8888 should be invalid"
        assert validator.is_valid_inet6_address("1111:2222::4444:5555:6666::8888") is False, "IPV6 1111:2222::4444:5555:6666::8888 should be invalid"
        assert validator.is_valid_inet6_address("1111:2222::4444:5555:6666:7777::") is False, "IPV6 1111:2222::4444:5555:6666:7777:: should be invalid"
        assert validator.is_valid_inet6_address("1111:2222:3333::5555::7777:8888") is False, "IPV6 1111:2222:3333::5555::7777:8888 should be invalid"
        assert validator.is_valid_inet6_address("1111:2222:3333::5555:6666::8888") is False, "IPV6 1111:2222:3333::5555:6666::8888 should be invalid"
        assert validator.is_valid_inet6_address("1111:2222:3333::5555:6666:7777::") is False, "IPV6 1111:2222:3333::5555:6666:7777:: should be invalid"
        assert validator.is_valid_inet6_address("1111:2222:3333:4444::6666::8888") is False, "IPV6 1111:2222:3333:4444::6666::8888 should be invalid"
        assert validator.is_valid_inet6_address("1111:2222:3333:4444::6666:7777::") is False, "IPV6 1111:2222:3333:4444::6666:7777:: should be invalid"
        assert validator.is_valid_inet6_address("1111:2222:3333:4444:5555::7777::") is False, "IPV6 1111:2222:3333:4444:5555::7777:: should be invalid"
        # Too many components
        assert validator.is_valid_inet6_address("1111:2222:3333:4444:5555:6666:7777:8888:1.2.3.4") is False, "IPV6 1111:2222:3333:4444:5555:6666:7777:8888:1.2.3.4 should be invalid"
        assert validator.is_valid_inet6_address("1111:2222:3333:4444:5555:6666:7777:1.2.3.4") is False, "IPV6 1111:2222:3333:4444:5555:6666:7777:1.2.3.4 should be invalid"
        assert validator.is_valid_inet6_address("1111:2222:3333:4444:5555:6666::1.2.3.4") is False, "IPV6 1111:2222:3333:4444:5555:6666::1.2.3.4 should be invalid"
        assert validator.is_valid_inet6_address("::2222:3333:4444:5555:6666:7777:1.2.3.4") is False, "IPV6 ::2222:3333:4444:5555:6666:7777:1.2.3.4 should be invalid"
        assert validator.is_valid_inet6_address("1111:2222:3333:4444:5555:6666:1.2.3.4.5") is False, "IPV6 1111:2222:3333:4444:5555:6666:1.2.3.4.5 should be invalid"
        # Too few components
        assert validator.is_valid_inet6_address("1111:2222:3333:4444:5555:1.2.3.4") is False, "IPV6 1111:2222:3333:4444:5555:1.2.3.4 should be invalid"
        assert validator.is_valid_inet6_address("1111:2222:3333:4444:1.2.3.4") is False, "IPV6 1111:2222:3333:4444:1.2.3.4 should be invalid"
        assert validator.is_valid_inet6_address("1111:2222:3333:1.2.3.4") is False, "IPV6 1111:2222:3333:1.2.3.4 should be invalid"
        assert validator.is_valid_inet6_address("1111:2222:1.2.3.4") is False, "IPV6 1111:2222:1.2.3.4 should be invalid"
        assert validator.is_valid_inet6_address("1111:1.2.3.4") is False, "IPV6 1111:1.2.3.4 should be invalid"
        assert validator.is_valid_inet6_address("1.2.3.4") is False, "IPV6 1.2.3.4 should be invalid"
        # Missing :
        assert validator.is_valid_inet6_address("11112222:3333:4444:5555:6666:1.2.3.4") is False, "IPV6 11112222:3333:4444:5555:6666:1.2.3.4 should be invalid"
        assert validator.is_valid_inet6_address("1111:22223333:4444:5555:6666:1.2.3.4") is False, "IPV6 1111:22223333:4444:5555:6666:1.2.3.4 should be invalid"
        assert validator.is_valid_inet6_address("1111:2222:33334444:5555:6666:1.2.3.4") is False, "IPV6 1111:2222:33334444:5555:6666:1.2.3.4 should be invalid"
        assert validator.is_valid_inet6_address("1111:2222:3333:44445555:6666:1.2.3.4") is False, "IPV6 1111:2222:3333:44445555:6666:1.2.3.4 should be invalid"
        assert validator.is_valid_inet6_address("1111:2222:3333:4444:55556666:1.2.3.4") is False, "IPV6 1111:2222:3333:4444:55556666:1.2.3.4 should be invalid"
        assert validator.is_valid_inet6_address("1111:2222:3333:4444:5555:66661.2.3.4") is False, "IPV6 1111:2222:3333:4444:5555:66661.2.3.4 should be invalid"
        # Missing .
        assert validator.is_valid_inet6_address("1111:2222:3333:4444:5555:6666:255255.255.255") is False, "IPV6 1111:2222:3333:4444:5555:6666:255255.255.255 should be invalid"
        assert validator.is_valid_inet6_address("1111:2222:3333:4444:5555:6666:255.255255.255") is False, "IPV6 1111:2222:3333:4444:5555:6666:255.255255.255 should be invalid"
        assert validator.is_valid_inet6_address("1111:2222:3333:4444:5555:6666:255.255.255255") is False, "IPV6 1111:2222:3333:4444:5555:6666:255.255.255255 should be invalid"
        # Missing : intended for ::
        assert validator.is_valid_inet6_address(":1.2.3.4") is False, "IPV6 :1.2.3.4 should be invalid"
        assert validator.is_valid_inet6_address(":6666:1.2.3.4") is False, "IPV6 :6666:1.2.3.4 should be invalid"
        assert validator.is_valid_inet6_address(":5555:6666:1.2.3.4") is False, "IPV6 :5555:6666:1.2.3.4 should be invalid"
        assert validator.is_valid_inet6_address(":4444:5555:6666:1.2.3.4") is False, "IPV6 :4444:5555:6666:1.2.3.4 should be invalid"
        assert validator.is_valid_inet6_address(":3333:4444:5555:6666:1.2.3.4") is False, "IPV6 :3333:4444:5555:6666:1.2.3.4 should be invalid"
        assert validator.is_valid_inet6_address(":2222:3333:4444:5555:6666:1.2.3.4") is False, "IPV6 :2222:3333:4444:5555:6666:1.2.3.4 should be invalid"
        assert validator.is_valid_inet6_address(":1111:2222:3333:4444:5555:6666:1.2.3.4") is False, "IPV6 :1111:2222:3333:4444:5555:6666:1.2.3.4 should be invalid"
        # :::
        assert validator.is_valid_inet6_address(":::2222:3333:4444:5555:6666:1.2.3.4") is False, "IPV6 :::2222:3333:4444:5555:6666:1.2.3.4 should be invalid"
        assert validator.is_valid_inet6_address("1111:::3333:4444:5555:6666:1.2.3.4") is False, "IPV6 1111:::3333:4444:5555:6666:1.2.3.4 should be invalid"
        assert validator.is_valid_inet6_address("1111:2222:::4444:5555:6666:1.2.3.4") is False, "IPV6 1111:2222:::4444:5555:6666:1.2.3.4 should be invalid"
        assert validator.is_valid_inet6_address("1111:2222:3333:::5555:6666:1.2.3.4") is False, "IPV6 1111:2222:3333:::5555:6666:1.2.3.4 should be invalid"
        assert validator.is_valid_inet6_address("1111:2222:3333:4444:::6666:1.2.3.4") is False, "IPV6 1111:2222:3333:4444:::6666:1.2.3.4 should be invalid"
        assert validator.is_valid_inet6_address("1111:2222:3333:4444:5555:::1.2.3.4") is False, "IPV6 1111:2222:3333:4444:5555:::1.2.3.4 should be invalid"
        # Double ::
        assert validator.is_valid_inet6_address("::2222::4444:5555:6666:1.2.3.4") is False, "IPV6 ::2222::4444:5555:6666:1.2.3.4 should be invalid"
        assert validator.is_valid_inet6_address("::2222:3333::5555:6666:1.2.3.4") is False, "IPV6 ::2222:3333::5555:6666:1.2.3.4 should be invalid"
        assert validator.is_valid_inet6_address("::2222:3333:4444::6666:1.2.3.4") is False, "IPV6 ::2222:3333:4444::6666:1.2.3.4 should be invalid"
        assert validator.is_valid_inet6_address("::2222:3333:4444:5555::1.2.3.4") is False, "IPV6 ::2222:3333:4444:5555::1.2.3.4 should be invalid"
        assert validator.is_valid_inet6_address("1111::3333::5555:6666:1.2.3.4") is False, "IPV6 1111::3333::5555:6666:1.2.3.4 should be invalid"
        assert validator.is_valid_inet6_address("1111::3333:4444::6666:1.2.3.4") is False, "IPV6 1111::3333:4444::6666:1.2.3.4 should be invalid"
        assert validator.is_valid_inet6_address("1111::3333:4444:5555::1.2.3.4") is False, "IPV6 1111::3333:4444:5555::1.2.3.4 should be invalid"
        assert validator.is_valid_inet6_address("1111:2222::4444::6666:1.2.3.4") is False, "IPV6 1111:2222::4444::6666:1.2.3.4 should be invalid"
        assert validator.is_valid_inet6_address("1111:2222::4444:5555::1.2.3.4") is False, "IPV6 1111:2222::4444:5555::1.2.3.4 should be invalid"
        assert validator.is_valid_inet6_address("1111:2222:3333::5555::1.2.3.4") is False, "IPV6 1111:2222:3333::5555::1.2.3.4 should be invalid"
        # Missing parts
        assert validator.is_valid_inet6_address("::.") is False, "IPV6 ::. should be invalid"
        assert validator.is_valid_inet6_address("::..") is False, "IPV6 ::.. should be invalid"
        assert validator.is_valid_inet6_address("::...") is False, "IPV6 ::... should be invalid"
        assert validator.is_valid_inet6_address("::1...") is False, "IPV6 ::1... should be invalid"
        assert validator.is_valid_inet6_address("::1.2..") is False, "IPV6 ::1.2.. should be invalid"
        assert validator.is_valid_inet6_address("::1.2.3.") is False, "IPV6 ::1.2.3. should be invalid"
        assert validator.is_valid_inet6_address("::.2..") is False, "IPV6 ::.2.. should be invalid"
        assert validator.is_valid_inet6_address("::.2.3.") is False, "IPV6 ::.2.3. should be invalid"
        assert validator.is_valid_inet6_address("::.2.3.4") is False, "IPV6 ::.2.3.4 should be invalid"
        assert validator.is_valid_inet6_address("::..3.") is False, "IPV6 ::..3. should be invalid"
        assert validator.is_valid_inet6_address("::..3.4") is False, "IPV6 ::..3.4 should be invalid"
        assert validator.is_valid_inet6_address("::...4") is False, "IPV6 ::...4 should be invalid"
        # Extra : in front
        assert validator.is_valid_inet6_address(":1111:2222:3333:4444:5555:6666:7777::") is False, "IPV6 :1111:2222:3333:4444:5555:6666:7777:: should be invalid"
        assert validator.is_valid_inet6_address(":1111:2222:3333:4444:5555:6666::") is False, "IPV6 :1111:2222:3333:4444:5555:6666:: should be invalid"
        assert validator.is_valid_inet6_address(":1111:2222:3333:4444:5555::") is False, "IPV6 :1111:2222:3333:4444:5555:: should be invalid"
        assert validator.is_valid_inet6_address(":1111:2222:3333:4444::") is False, "IPV6 :1111:2222:3333:4444:: should be invalid"
        assert validator.is_valid_inet6_address(":1111:2222:3333::") is False, "IPV6 :1111:2222:3333:: should be invalid"
        assert validator.is_valid_inet6_address(":1111:2222::") is False, "IPV6 :1111:2222:: should be invalid"
        assert validator.is_valid_inet6_address(":1111::") is False, "IPV6 :1111:: should be invalid"
        assert validator.is_valid_inet6_address(":1111:2222:3333:4444:5555:6666::8888") is False, "IPV6 :1111:2222:3333:4444:5555:6666::8888 should be invalid"
        assert validator.is_valid_inet6_address(":1111:2222:3333:4444:5555::8888") is False, "IPV6 :1111:2222:3333:4444:5555::8888 should be invalid"
        assert validator.is_valid_inet6_address(":1111:2222:3333:4444::8888") is False, "IPV6 :1111:2222:3333:4444::8888 should be invalid"
        assert validator.is_valid_inet6_address(":1111:2222:3333::8888") is False, "IPV6 :1111:2222:3333::8888 should be invalid"
        assert validator.is_valid_inet6_address(":1111:2222::8888") is False, "IPV6 :1111:2222::8888 should be invalid"
        assert validator.is_valid_inet6_address(":1111::8888") is False, "IPV6 :1111::8888 should be invalid"
        assert validator.is_valid_inet6_address(":::8888") is False, "IPV6 :::8888 should be invalid"
        assert validator.is_valid_inet6_address(":1111:2222:3333:4444:5555::7777:8888") is False, "IPV6 :1111:2222:3333:4444:5555::7777:8888 should be invalid"
        assert validator.is_valid_inet6_address(":1111:2222:3333:4444::7777:8888") is False, "IPV6 :1111:2222:3333:4444::7777:8888 should be invalid"
        assert validator.is_valid_inet6_address(":1111:2222:3333::7777:8888") is False, "IPV6 :1111:2222:3333::7777:8888 should be invalid"
        assert validator.is_valid_inet6_address(":1111:2222::7777:8888") is False, "IPV6 :1111:2222::7777:8888 should be invalid"
        assert validator.is_valid_inet6_address(":1111::7777:8888") is False, "IPV6 :1111::7777:8888 should be invalid"
        assert validator.is_valid_inet6_address(":::7777:8888") is False, "IPV6 :::7777:8888 should be invalid"
        assert validator.is_valid_inet6_address(":1111:2222:3333:4444::6666:7777:8888") is False, "IPV6 :1111:2222:3333:4444::6666:7777:8888 should be invalid"
        assert validator.is_valid_inet6_address(":1111:2222:3333::6666:7777:8888") is False, "IPV6 :1111:2222:3333::6666:7777:8888 should be invalid"
        assert validator.is_valid_inet6_address(":1111:2222::6666:7777:8888") is False, "IPV6 :1111:2222::6666:7777:8888 should be invalid"
        assert validator.is_valid_inet6_address(":1111::6666:7777:8888") is False, "IPV6 :1111::6666:7777:8888 should be invalid"
        assert validator.is_valid_inet6_address(":::6666:7777:8888") is False, "IPV6 :::6666:7777:8888 should be invalid"
        assert validator.is_valid_inet6_address(":1111:2222:3333::5555:6666:7777:8888") is False, "IPV6 :1111:2222:3333::5555:6666:7777:8888 should be invalid"
        assert validator.is_valid_inet6_address(":1111:2222::5555:6666:7777:8888") is False, "IPV6 :1111:2222::5555:6666:7777:8888 should be invalid"
        assert validator.is_valid_inet6_address(":1111::5555:6666:7777:8888") is False, "IPV6 :1111::5555:6666:7777:8888 should be invalid"
        assert validator.is_valid_inet6_address(":::5555:6666:7777:8888") is False, "IPV6 :::5555:6666:7777:8888 should be invalid"
        assert validator.is_valid_inet6_address(":1111:2222::4444:5555:6666:7777:8888") is False, "IPV6 :1111:2222::4444:5555:6666:7777:8888 should be invalid"
        assert validator.is_valid_inet6_address(":1111::4444:5555:6666:7777:8888") is False, "IPV6 :1111::4444:5555:6666:7777:8888 should be invalid"
        assert validator.is_valid_inet6_address(":::4444:5555:6666:7777:8888") is False, "IPV6 :::4444:5555:6666:7777:8888 should be invalid"
        assert validator.is_valid_inet6_address(":1111::3333:4444:5555:6666:7777:8888") is False, "IPV6 :1111::3333:4444:5555:6666:7777:8888 should be invalid"
        assert validator.is_valid_inet6_address(":::3333:4444:5555:6666:7777:8888") is False, "IPV6 :::3333:4444:5555:6666:7777:8888 should be invalid"
        assert validator.is_valid_inet6_address(":::2222:3333:4444:5555:6666:7777:8888") is False, "IPV6 :::2222:3333:4444:5555:6666:7777:8888 should be invalid"
        assert validator.is_valid_inet6_address(":1111:2222:3333:4444:5555:6666:1.2.3.4") is False, "IPV6 :1111:2222:3333:4444:5555:6666:1.2.3.4 should be invalid"
        assert validator.is_valid_inet6_address(":1111:2222:3333:4444:5555::1.2.3.4") is False, "IPV6 :1111:2222:3333:4444:5555::1.2.3.4 should be invalid"
        assert validator.is_valid_inet6_address(":1111:2222:3333:4444::1.2.3.4") is False, "IPV6 :1111:2222:3333:4444::1.2.3.4 should be invalid"
        assert validator.is_valid_inet6_address(":1111:2222:3333::1.2.3.4") is False, "IPV6 :1111:2222:3333::1.2.3.4 should be invalid"
        assert validator.is_valid_inet6_address(":1111:2222::1.2.3.4") is False, "IPV6 :1111:2222::1.2.3.4 should be invalid"
        assert validator.is_valid_inet6_address(":1111::1.2.3.4") is False, "IPV6 :1111::1.2.3.4 should be invalid"
        assert validator.is_valid_inet6_address(":::1.2.3.4") is False, "IPV6 :::1.2.3.4 should be invalid"
        assert validator.is_valid_inet6_address(":1111:2222:3333:4444::6666:1.2.3.4") is False, "IPV6 :1111:2222:3333:4444::6666:1.2.3.4 should be invalid"
        assert validator.is_valid_inet6_address(":1111:2222:3333::6666:1.2.3.4") is False, "IPV6 :1111:2222:3333::6666:1.2.3.4 should be invalid"
        assert validator.is_valid_inet6_address(":1111:2222::6666:1.2.3.4") is False, "IPV6 :1111:2222::6666:1.2.3.4 should be invalid"
        assert validator.is_valid_inet6_address(":1111::6666:1.2.3.4") is False, "IPV6 :1111::6666:1.2.3.4 should be invalid"
        assert validator.is_valid_inet6_address(":::6666:1.2.3.4") is False, "IPV6 :::6666:1.2.3.4 should be invalid"
        assert validator.is_valid_inet6_address(":1111:2222:3333::5555:6666:1.2.3.4") is False, "IPV6 :1111:2222:3333::5555:6666:1.2.3.4 should be invalid"
        assert validator.is_valid_inet6_address(":1111:2222::5555:6666:1.2.3.4") is False, "IPV6 :1111:2222::5555:6666:1.2.3.4 should be invalid"
        assert validator.is_valid_inet6_address(":1111::5555:6666:1.2.3.4") is False, "IPV6 :1111::5555:6666:1.2.3.4 should be invalid"
        assert validator.is_valid_inet6_address(":::5555:6666:1.2.3.4") is False, "IPV6 :::5555:6666:1.2.3.4 should be invalid"
        assert validator.is_valid_inet6_address(":1111:2222::4444:5555:6666:1.2.3.4") is False, "IPV6 :1111:2222::4444:5555:6666:1.2.3.4 should be invalid"
        assert validator.is_valid_inet6_address(":1111::4444:5555:6666:1.2.3.4") is False, "IPV6 :1111::4444:5555:6666:1.2.3.4 should be invalid"
        assert validator.is_valid_inet6_address(":::4444:5555:6666:1.2.3.4") is False, "IPV6 :::4444:5555:6666:1.2.3.4 should be invalid"
        assert validator.is_valid_inet6_address(":1111::3333:4444:5555:6666:1.2.3.4") is False, "IPV6 :1111::3333:4444:5555:6666:1.2.3.4 should be invalid"
        assert validator.is_valid_inet6_address(":::2222:3333:4444:5555:6666:1.2.3.4") is False, "IPV6 :::2222:3333:4444:5555:6666:1.2.3.4 should be invalid"
        # Extra : at end
        assert validator.is_valid_inet6_address("1111:2222:3333:4444:5555:6666:7777:::") is False, "IPV6 1111:2222:3333:4444:5555:6666:7777::: should be invalid"
        assert validator.is_valid_inet6_address("1111:2222:3333:4444:5555:6666:::") is False, "IPV6 1111:2222:3333:4444:5555:6666::: should be invalid"
        assert validator.is_valid_inet6_address("1111:2222:3333:4444:5555:::") is False, "IPV6 1111:2222:3333:4444:5555::: should be invalid"
        assert validator.is_valid_inet6_address("1111:2222:3333:4444:::") is False, "IPV6 1111:2222:3333:4444::: should be invalid"
        assert validator.is_valid_inet6_address("1111:2222:3333:::") is False, "IPV6 1111:2222:3333::: should be invalid"
        assert validator.is_valid_inet6_address("1111:2222:::") is False, "IPV6 1111:2222::: should be invalid"
        assert validator.is_valid_inet6_address("1111:::") is False, "IPV6 1111::: should be invalid"
        assert validator.is_valid_inet6_address("1111:2222:3333:4444:5555:6666::8888:") is False, "IPV6 1111:2222:3333:4444:5555:6666::8888: should be invalid"
        assert validator.is_valid_inet6_address("1111:2222:3333:4444:5555::8888:") is False, "IPV6 1111:2222:3333:4444:5555::8888: should be invalid"
        assert validator.is_valid_inet6_address("1111:2222:3333:4444::8888:") is False, "IPV6 1111:2222:3333:4444::8888: should be invalid"
        assert validator.is_valid_inet6_address("1111:2222:3333::8888:") is False, "IPV6 1111:2222:3333::8888: should be invalid"
        assert validator.is_valid_inet6_address("1111:2222::8888:") is False, "IPV6 1111:2222::8888: should be invalid"
        assert validator.is_valid_inet6_address("1111::8888:") is False, "IPV6 1111::8888: should be invalid"
        assert validator.is_valid_inet6_address("::8888:") is False, "IPV6 ::8888: should be invalid"
        assert validator.is_valid_inet6_address("1111:2222:3333:4444:5555::7777:8888:") is False, "IPV6 1111:2222:3333:4444:5555::7777:8888: should be invalid"
        assert validator.is_valid_inet6_address("1111:2222:3333:4444::7777:8888:") is False, "IPV6 1111:2222:3333:4444::7777:8888: should be invalid"
        assert validator.is_valid_inet6_address("1111:2222:3333::7777:8888:") is False, "IPV6 1111:2222:3333::7777:8888: should be invalid"
        assert validator.is_valid_inet6_address("1111:2222::7777:8888:") is False, "IPV6 1111:2222::7777:8888: should be invalid"
        assert validator.is_valid_inet6_address("1111::7777:8888:") is False, "IPV6 1111::7777:8888: should be invalid"
        assert validator.is_valid_inet6_address("::7777:8888:") is False, "IPV6 ::7777:8888: should be invalid"
        assert validator.is_valid_inet6_address("1111:2222:3333:4444::6666:7777:8888:") is False, "IPV6 1111:2222:3333:4444::6666:7777:8888: should be invalid"
        assert validator.is_valid_inet6_address("1111:2222:3333::6666:7777:8888:") is False, "IPV6 1111:2222:3333::6666:7777:8888: should be invalid"
        assert validator.is_valid_inet6_address("1111:2222::6666:7777:8888:") is False, "IPV6 1111:2222::6666:7777:8888: should be invalid"
        assert validator.is_valid_inet6_address("1111::6666:7777:8888:") is False, "IPV6 1111::6666:7777:8888: should be invalid"
        assert validator.is_valid_inet6_address("::6666:7777:8888:") is False, "IPV6 ::6666:7777:8888: should be invalid"
        assert validator.is_valid_inet6_address("1111:2222:3333::5555:6666:7777:8888:") is False, "IPV6 1111:2222:3333::5555:6666:7777:8888: should be invalid"
        assert validator.is_valid_inet6_address("1111:2222::5555:6666:7777:8888:") is False, "IPV6 1111:2222::5555:6666:7777:8888: should be invalid"
        assert validator.is_valid_inet6_address("1111::5555:6666:7777:8888:") is False, "IPV6 1111::5555:6666:7777:8888: should be invalid"
        assert validator.is_valid_inet6_address("::5555:6666:7777:8888:") is False, "IPV6 ::5555:6666:7777:8888: should be invalid"
        assert validator.is_valid_inet6_address("1111:2222::4444:5555:6666:7777:8888:") is False, "IPV6 1111:2222::4444:5555:6666:7777:8888: should be invalid"
        assert validator.is_valid_inet6_address("1111::4444:5555:6666:7777:8888:") is False, "IPV6 1111::4444:5555:6666:7777:8888: should be invalid"
        assert validator.is_valid_inet6_address("::4444:5555:6666:7777:8888:") is False, "IPV6 ::4444:5555:6666:7777:8888: should be invalid"
        assert validator.is_valid_inet6_address("1111::3333:4444:5555:6666:7777:8888:") is False, "IPV6 1111::3333:4444:5555:6666:7777:8888: should be invalid"
        assert validator.is_valid_inet6_address("::3333:4444:5555:6666:7777:8888:") is False, "IPV6 ::3333:4444:5555:6666:7777:8888: should be invalid"
        assert validator.is_valid_inet6_address("::2222:3333:4444:5555:6666:7777:8888:") is False, "IPV6 ::2222:3333:4444:5555:6666:7777:8888: should be invalid"
        assert validator.is_valid_inet6_address("0:a:b:c:d:e:f::") is True, "IPV6 0:a:b:c:d:e:f:: should be valid"
        assert validator.is_valid_inet6_address("::0:a:b:c:d:e:f") is True, "IPV6 ::0:a:b:c:d:e:f should be valid"
        assert validator.is_valid_inet6_address("a:b:c:d:e:f:0::") is True, "IPV6 a:b:c:d:e:f:0:: should be valid"
        assert validator.is_valid_inet6_address("':10.0.0.1") is False, "IPV6 ':10.0.0.1 should be invalid"

    def test_reserved_inet_addresses(self):
        """
        Test reserved IPs.
        """
        validator = InetAddressValidator()
        assert validator.is_valid("127.0.0.1") is True, "localhost IP should be valid"
        assert validator.is_valid("255.255.255.255") is True, "broadcast IP should be valid"
    
    def test_validator_365(self):
        validator = InetAddressValidator()
        assert validator.is_valid("2001:0438:FFFE:0000:0000:0000:0000:0A35") is True, "2001:0438:FFFE:0000:0000:0000:0000:0A35 should be valid"
    
    def test_validator_419(self):
        validator = InetAddressValidator()
        addr = "0:0:0:0:0:0:13.1.68.3"
        assert validator.is_valid(addr) is True
        addr = "0:0:0:0:0:FFFF:129.144.52.38"
        assert validator.is_valid(addr) is True
        addr = "::13.1.68.3"
        assert validator.is_valid(addr) is True
        addr = "::FFFF:129.144.52.38"
        assert validator.is_valid(addr) is True

        addr = "::ffff:192.168.1.1:192.168.1.1"
        assert validator.is_valid(addr) is False
        addr = "::192.168.1.1:192.168.1.1"
        assert validator.is_valid(addr) is False
    
    def test_validator_445(self):
        """
        Inet6Address may also contain a scope id.
        """
        validator = InetAddressValidator()
        valid = ["2001:0000:1234:0000:0000:C1C0:ABCD:0876",
                "2001:0000:1234:0000:0000:C1C0:ABCD:0876/123",
                "2001:0000:1234:0000:0000:C1C0:ABCD:0876/0",
                "2001:0000:1234:0000:0000:C1C0:ABCD:0876%0",
                "2001:0000:1234:0000:0000:C1C0:ABCD:0876%abcdefgh"]
        
        invalid = ["2001:0000:1234:0000:0000:C1C0:ABCD:0876/129",    # too big
                "2001:0000:1234:0000:0000:C1C0:ABCD:0876/-0",        # sign not allowed
                "2001:0000:1234:0000:0000:C1C0:ABCD:0876/+0",        # sign not allowed
                "2001:0000:1234:0000:0000:C1C0:ABCD:0876/10O",       # non-digit
                "2001:0000:1234:0000:0000:C1C0:ABCD:0876/0%0",       # bits before node-id
                "2001:0000:1234:0000:0000:C1C0:ABCD:0876%abc defgh", # space in node id
                "2001:0000:1234:0000:0000:C1C0:ABCD:0876%abc%defgh", # '%' in node id
        ]
        for item in valid:
            assert validator.is_valid(item) is True, f"{item} should be valid"
        for item in invalid:
            assert validator.is_valid(item) is False, f"{item} should be invalid"
