from __future__ import annotations
from functools import lru_cache
from typing import Dict


class Settings:
    """Central project configuration (pure data, no logic)."""

    # --- Max values for cyclic (circular) features ---
    circular_max_values: Dict[str, int] = {
        "hour_of_day": 24,
        "day_of_week": 7,
        "gameweek": 38,
        "day_of_year": 365,
    }

    # CSV paths
    paths: Dict[str, str] = {
        "matches": "app/data/raw/matches/matches.csv",
        "players": "app/data/raw/players/players.csv",
        "keepers": "app/data/raw/players/keepers.csv",
        "players_features": "app/data/features/players/players.csv",
        "keepers_features": "app/data/features/players/keepers.csv",
        "ranking": "app/data/raw/rankings/{country}/{last_season[0]}/rankings_{competition}_{last_season[0]}.csv",
        "trophies": "app/data/raw/trophies/{country}/{last_season[0]}/trophies_{competition}_{last_season[0]}.csv",
        "dataset_global": "app/data/features/global/dataset.csv",
        "dataset_country": "app/data/features/{country}/dataset_{country}.csv",
        "dataset_competition": "app/data/features/{country}/{competition}/dataset_{competition}.csv",
    }

    # Random state
    random_state: int = 42

    # Centralized feature configuration
    features_config: Dict[str, any] = {
        # Base categorical columns (no processing needed)
        "fbref_informations": [
            "gameweek",
            "competition",
            "date_of_match",
            "hour_of_the_match",
            "day_of_week",
            "day_of_year",
            "hour_of_day",
            "home_team_name",
            "away_team_name",
            "home_trainer",
            "away_trainer",
            "stadium",
            "attendance",
            "referee",
            "var",
            "competition_type",
            "competition_country",
            "result",
            "rest_time_home_team",
            "rest_time_away_team",
        ],
        # Ranking and odds columns (carried from matches dataset)
        "ranking": [
            "home_team_rank",
            "away_team_rank",
            "home_team_points",
            "away_team_points",
            "home_team_goals_for",
            "away_team_goals_for",
            "home_team_goals_against",
            "away_team_goals_against",
            "home_team_goals_difference",
            "away_team_goals_difference",
            "first_place",
            "second_place",
            "third_place",
            "years_first_division_total",
            "years_first_division_consecutive",
        ],
        # Supervised target column
        "target": "result",
        # Features from fotmob
        "fotmob": {},
        # Features from fotmob
        "transfermarkt": {},
        # Features with binning configuration: thresholds + operators
        "fbref_features": {
            "possession": {
                "thresholds": [20, 30, 40, 50, 60, 70, 80],
                "operators": ["le", "le", "le", "ge", "ge", "ge", "ge"],
            },
            "goals": {
                "thresholds": [0, 1, 2, 3, 4, 5, 6],
                "operators": ["eq", "ge", "ge", "ge", "ge", "ge", "ge"],
            },
            "PlayersAge": {
                "thresholds": [23.0, 27.0, 30.0, 30.0],
                "operators": ["le", "le", "le", "gt"],
            },
            "PlayersAvgMinutes": {
                "thresholds": [70.0, 70.0, 80.0, 80.0],
                "operators": ["le", "gt", "le", "gt"],
            },
            "PlayersShots": {
                "thresholds": [
                    0.0,
                    1.0,
                    2.0,
                    3.0,
                    4.0,
                    5.0,
                    6.0,
                    7.0,
                    8.0,
                    9.0,
                    10.0,
                    11.0,
                    12.0,
                    13.0,
                    14.0,
                    15.0,
                ],
                "operators": [
                    "eq",
                    "eq",
                    "eq",
                    "eq",
                    "eq",
                    "eq",
                    "eq",
                    "eq",
                    "eq",
                    "eq",
                    "eq",
                    "eq",
                    "eq",
                    "eq",
                    "eq",
                    "ge",
                ],
            },
            "PlayersShotsOnTarget": {
                "thresholds": [0.0, 1.0, 2.0, 3.0, 4.0, 5.0, 6.0, 7.0, 8.0, 9.0, 10.0],
                "operators": [
                    "eq",
                    "eq",
                    "eq",
                    "eq",
                    "eq",
                    "eq",
                    "eq",
                    "eq",
                    "eq",
                    "eq",
                    "ge",
                ],
            },
            "PlayersCompletedPasses": {
                "thresholds": [100.0, 200.0, 300.0, 400.0, 500.0, 600.0],
                "operators": ["le", "le", "le", "ge", "ge", "ge"],
            },
            "PlayersAttemptedPasses": {
                "thresholds": [200.0, 300.0, 400.0, 500.0, 600.0, 700.0, 800.0],
                "operators": ["le", "le", "le", "le", "le", "le", "ge"],
            },
            "Players%CompletedPasses": {
                "thresholds": [40.0, 50.0, 60.0, 70.0, 80.0, 90.0],
                "operators": ["le", "le", "ge", "ge", "ge", "ge"],
            },
            "PlayersDistancePasses": {
                "thresholds": [50, 100, 150, 200, 250, 300],
                "operators": ["le", "le", "le", "ge", "ge", "ge"],
            },
            "PlayersDistanceProgression": {
                "thresholds": [20, 40, 60, 80, 100, 120],
                "operators": ["le", "le", "le", "ge", "ge", "ge"],
            },
            "PlayersShortPasses": {
                "thresholds": [50, 100, 150, 200, 250],
                "operators": ["le", "le", "le", "ge", "ge"],
            },
            "PlayersAttemptedShortPasses": {
                "thresholds": [60, 120, 180, 240, 300],
                "operators": ["le", "le", "le", "ge", "ge"],
            },
            "Players%ShortCompletedPasses": {
                "thresholds": [60, 70, 80, 90],
                "operators": ["le", "le", "ge", "ge"],
            },
            "PlayersMediumPasses": {
                "thresholds": [30, 60, 90, 120, 150],
                "operators": ["le", "le", "le", "ge", "ge"],
            },
            "PlayersAttemptedMediumPasses": {
                "thresholds": [40, 80, 120, 160, 200],
                "operators": ["le", "le", "le", "ge", "ge"],
            },
            "Players%MediumCompletedPasses": {
                "thresholds": [60, 70, 80, 90],
                "operators": ["le", "le", "ge", "ge"],
            },
            "PlayersLongPasses": {
                "thresholds": [10, 20, 30, 40, 50],
                "operators": ["le", "le", "le", "ge", "ge"],
            },
            "PlayersAttemptedLongPasses": {
                "thresholds": [15, 30, 45, 60, 75],
                "operators": ["le", "le", "le", "ge", "ge"],
            },
            "Players%LongCompletedPasses": {
                "thresholds": [40, 50, 60, 70],
                "operators": ["le", "le", "ge", "ge"],
            },
            "PlayersAssistance": {
                "thresholds": [0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10],
                "operators": [
                    "eq",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                ],
            },
            "PlayersExpectedGoalsAssistance": {
                "thresholds": [0.1, 0.2, 0.3, 0.4, 0.5, 0.6, 0.7, 0.8, 0.9, 1.0],
                "operators": [
                    "le",
                    "le",
                    "le",
                    "le",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                ],
            },
            "PlayersExpectedAssistance": {
                "thresholds": [0.1, 0.2, 0.3, 0.4, 0.5, 0.6, 0.7, 0.8, 0.9, 1.0],
                "operators": [
                    "le",
                    "le",
                    "le",
                    "le",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                ],
            },
            "PlayersKeyPasses": {
                "thresholds": [1, 2, 3, 4, 5, 6, 7, 8, 9, 10],
                "operators": [
                    "eq",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                ],
            },
            "PlayersLast1/3Passes": {
                "thresholds": [5, 10, 15, 20, 25, 30, 35, 40, 45, 50],
                "operators": [
                    "le",
                    "le",
                    "le",
                    "le",
                    "le",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                ],
            },
            "PlayersGoalAreaPasses": {
                "thresholds": [1, 3, 5, 7, 9, 11, 13, 15, 17, 19],
                "operators": [
                    "le",
                    "le",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                ],
            },
            "PlayersGoalAreaCrosses": {
                "thresholds": [0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10],
                "operators": [
                    "eq",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                ],
            },
            "PlayersGoalPasses": {
                "thresholds": [0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10],
                "operators": [
                    "eq",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                ],
            },
            "PlayersLiveBallPasses": {
                "thresholds": [20, 40, 60, 80, 100, 120, 140, 160, 180, 200],
                "operators": [
                    "le",
                    "le",
                    "le",
                    "le",
                    "le",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                ],
            },
            "PlayersDeadBallPasses": {
                "thresholds": [5, 10, 15, 20, 25, 30, 35, 40, 45, 50],
                "operators": [
                    "le",
                    "le",
                    "le",
                    "le",
                    "le",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                ],
            },
            "PlayersFreeKick": {
                "thresholds": [0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10],
                "operators": [
                    "eq",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                ],
            },
            "PlayersThroughPasses": {
                "thresholds": [0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10],
                "operators": [
                    "eq",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                ],
            },
            "PlayersSidePasses": {
                "thresholds": [10, 20, 30, 40, 50, 60, 70, 80, 90, 100],
                "operators": [
                    "le",
                    "le",
                    "le",
                    "le",
                    "le",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                ],
            },
            "result": {
                "thresholds": [2, 1, 0, 2, 0],
                "operators": ["eq", "eq", "eq", "gt", "lt"],
            },
            "PlayersCrosses": {
                "thresholds": [5, 10, 15, 20, 25, 30, 35, 40, 45, 50],
                "operators": [
                    "le",
                    "le",
                    "le",
                    "le",
                    "le",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                ],
            },
            "PlayersStrongcrosses": {
                "thresholds": [0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10],
                "operators": [
                    "eq",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                ],
            },
            "PlayersCorner": {
                "thresholds": [0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10],
                "operators": [
                    "eq",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                ],
            },
            "PlayersCornerIn": {
                "thresholds": [0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10],
                "operators": [
                    "eq",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                ],
            },
            "PlayersCornerOut": {
                "thresholds": [0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10],
                "operators": [
                    "eq",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                ],
            },
            "PlayersCornerRect": {
                "thresholds": [0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10],
                "operators": [
                    "eq",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                ],
            },
            "PlayersOffsidePasses": {
                "thresholds": [0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10],
                "operators": [
                    "eq",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                ],
            },
            "PlayersPassesBlocked": {
                "thresholds": [0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10],
                "operators": [
                    "eq",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                ],
            },
            "PlayersTackles": {
                "thresholds": [5, 10, 15, 20, 25, 30, 35, 40, 45, 50],
                "operators": [
                    "le",
                    "le",
                    "le",
                    "le",
                    "le",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                ],
            },
            "PlayersSuccessfulTackles": {
                "thresholds": [0, 2, 4, 6, 8, 10, 12, 14, 16, 18, 20],
                "operators": [
                    "eq",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                ],
            },
            "PlayersTacklesInDefense": {
                "thresholds": [0, 2, 4, 6, 8, 10, 12, 14, 16, 18, 20],
                "operators": [
                    "eq",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                ],
            },
            "PlayersTacklesInMedium": {
                "thresholds": [0, 2, 4, 6, 8, 10, 12, 14, 16, 18, 20],
                "operators": [
                    "eq",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                ],
            },
            "PlayersTacklesInAttack": {
                "thresholds": [0, 1, 2, 3, 4, 5, 6],
                "operators": ["eq", "ge", "ge", "ge", "ge", "ge", "ge"],
            },
            "PlayersDribblerTackles": {
                "thresholds": [0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10],
                "operators": [
                    "eq",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                ],
            },
            "PlayersAttemptedDribblerTackles": {
                "thresholds": [0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10],
                "operators": [
                    "eq",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                ],
            },
            "Players%DribblerTacklesCompleted": {
                "thresholds": [0, 10, 20, 30, 40, 50, 60, 70, 80, 90, 100],
                "operators": [
                    "le",
                    "le",
                    "le",
                    "le",
                    "le",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                ],
            },
            "PlayersDribblerTacklesNonCompleted": {
                "thresholds": [0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10],
                "operators": [
                    "eq",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                ],
            },
            "PlayersBallsBlocked": {
                "thresholds": [0, 2, 4, 6, 8, 10, 12, 14, 16, 18, 20],
                "operators": [
                    "eq",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                ],
            },
            "PlayersShotsBlocked": {
                "thresholds": [0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10],
                "operators": [
                    "eq",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                ],
            },
            "PlayersInterceptions": {
                "thresholds": [0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10],
                "operators": [
                    "eq",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                ],
            },
            "PlayersTackles+Interceptions": {
                "thresholds": [0, 2, 4, 6, 8, 10, 12, 14, 16, 18, 20],
                "operators": [
                    "eq",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                ],
            },
            "PlayersClearances": {
                "thresholds": [0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10],
                "operators": [
                    "eq",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                ],
            },
            "PlayersMistakesRivalShots": {
                "thresholds": [0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10],
                "operators": [
                    "eq",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                ],
            },
            "PlayersTouches": {
                "thresholds": [20, 30, 40, 50, 60, 70, 80, 90, 100],
                "operators": ["le", "le", "le", "le", "le", "ge", "ge", "ge", "ge"],
            },
            "PlayersOwnPenaltyAreaTouches": {
                "thresholds": [0, 2, 4, 6, 8, 10],
                "operators": ["eq", "ge", "ge", "ge", "ge", "ge"],
            },
            "PlayersTouchesInDefense": {
                "thresholds": [0, 5, 10, 15, 20, 25],
                "operators": ["le", "le", "le", "le", "le", "ge"],
            },
            "PlayersTouchesInMedium": {
                "thresholds": [0, 5, 10, 15, 20, 25],
                "operators": ["le", "le", "le", "le", "le", "ge"],
            },
            "PlayersTouchesInAttack": {
                "thresholds": [0, 5, 10, 15, 20, 25],
                "operators": ["le", "le", "le", "le", "le", "ge"],
            },
            "PlayersAwayPenaltyAreaTouches": {
                "thresholds": [0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10],
                "operators": [
                    "eq",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                ],
            },
            "PlayersLiveBallTouches": {
                "thresholds": [10, 20, 30, 40, 50, 60, 70, 80, 90, 100],
                "operators": [
                    "le",
                    "le",
                    "le",
                    "le",
                    "le",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                ],
            },
            "PlayersAttemptedDribbles": {
                "thresholds": [0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10],
                "operators": [
                    "eq",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                ],
            },
            "PlayersDribblesCompleted": {
                "thresholds": [0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10],
                "operators": [
                    "eq",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                ],
            },
            "Players%DribblesCompleted": {
                "thresholds": [0, 20, 40, 60, 80, 100],
                "operators": ["le", "le", "le", "le", "le", "ge"],
            },
            "PlayersBallCarries": {
                "thresholds": [0, 10, 20, 30, 40, 50, 60, 70],
                "operators": ["eq", "ge", "ge", "ge", "ge", "ge", "ge", "ge"],
            },
            "PlayersDistanceCarried": {
                "thresholds": [0, 10, 20, 30, 40, 50, 60],
                "operators": ["eq", "ge", "ge", "ge", "ge", "ge", "ge"],
            },
            "PlayersForwardDistanceCarried": {
                "thresholds": [0, 5, 10, 15, 20, 25],
                "operators": ["eq", "ge", "ge", "ge", "ge", "ge"],
            },
            "PlayersForwardCarries": {
                "thresholds": [0, 1, 2, 3, 4, 5, 6, 7],
                "operators": ["eq", "ge", "ge", "ge", "ge", "ge", "ge", "ge"],
            },
            "PlayersCarriesInAttack": {
                "thresholds": [0, 1, 2, 3, 4, 5, 6],
                "operators": ["eq", "ge", "ge", "ge", "ge", "ge", "ge"],
            },
            "PlayersAwayPenaltyAreaCarries": {
                "thresholds": [0, 1, 2, 3, 4, 5, 6],
                "operators": ["eq", "ge", "ge", "ge", "ge", "ge", "ge"],
            },
            "PlayersLostControlCarries": {
                "thresholds": [0, 1, 2, 3, 4, 5],
                "operators": ["eq", "ge", "ge", "ge", "ge", "ge"],
            },
            "PlayersLostCarries": {
                "thresholds": [0, 1, 2, 3, 4, 5],
                "operators": ["eq", "ge", "ge", "ge", "ge", "ge"],
            },
            "PlayersPassesReception": {
                "thresholds": [0, 10, 20, 30, 40, 50, 60, 70],
                "operators": ["eq", "ge", "ge", "ge", "ge", "ge", "ge", "ge"],
            },
            "PlayersAttackPassesReception": {
                "thresholds": [0, 5, 10, 15, 20],
                "operators": ["eq", "ge", "ge", "ge", "ge"],
            },
            "PlayersYellowCards": {
                "thresholds": [0, 1, 2, 3, 4],
                "operators": ["eq", "ge", "ge", "ge", "ge"],
            },
            "PlayersRedCards": {
                "thresholds": [0, 1, 2, 3],
                "operators": ["eq", "ge", "ge", "ge"],
            },
            "PlayersSecondYellowCards": {
                "thresholds": [0, 1, 2],
                "operators": ["eq", "ge", "ge"],
            },
            "PlayersFouls": {
                "thresholds": [0, 10, 20, 30, 40, 50, 60, 70, 80, 90],
                "operators": [
                    "eq",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                ],
            },
            "PlayersFoulsReceived": {
                "thresholds": [0, 10, 20, 30, 40, 50, 60, 70, 80, 90],
                "operators": [
                    "eq",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                    "ge",
                ],
            },
            "PlayersPenalties": {
                "thresholds": [0, 1, 2, 3],
                "operators": ["eq", "ge", "ge", "ge"],
            },
            "PlayersPenaltiesConceded": {
                "thresholds": [0, 1, 2, 3],
                "operators": ["eq", "ge", "ge", "ge"],
            },
            "PlayersLostBallRecoveries": {
                "thresholds": [0, 10, 20, 30, 40, 50],
                "operators": ["eq", "ge", "ge", "ge", "ge", "ge"],
            },
            "PlayersAerialsWon": {
                "thresholds": [0, 10, 20, 30, 40, 50, 60, 70],
                "operators": ["eq", "ge", "ge", "ge", "ge", "ge", "ge", "ge"],
            },
            "PlayersAerialsLost": {
                "thresholds": [0, 10, 20, 30, 40, 50],
                "operators": ["eq", "ge", "ge", "ge", "ge", "ge"],
            },
            "Players%AerialsWon": {
                "thresholds": [0, 10, 20, 30, 40, 50, 60],
                "operators": ["eq", "ge", "ge", "ge", "ge", "ge", "ge"],
            },
            "KeepersShotsOnTargetAgainst": {
                "thresholds": [0, 1, 2, 3, 4, 5, 6],
                "operators": ["eq", "ge", "ge", "ge", "ge", "ge", "ge"],
            },
            "KeepersGoalsAgainst": {
                "thresholds": [0, 1, 2, 3, 4, 5],
                "operators": ["eq", "ge", "ge", "ge", "ge", "ge"],
            },
            "KeepersSaved": {
                "thresholds": [0, 1, 2, 3, 4, 5],
                "operators": ["eq", "ge", "ge", "ge", "ge", "ge"],
            },
            "Keepers%Saved": {
                "thresholds": [0, 10, 20, 30, 40, 50, 60],
                "operators": ["eq", "ge", "ge", "ge", "ge", "ge", "ge"],
            },
            "KeepersxG": {
                "thresholds": [0, 0.1, 0.2, 0.3, 0.4, 0.5],
                "operators": ["eq", "ge", "ge", "ge", "ge", "ge"],
            },
            "KeepersPassesLaunched": {
                "thresholds": [0, 1, 2, 3, 4, 5],
                "operators": ["eq", "ge", "ge", "ge", "ge", "ge"],
            },
            "KeepersAttemptedPassesLaunched": {
                "thresholds": [0, 1, 2, 3, 4, 5],
                "operators": ["eq", "ge", "ge", "ge", "ge", "ge"],
            },
            "Keepers%CompletedPassesLaunched": {
                "thresholds": [0, 20, 40, 60, 80, 100],
                "operators": ["eq", "ge", "ge", "ge", "ge", "ge"],
            },
            "KeepersPasses": {
                "thresholds": [0, 1, 2, 3, 4, 5],
                "operators": ["eq", "ge", "ge", "ge", "ge", "ge"],
            },
            "KeepersAttemptedPasses": {
                "thresholds": [0, 1, 2, 3, 4, 5],
                "operators": ["eq", "ge", "ge", "ge", "ge", "ge"],
            },
            "Keepers%CompletedPasses": {
                "thresholds": [0, 20, 40, 60, 80, 100],
                "operators": ["eq", "ge", "ge", "ge", "ge", "ge"],
            },
            "KeepersPassesDistance": {
                "thresholds": [0, 10, 20, 30, 40, 50],
                "operators": ["eq", "ge", "ge", "ge", "ge", "ge"],
            },
            "KeepersAttemptedKicks": {
                "thresholds": [0, 1, 2, 3, 4],
                "operators": ["eq", "ge", "ge", "ge", "ge"],
            },
            "Keepers%Kicks": {
                "thresholds": [0, 20, 40, 60, 80, 100],
                "operators": ["eq", "ge", "ge", "ge", "ge", "ge"],
            },
            "KeepersKicksDistance": {
                "thresholds": [0, 10, 20, 30, 40, 50],
                "operators": ["eq", "ge", "ge", "ge", "ge", "ge"],
            },
            "KeepersCrosses": {
                "thresholds": [0, 1, 2, 3, 4],
                "operators": ["eq", "ge", "ge", "ge", "ge"],
            },
            "KeepersCrossesStopped": {
                "thresholds": [0, 1, 2, 3, 4],
                "operators": ["eq", "ge", "ge", "ge", "ge"],
            },
            "Keepers%CrossesStopped": {
                "thresholds": [0, 20, 40, 60, 80, 100],
                "operators": ["eq", "ge", "ge", "ge", "ge", "ge"],
            },
            "KeepersActionsOutsideArea": {
                "thresholds": [0, 1, 2, 3, 4],
                "operators": ["eq", "ge", "ge", "ge", "ge"],
            },
            "KeepersDistanceActionsArea": {
                "thresholds": [0, 10, 20, 30, 40, 50],
                "operators": ["eq", "ge", "ge", "ge", "ge", "ge"],
            },
        },
    }

    # Competition configuration
    competitions_config: Dict[str, Dict] = {
        # ==================== SPANISH COMPETITIONS ====================
        "liga": {
            "country": "spain",
            "information_scraping_urls": {
                "matches": "https://fbref.com/en/comps/12/{last_season[0]}/schedule/{last_season[0]}-La-Liga-Scores-and-Fixtures",
                "fotmob": "https://www.fotmob.com/en/leagues/87/matches/laliga?season={last_season[0]}&group=by-date&page={page}",
                # "teams": "https://www.transfermarkt.com/xxxxxx/startseite/verein/{transfermarkt_id}/saison_id/{last_season[1]}"
            },
            "betting_scraping_urls": {
                "betclic": "https://www.betclic.fr/football-sfootball/espagne-laliga-c7",
                "winamax": "https://www.winamax.fr/paris-sportifs/sports/1/32/36",
                "unibet": "https://www.unibet.fr/sport/football/espagne/laliga?filter=Top+Paris&subFilter=R%C3%A9sultat+du+match",
                "parionssport": "https://www.enligne.parionssport.fdj.fr/paris-football/espagne/laliga",
                "pmu": "https://parisportif.pmu.fr/home/wrapper/events?fId=1&activeSportId=1&leagues=%5B486%5D",
                "bwin": "https://www.bwin.fr/fr/sports/football-4/paris-sportifs/espagne-28/laliga-102829",
                "netbet": "https://www.netbet.fr/football/espagne/laliga",
                "betsson": "https://betsson.fr/fr/competitions/football/laliga?competition_id=55775.1",
            },
        },
        #     "copa_del_rey": {
        #         "country": "spain",
        #         "paths": {
        #             "ranking": "app/data/raw/spain/{last_season[0]}/rankings_cup_{last_season[0]}.csv",
        #             "matches": "app/data/raw/spain/{last_season[0]}/matches_cup_{last_season[0]}.csv",
        #             "trophies": "app/data/raw/spain/{last_season[0]}/trophies_{last_season[0]}.csv",
        #         },
        #         "information_scraping_urls": {
        #             "matches": "https://fbref.com/en/comps/569/{last_season[0]}/schedule/{last_season[0]}-Copa-del-Rey-Scores-and-Fixtures",
        #             "teams": "https://www.transfermarkt.com/xxxxxx/startseite/verein/{transfermarkt_id}/saison_id/{last_season[1]}"
        #         },
        #         "betting_scraping_urls": {
        #             #"betclic": "https://www.betclic.fr/football-sfootball/espagne-coupe-du-roi-c47",
        #             #"unibet": "https://www.unibet.fr/sport/football/espagne/coupe-du-roi?filter=Comp%C3%A9tition&subFilter=Vainqueur+de+la+comp%C3%A9tition",
        #             "parionssport": "https://www.enligne.parionssport.fdj.fr/paris-football/espagne/copa-del-rey",
        #             "netbet": "https://www.netbet.fr/football/espagne/laliga",
        #             #"betsson": "https://betsson.fr/fr/competitions/football/coupe-du-roi-espagne?competition_id=4117860.1&match_id=outrights",
        #         },
        #     },
        #     "supercopa": {
        #         "country": "spain",
        #         "paths": {
        #             "ranking": "app/data/raw/spain/{last_season[0]}/rankings_supercup_{last_season[0]}.csv",
        #             "matches": "app/data/raw/spain/{last_season[0]}/matches_supercup_{last_season[0]}.csv",
        #             "trophies": "app/data/raw/spain/{last_season[0]}/trophies_{last_season[0]}.csv",
        #         },
        #         "scraping_urls": {
        #             "matches": "https://fbref.com/en/comps/646/{last_season[2]}/schedule/{last_season[2]}-Scores-and-Fixtures",
        #             "teams": "https://www.transfermarkt.com/xxxxxx/startseite/verein/{transfermarkt_id}/saison_id/{last_season[1]}"
        #         },
        #     },
        # ==================== ENGLISH COMPETITIONS ====================
        "premier_league": {
            "country": "england",
            "information_scraping_urls": {
                "matches": "https://fbref.com/en/comps/9/{last_season[0]}/schedule/{last_season[0]}-Premier-League-Scores-and-Fixtures",
                "fotmob": "https://www.fotmob.com/en/leagues/47/matches/premier-league?group=by-date&season={last_season[0]}&group=by-date&page={page}",
                # "teams": "https://www.transfermarkt.com/xxxxxx/startseite/verein/{transfermarkt_id}/saison_id/{last_season[1]}"
            },
            "betting_scraping_urls": {
                "betclic": "https://www.betclic.fr/football-sfootball/angl-premier-league-c3",
                "winamax": "https://www.winamax.fr/paris-sportifs/sports/1/1/1",
                "unibet": "https://www.unibet.fr/sport/football/angleterre/premier-league?filter=Top+Paris&subFilter=R%C3%A9sultat+du+match",
                "parionssport": "https://www.enligne.parionssport.fdj.fr/paris-football/angleterre/premier-league",
                "pmu": "https://parisportif.pmu.fr/home/wrapper/events?fId=1&activeSportId=1&leagues=%5B261%5D&boost=%5B%5D",
                "bwin": "https://www.bwin.fr/fr/sports/football-4/paris-sportifs/angleterre-14/premier-league-102841",
                "netbet": "https://www.netbet.fr/football/angleterre/premier-league",
                "betsson": "https://betsson.fr/fr/competitions/football/premier-league?competition_id=55768.1",
            },
        },
        #     "fa_cup": {
        #         "country": "england",
        #         "paths": {
        #             "ranking": "app/data/raw/england/{last_season[0]}/rankings_cup_{last_season[0]}.csv",
        #             "matches": "app/data/raw/england/{last_season[0]}/matches_cup_{last_season[0]}.csv",
        #             "trophies": "app/data/raw/england/{last_season[0]}/trophies_{last_season[0]}.csv",
        #         },
        #         "information_scraping_urls": {
        #             "matches": "https://fbref.com/en/comps/514/{last_season[0]}/schedule/{last_season[0]}-FA-Cup-Scores-and-Fixtures",
        #             "teams": "https://www.transfermarkt.com/xxxxxx/startseite/verein/{transfermarkt_id}/saison_id/{last_season[1]}"
        #         },
        #         "betting_scraping_urls": {
        #             #"betclic": "https://www.betclic.fr/football-sfootball/angleterre-fa-cup-c44",
        #             #"winamax": "https://www.winamax.fr/paris-sportifs/match/1000095499",
        #             #"unibet": "https://www.unibet.fr/sport/football/angleterre/fa-cup?filter=Comp%C3%A9tition&subFilter=Vainqueur+de+la+comp%C3%A9tition",
        #             #"parionssport": "https://www.enligne.parionssport.fdj.fr/paris-football/angleterre/fa-cup",
        #             #"netbet": "https://www.netbet.fr/football/angleterre/fa-cup",
        #             #"betsson": "https://betsson.fr/fr/competitions/football/fa-cup?competition_id=55765.1",
        #         },
        #     },
        #     "carabao_cup": {
        #         "country": "england",
        #         "paths": {
        #             "ranking": "app/data/raw/england/{last_season[0]}/rankings_carabao_{last_season[0]}.csv",
        #             "matches": "app/data/raw/england/{last_season[0]}/matches_carabao_{last_season[0]}.csv",
        #             "trophies": "app/data/raw/england/{last_season[0]}/trophies_{last_season[0]}.csv",
        #         },
        #         "scraping_urls": {
        #             "matches": "https://fbref.com/en/comps/690/{last_season[0]}/schedule/{last_season[0]}-EFL-Cup-Scores-and-Fixtures",
        #             "odds": "https://www.oddschecker.com/es/futbol/inglaterra/efl-cup",
        #         },
        #     },
        #     "community_shield": {
        #         "country": "england",
        #         "paths": {
        #             "ranking": "app/data/raw/england/{last_season[0]}/rankings_supercup_{last_season[0]}.csv",
        #             "matches": "app/data/raw/england/{last_season[0]}/matches_supercup_{last_season[0]}.csv",
        #             "trophies": "app/data/raw/england/{last_season[0]}/trophies_{last_season[0]}.csv",
        #         },
        #         "scraping_urls": {
        #             "matches": "https://fbref.com/en/comps/602/2024/2024-Stats", #Aqui hay que cambiar como se pone last_season[0]
        #         },
        #     },
        #     # ==================== ITALIAN COMPETITIONS ====================
        #     "seria_a": {
        #         "country": "italy",
        #         "paths": {
        #             "ranking": "app/data/raw/italy/{last_season[0]}/rankings_league_{last_season[0]}.csv",
        #             "matches": "app/data/raw/italy/{last_season[0]}/matches_league_{last_season[0]}.csv",
        #             "teams": "app/data/raw/italy/{last_season[0]}/teams_{last_season[0]}.csv",
        #             "trophies": "app/data/raw/italy/{last_season[0]}/trophies_{last_season[0]}.csv",
        #         },
        #         "scraping_urls": {
        #             "matches": "https://fbref.com/en/comps/11/{last_season[0]}/schedule/{last_season[0]}-Serie-A-Scores-and-Fixtures",
        #             "odds": "https://www.oddschecker.com/es/futbol/italia/serie-a",
        #         },
        #     },
        #     "coppa": {
        #         "country": "italy",
        #         "paths": {
        #             "ranking": "app/data/raw/italy/{last_season[0]}/rankings_cup_{last_season[0]}.csv",
        #             "matches": "app/data/raw/italy/{last_season[0]}/matches_cup_{last_season[0]}.csv",
        #             "trophies": "app/data/raw/italy/{last_season[0]}/trophies_{last_season[0]}.csv",
        #         },
        #         "scraping_urls": {
        #             "matches": "https://fbref.com/en/comps/529/{last_season[0]}/schedule/{last_season[0]}-Coppa-Italia-Scores-and-Fixtures",
        #             "odds": "https://www.oddsportal.com/football/italy/coppa-italia/",
        #         },
        #     },
        #     "supercopa": {
        #         "country": "italy",
        #         "paths": {
        #             "ranking": "app/data/raw/italy/{last_season[0]}/rankings_supercup_{last_season[0]}.csv",
        #             "matches": "app/data/raw/italy/{last_season[0]}/matches_supercup_{last_season[0]}.csv",
        #             "trophies": "app/data/raw/italy/{last_season[0]}/trophies_{last_season[0]}.csv",
        #         },
        #         "scraping_urls": {
        #             "matches": "https://fbref.com/en/comps/612/2025/schedule/2025-Scores-and-Fixtures", #Aqui hay que cambiar como se pone last_season[0]
        #         },
        #     },
        #     # ==================== FRENCH COMPETITIONS ====================
        #     "ligue_1": {
        #         "country": "France",
        #         "paths": {
        #             "ranking": BASE_DATA_DIR / "french_league" / "ranking.csv",
        #             "matches": BASE_DATA_DIR / "french_league" / "matches.csv",
        #             "teams": BASE_DATA_DIR / "french_league" / "teams.csv",
        #             "trophies": BASE_DATA_DIR / "french_league" / "trophies.csv",
        #         },
        #         "scraping_urls": {
        #             "matches": "https://fbref.com/en/comps/13/Ligue-1-Stats",
        #             "odds": "https://www.oddsportal.com/football/france/ligue-1/",
        #             "ranking": "https://fbref.com/en/comps/13/Ligue-1-Stats",
        #         },
        #     },
        #     "coupe_de_france": {
        #         "country": "France",
        #         "paths": {
        #             "ranking": BASE_DATA_DIR / "french_cup" / "ranking.csv",
        #             "matches": BASE_DATA_DIR / "french_cup" / "matches.csv",
        #             "teams": BASE_DATA_DIR / "french_cup" / "teams.csv",
        #             "trophies": BASE_DATA_DIR / "french_cup" / "trophies.csv",
        #         },
        #         "scraping_urls": {
        #             "matches": "https://fbref.com/en/comps/69/Coupe-de-France-Stats",
        #             "odds": "https://www.oddsportal.com/football/france/coupe-de-france/",
        #             "ranking": None,
        #         },
        #     },
        #     "trophee_des_champions": {
        #         "country": "France",
        #         "paths": {
        #             "ranking": BASE_DATA_DIR / "french_supercup" / "ranking.csv",
        #             "matches": BASE_DATA_DIR / "french_supercup" / "matches.csv",
        #             "teams": BASE_DATA_DIR / "french_supercup" / "teams.csv",
        #             "trophies": BASE_DATA_DIR / "french_supercup" / "trophies.csv",
        #         },
        #         "scraping_urls": {
        #             "matches": None,
        #             "odds": None,
        #             "ranking": None,
        #         },
        #     },
        #     "coupe_de_la_ligue": {
        #         "country": "France",
        #         "paths": {
        #             "ranking": BASE_DATA_DIR / "french_coupe_ligue" / "ranking.csv",
        #             "matches": BASE_DATA_DIR / "french_coupe_ligue" / "matches.csv",
        #             "teams": BASE_DATA_DIR / "french_coupe_ligue" / "teams.csv",
        #             "trophies": BASE_DATA_DIR / "french_coupe_ligue" / "trophies.csv",
        #         },
        #         "scraping_urls": {
        #             "matches": None,
        #             "odds": None,
        #             "ranking": None,
        #         },
        #     },
        #     # ==================== GERMAN COMPETITIONS ====================
        #     "bundesliga": {
        #         "country": "Germany",
        #         "paths": {
        #             "ranking": "app/data/raw/german/{last_season[0]}/rankings_league_{last_season[0]}.csv",
        #             "matches": "app/data/raw/german/{last_season[0]}/matches_league_{last_season[0]}.csv",
        #             "teams": "app/data/raw/german/{last_season[0]}/teams_{last_season[0]}.csv",
        #             "trophies": "app/data/raw/german/{last_season[0]}/trophies_{last_season[0]}.csv",
        #         },
        #         "scraping_urls": {
        #             "matches": "https://fbref.com/en/comps/20/{last_season[0]}/schedule/{last_season[0]}-Bundesliga-Scores-and-Fixtures",
        #             "odds": "https://www.oddschecker.com/es/futbol/alemania/bundesliga",
        #         },
        #     },
        #     "dfb_pokal": {
        #         "country": "Germany",
        #         "paths": {
        #             "ranking": "app/data/raw/german/{last_season[0]}/rankings_cup_{last_season[0]}.csv",
        #             "matches": "app/data/raw/german/{last_season[0]}/matches_cup_{last_season[0]}.csv",
        #             "trophies": "app/data/raw/german/{last_season[0]}/trophies_{last_season[0]}.csv",
        #         },
        #         "scraping_urls": {
        #             "matches": "https://fbref.com/en/comps/521/{last_season[0]}/schedule/{last_season[0]}-DFB-Pokal-Scores-and-Fixtures",
        #             "odds": "https://www.oddsportal.com/football/germany/dfb-pokal/", # No existe mirar otros
        #         },
        #     },
        #     "dfl_supercup": {
        #         "country": "Germany",
        #         "paths": {
        #             "ranking": "app/data/raw/german/{last_season[0]}/rankings_supercup_{last_season[0]}.csv",
        #             "matches": "app/data/raw/german/{last_season[0]}/matches_supercup_{last_season[0]}.csv",
        #             "trophies": "app/data/raw/german/{last_season[0]}/trophies_{last_season[0]}.csv",
        #         },
        #         "scraping_urls": {
        #             "matches": "https://fbref.com/en/comps/606/2025/2025-Stats", #Aqui hay que cambiar como se pone last_season[0]
        #         },
        #     },
        #     # ==================== EUROPEAN COMPETITIONS ====================
        #     "uefa_champions_league": {
        #         "country": "Europe",
        #         "paths": {
        #             "ranking": BASE_DATA_DIR / "champions_league" / "ranking.csv",
        #             "matches": BASE_DATA_DIR / "champions_league" / "matches.csv",
        #             "teams": BASE_DATA_DIR / "champions_league" / "teams.csv",
        #             "trophies": BASE_DATA_DIR / "champions_league" / "trophies.csv",
        #         },
        #         "scraping_urls": {
        #             "matches": "https://fbref.com/en/comps/8/Champions-League-Stats",
        #             "odds": "https://www.oddsportal.com/football/europe/champions-league/",
        #             "ranking": None,
        #         },
        #     },
        #     "uefa_europa_league": {
        #         "country": "Europe",
        #         "paths": {
        #             "ranking": BASE_DATA_DIR / "europa_league" / "ranking.csv",
        #             "matches": BASE_DATA_DIR / "europa_league" / "matches.csv",
        #             "teams": BASE_DATA_DIR / "europa_league" / "teams.csv",
        #             "trophies": BASE_DATA_DIR / "europa_league" / "trophies.csv",
        #         },
        #         "scraping_urls": {
        #             "matches": "https://fbref.com/en/comps/19/Europa-League-Stats",
        #             "odds": "https://www.oddsportal.com/football/europe/europa-league/",
        #             "ranking": None,
        #         },
        #     },
        #     "uefa_conference_league": {
        #         "country": "Europe",
        #         "paths": {
        #             "ranking": BASE_DATA_DIR / "conference_league" / "ranking.csv",
        #             "matches": BASE_DATA_DIR / "conference_league" / "matches.csv",
        #             "teams": BASE_DATA_DIR / "conference_league" / "teams.csv",
        #             "trophies": BASE_DATA_DIR / "conference_league" / "trophies.csv",
        #         },
        #         "scraping_urls": {
        #             "matches": "https://fbref.com/en/comps/882/Europa-Conference-League-Stats",
        #             "odds": "https://www.oddsportal.com/football/europe/europa-conference-league/",
        #             "ranking": None,
        #         },
        #     },
        #     "uefa_supercup": {
        #         "country": "Europe",
        #         "paths": {
        #             "ranking": BASE_DATA_DIR / "uefa_supercup" / "ranking.csv",
        #             "matches": BASE_DATA_DIR / "uefa_supercup" / "matches.csv",
        #             "teams": BASE_DATA_DIR / "uefa_supercup" / "teams.csv",
        #             "trophies": BASE_DATA_DIR / "uefa_supercup" / "trophies.csv",
        #         },
        #         "scraping_urls": {
        #             "matches": None,
        #             "odds": None,
        #             "ranking": None,
        #         },
        #    },
        #     # ==================== INTERNATIONAL COMPETITIONS ====================
        #     "fifa_world_cup": {
        #         "country": "International",
        #         "paths": {
        #             "ranking": BASE_DATA_DIR / "world_cup" / "ranking.csv",
        #             "matches": BASE_DATA_DIR / "world_cup" / "matches.csv",
        #             "teams": BASE_DATA_DIR / "world_cup" / "teams.csv",
        #             "trophies": BASE_DATA_DIR / "world_cup" / "trophies.csv",
        #         },
        #         "scraping_urls": {
        #             "matches": "https://fbref.com/en/comps/1/World-Cup-Stats",
        #             "odds": "https://www.oddsportal.com/football/world/world-cup/",
        #             "ranking": None,
        #         },
        #     },
    }

    # TEAMS_CONFIG = {
    #     "Real Madrid": {
    #         # --- Names & IDs ---
    #         "name": "Real_Madrid",
    #         "abbrev": "RMA",
    #         "scraping_names": ["Real Madrid CF", "Real_Madrid"], #Ver cómo se pueden llamar por todos los scraping
    #         "scraping_id": 5,

    #         # --- Geography & logistics ---
    #         "country": "Spain",
    #         "city": "Madrid",
    #         "timezone": "Europe/Madrid",
    #         "geo": {"lat": 40.4165, "lon": -3.70256, "ele": 657},

    #         # --- Branding & history ---
    #         "colors": "white",
    #         "founded_year": 1902,
    #         "ownership_type": "member-owned",
    #         "budget_tier": "high",

    #         # --- Rivalries ---
    #         "rivalries": ["Barcelona", "Atletico_Madrid"],

    #         # --- Stadiums ---
    #         "stadiums": [
    #             {
    #                 "scraping_name": "Santiago Bernabéu", # Mirar
    #                 "capacity": 81044,
    #                 "pitch": {"length_m": 105, "width_m": 68, "surface": "grass"},
    #             },
    #         ],
    #     },
    #     "FC Barcelona": {
    #         "scraping_name": "Barcelona",
    #         "odds_name": "Barcelona",
    #         "country": "Spain",
    #         "stadiums": [
    #             {
    #                 "name": "Camp Nou",
    #                 "scraping_name": "Camp Nou",
    #                 "capacity": 99354,
    #                 "city": "Barcelona",
    #             },
    #             {
    #                 "name": "Estadi Olímpic Lluís Companys",
    #                 "scraping_name": "Estadi Olímpic Lluís Companys",
    #                 "capacity": 54367,
    #                 "city": "Barcelona",
    #             }
    #         ],
    #     },
    #     "Atlético Madrid": {
    #         "scraping_name": "Atlético Madrid",
    #         "odds_name": "Atletico Madrid",
    #         "country": "Spain",
    #         "stadiums": [
    #             {
    #                 "name": "Cívitas Metropolitano",
    #                 "scraping_name": "Cívitas Metropolitano",
    #                 "capacity": 68456,
    #                 "city": "Madrid",
    #             }
    #         ],
    #     },
    #     # Add more teams following the same structure...
    # }


@lru_cache(maxsize=1)
def get_settings() -> Settings:
    """Return a cached instance of Settings."""
    return Settings()
