import pandas as pd
import numpy as np


class LeagueRanking:
    """
    Class to fetch team rankings from a rankings CSV file based on either:
    - The gameweek of recent matches (if more than 5 matches available)
    - The last match of each team in the competition (if less than 5 matches)
    """
    
    def __init__(self):
        """Initialize the TeamRankingFetcher"""
        pass
    
    def get_team_rankings(
        self,
        ranking_csv_path: str,
        recent_matches_df: pd.DataFrame,
        historical_matches_csv_path: str
    ) -> pd.DataFrame:
        """
        Add team ranking columns to the recent matches DataFrame.
        
        Args:
            ranking_csv_path: Path to the CSV file containing team rankings by gameweek
            recent_matches_df: DataFrame containing recent matches
            historical_matches_csv_path: Path to CSV file with all historical matches
            
        Returns:
            DataFrame with added columns:
            - home_team_rank
            - away_team_rank
            - home_team_points
            - away_team_points
            - home_team_goals_for
            - away_team_goals_for
            - home_team_goals_against
            - away_team_goals_against
            - home_team_goals_difference
            - away_team_goals_difference
        """
        print(f"[DEBUG] get_team_rankings - START")
        print(f"[DEBUG] recent_matches_df shape: {recent_matches_df.shape}")
        
        # Make a copy to avoid modifying the original DataFrame
        df_updated = recent_matches_df.copy()

        print(df_updated.head())
        
        # Load the rankings CSV
        print(f"[DEBUG] Loading rankings from: {ranking_csv_path}")
        rankings_df = pd.read_csv(ranking_csv_path)
        print(f"[DEBUG] Rankings loaded - shape: {rankings_df.shape}")
        
        # Check if there are more than 5 matches in the recent matches DataFrame
        if len(recent_matches_df) > 5:
            print(f"[DEBUG] More than 5 matches ({len(recent_matches_df)}) - using gameweek method")
            df_updated = self._add_rankings_by_gameweek(
                df_updated, 
                rankings_df
            )
        else:
            print(f"[DEBUG] Less than 5 matches ({len(recent_matches_df)}) - using last match method")
            df_updated = self._add_rankings_from_last_match(
                df_updated,
                historical_matches_csv_path,
                rankings_df
            )
        
        print(f"[DEBUG] get_team_rankings - END")
        return df_updated
    
    def _add_rankings_by_gameweek(
        self,
        df: pd.DataFrame,
        rankings_df: pd.DataFrame
    ) -> pd.DataFrame:
        """
        Add ranking columns by identifying the gameweek from each match.
        
        Assumes df has columns: 'gameweek', 'home_team', 'away_team'
        (adjust column names as needed for your actual data structure)
        """
        print(f"[DEBUG] _add_rankings_by_gameweek - START")
        
        # Identify column names
        if 'gameweek' in df.columns:
            gameweek_col = 'gameweek'
        else:
            raise ValueError("DataFrame must contain 'gameweek' or 'round' column")
        
        home_team_col = 'home_team' if 'home_team' in df.columns else ('home_team_name' if 'home_team_name' in df.columns else 'Home')
        away_team_col = 'away_team' if 'away_team' in df.columns else ('away_team_name' if 'away_team_name' in df.columns else 'Away')
        
        print(f"[DEBUG] Columns: gameweek='{gameweek_col}', home='{home_team_col}', away='{away_team_col}'")
        
        # Initialize new columns with None
        df['home_team_rank'] = None
        df['away_team_rank'] = None
        df['home_team_points'] = None
        df['away_team_points'] = None
        df['home_team_goals_for'] = None
        df['away_team_goals_for'] = None
        df['home_team_goals_against'] = None
        df['away_team_goals_against'] = None
        df['home_team_goals_difference'] = None
        df['away_team_goals_difference'] = None
        
        print(f"[DEBUG] Processing {len(df)} matches...")
        
        # Iterate over each match and add ranking information
        for idx, match in df.iterrows():
            print(f"[DEBUG] Match {idx} - START")
            
            gameweek = match[gameweek_col]
            print(f"[DEBUG]   gameweek: {gameweek}")
            
            home_team = match[home_team_col]
            print(f"[DEBUG]   home_team: {home_team}")
            
            away_team = match[away_team_col]
            print(f"[DEBUG]   away_team: {away_team}")
            
            # Filter rankings for the gameweek
            gameweek_rankings = rankings_df[rankings_df['gameweek'] == gameweek]
            print(f"[DEBUG]   Rankings found for GW {gameweek}: {len(gameweek_rankings)}")
            
            # Get home team ranking
            home_ranking = gameweek_rankings[gameweek_rankings['team_name'] == home_team]
            # Get away team ranking
            away_ranking = gameweek_rankings[gameweek_rankings['team_name'] == away_team]
            
            print(f"[DEBUG]   Home team '{home_team}' found: {not home_ranking.empty}")
            print(f"[DEBUG]   Away team '{away_team}' found: {not away_ranking.empty}")
            
            # If team found, add the data
            if not home_ranking.empty:
                home_ranking = home_ranking.iloc[0]
                df.at[idx, 'home_team_rank'] = int(home_ranking['team_rank'])
                df.at[idx, 'home_team_points'] = int(home_ranking['team_points'])
                df.at[idx, 'home_team_goals_for'] = int(home_ranking['team_goals_for'])
                df.at[idx, 'home_team_goals_against'] = int(home_ranking['team_goals_against'])
                df.at[idx, 'home_team_goals_difference'] = int(home_ranking['team_goals_difference'])
            
            if not away_ranking.empty:
                away_ranking = away_ranking.iloc[0]
                df.at[idx, 'away_team_rank'] = int(away_ranking['team_rank'])
                df.at[idx, 'away_team_points'] = int(away_ranking['team_points'])
                df.at[idx, 'away_team_goals_for'] = int(away_ranking['team_goals_for'])
                df.at[idx, 'away_team_goals_against'] = int(away_ranking['team_goals_against'])
                df.at[idx, 'away_team_goals_difference'] = int(away_ranking['team_goals_difference'])
        
        print(f"[DEBUG] _add_rankings_by_gameweek - END")
        return df
    
    def _add_rankings_from_last_match(
        self,
        df: pd.DataFrame,
        historical_matches_csv_path: str,
        rankings_df: pd.DataFrame
    ) -> pd.DataFrame:
        """
        Add ranking columns from the last historical match of each team in the competition.
        
        Assumes historical matches CSV has columns like: 'date_of_match', 'gameweek', 'competition',
        'home_team', 'away_team', etc.
        """
        print(f"[DEBUG] _add_rankings_from_last_match - START")
        
        # Load historical matches
        print(f"[DEBUG] Loading historical matches from: {historical_matches_csv_path}")
        historical_df = pd.read_csv(historical_matches_csv_path)
        print(f"[DEBUG] Historical matches loaded - shape: {historical_df.shape}")
        
        # Get team column names
        home_team_col = 'home_team' if 'home_team' in df.columns else ('home_team_name' if 'home_team_name' in df.columns else 'Home')
        away_team_col = 'away_team' if 'away_team' in df.columns else ('away_team_name' if 'away_team_name' in df.columns else 'Away')
        
        # Get team column names for historical_df (may be different)
        hist_home_col = 'home_team' if 'home_team' in historical_df.columns else ('home_team_name' if 'home_team_name' in historical_df.columns else 'Home')
        hist_away_col = 'away_team' if 'away_team' in historical_df.columns else ('away_team_name' if 'away_team_name' in historical_df.columns else 'Away')
        
        # Identify the competition column
        competition_col = 'competition' if 'competition' in df.columns else 'Competition'
        
        print(f"[DEBUG] Columns: home='{home_team_col}', away='{away_team_col}', comp='{competition_col}'")
        print(f"[DEBUG] Historical columns: home='{hist_home_col}', away='{hist_away_col}'")
        
        # Ensure historical_df has date column sorted
        if 'date_of_match' in historical_df.columns:
            print(f"[DEBUG] Sorting historical matches by date...")
            historical_df['date_of_match'] = pd.to_datetime(historical_df['date_of_match'])
            historical_df = historical_df.sort_values('date_of_match', ascending=False)
        
        # Initialize new columns with None
        df['home_team_rank'] = None
        df['away_team_rank'] = None
        df['home_team_points'] = None
        df['away_team_points'] = None
        df['home_team_goals_for'] = None
        df['away_team_goals_for'] = None
        df['home_team_goals_against'] = None
        df['away_team_goals_against'] = None
        df['home_team_goals_difference'] = None
        df['away_team_goals_difference'] = None
        
        print(f"[DEBUG] Processing {len(df)} matches...")
        
        # Iterate over each match
        for idx, match in df.iterrows():
            home_team = match[home_team_col]
            away_team = match[away_team_col]
            
            print(f"[DEBUG] Match {idx}: {home_team} vs {away_team}")
            
            # Filter historical matches by competition if available
            hist_df_filtered = historical_df.copy()
            if competition_col in match.index and competition_col in historical_df.columns:
                competition = match[competition_col]
                print(f"[DEBUG]   Filtering by competition: {competition}")
                hist_df_filtered = hist_df_filtered[hist_df_filtered[competition_col] == competition]
                print(f"[DEBUG]   Filtered historical matches: {len(hist_df_filtered)}")
            
            # Find last match for home team
            home_team_matches = hist_df_filtered[
                (hist_df_filtered[hist_home_col] == home_team) | 
                (hist_df_filtered[hist_away_col] == home_team)
            ]
            print(f"[DEBUG]   Home team '{home_team}' historical matches: {len(home_team_matches)}")
            
            # Find last match for away team
            away_team_matches = hist_df_filtered[
                (hist_df_filtered[hist_home_col] == away_team) | 
                (hist_df_filtered[hist_away_col] == away_team)
            ]
            print(f"[DEBUG]   Away team '{away_team}' historical matches: {len(away_team_matches)}")
            
            # Extract gameweek column name
            gameweek_col = 'gameweek' if 'gameweek' in historical_df.columns else 'round'
            
            # Get rankings for home team from their last match
            if not home_team_matches.empty:
                last_home_match = home_team_matches.iloc[0]
                home_last_gameweek = last_home_match[gameweek_col]
                print(f"[DEBUG]   Home team last gameweek: {home_last_gameweek}")
                
                home_ranking = rankings_df[
                    (rankings_df['gameweek'] == home_last_gameweek) &
                    (rankings_df['team_name'] == home_team)
                ]
                print(f"[DEBUG]   Home team ranking found: {not home_ranking.empty}")
                
                if not home_ranking.empty:
                    home_ranking = home_ranking.iloc[0]
                    df.at[idx, 'home_team_rank'] = int(home_ranking['team_rank'])
                    df.at[idx, 'home_team_points'] = int(home_ranking['team_points'])
                    df.at[idx, 'home_team_goals_for'] = int(home_ranking['team_goals_for'])
                    df.at[idx, 'home_team_goals_against'] = int(home_ranking['team_goals_against'])
                    df.at[idx, 'home_team_goals_difference'] = int(home_ranking['team_goals_difference'])
            
            # Get rankings for away team from their last match
            if not away_team_matches.empty:
                last_away_match = away_team_matches.iloc[0]
                away_last_gameweek = last_away_match[gameweek_col]
                print(f"[DEBUG]   Away team last gameweek: {away_last_gameweek}")
                
                away_ranking = rankings_df[
                    (rankings_df['gameweek'] == away_last_gameweek) &
                    (rankings_df['team_name'] == away_team)
                ]
                print(f"[DEBUG]   Away team ranking found: {not away_ranking.empty}")
                
                if not away_ranking.empty:
                    away_ranking = away_ranking.iloc[0]
                    df.at[idx, 'away_team_rank'] = int(away_ranking['team_rank'])
                    df.at[idx, 'away_team_points'] = int(away_ranking['team_points'])
                    df.at[idx, 'away_team_goals_for'] = int(away_ranking['team_goals_for'])
                    df.at[idx, 'away_team_goals_against'] = int(away_ranking['team_goals_against'])
                    df.at[idx, 'away_team_goals_difference'] = int(away_ranking['team_goals_difference'])
        
        print(f"[DEBUG] _add_rankings_from_last_match - END")
        return df
    
    def get_last_match_stats(self, df_matches, past_matches):
        """
        For each match in df_matches, finds the last match played by each team
        in any non-UEFA competition (regardless of opponent or home/away status).
        """
        
        # Columns to extract
        stats_columns = [
            'team_rank',
            'matchs_played',
            'matchs_won',
            'matchs_drawn',
            'matchs_lost',
            'team_goals_for',
            'team_goals_against',
            'team_goals_difference',
            'team_points'
        ]
        
        # Initialize columns in df_matches
        for prefix in ['home', 'away']:
            for col in stats_columns:
                df_matches[f'{prefix}_{col}'] = np.nan
        
        # Process each match
        for idx, row in df_matches.iterrows():
            home_team = row['home_team_name']
            away_team = row['away_team_name']
            match_date = row['date_of_match']
            
            # Find last match for home team in any non-UEFA competition
            home_last_match = past_matches[
                (
                    ((past_matches['home_team_name'] == home_team) | 
                    (past_matches['away_team_name'] == home_team))
                ) &
                (past_matches['date_of_match'] < match_date) &
                (~past_matches['competition'].str.contains('uefa', case=False, na=False))  # Exclude UEFA matches
            ].sort_values('date_of_match', ascending=False)
            
            if not home_last_match.empty:
                last_match = home_last_match.iloc[0]
                # Determine if they played as home or away
                if last_match['home_team_name'] == home_team:
                    prefix_source = 'home'
                else:
                    prefix_source = 'away'
                
                # Copy statistics
                for col in stats_columns:
                    df_matches.at[idx, f'home_{col}'] = last_match[f'{prefix_source}_{col}']
            
            # Find last match for away team in any non-UEFA competition
            away_last_match = past_matches[
                (
                    ((past_matches['home_team_name'] == away_team) | 
                    (past_matches['away_team_name'] == away_team))
                ) &
                (past_matches['date_of_match'] < match_date) &
                (~past_matches['competition'].str.contains('uefa', case=False, na=False))  # Exclude UEFA matches
            ].sort_values('date_of_match', ascending=False)
            
            if not away_last_match.empty:
                last_match = away_last_match.iloc[0]
                # Determine if they played as home or away
                if last_match['home_team_name'] == away_team:
                    prefix_source = 'home'
                else:
                    prefix_source = 'away'
                
                # Copy statistics
                for col in stats_columns:
                    df_matches.at[idx, f'away_{col}'] = last_match[f'{prefix_source}_{col}']
        
        return df_matches