"""
renderers/apibb-python/src/index.py
APIBB Python Renderer - Terminal/CLI Implementation
Follows the 7-property contract with reactive state management
"""

import re
import json
import zipfile
import requests
from typing import Any, Dict, List, Set, Callable, Optional
from io import BytesIO


# ──────────────────────────────────────────────────────────────
# 1. Global Store + {{path}} templating
# ──────────────────────────────────────────────────────────────
class Store:
    """Global reactive state store with {{path}} template resolution."""

    def __init__(self):
        self.state: Dict[str, Any] = {}
        self.listeners: Dict[str, Set[Callable]] = {}

    def set(self, path: str, value: Any) -> None:
        """Set a value in the store and notify listeners."""
        if self.state.get(path) == value:
            return
        self.state[path] = value

        # Notify specific path listeners
        if path in self.listeners:
            for callback in self.listeners[path]:
                callback()

        # Notify wildcard listeners
        if '*' in self.listeners:
            for callback in self.listeners['*']:
                callback()

    def get(self, path: str) -> Any:
        """Get a value from the store."""
        return self.state.get(path)

    def subscribe(self, path: str, callback: Callable) -> Callable:
        """Subscribe to changes on a path."""
        if path not in self.listeners:
            self.listeners[path] = set()
        self.listeners[path].add(callback)

        def unsubscribe():
            if path in self.listeners:
                self.listeners[path].discard(callback)

        return unsubscribe

    def resolve(self, value: Any) -> Any:
        """Resolve {{path}} templates in strings."""
        if not isinstance(value, str):
            return value

        def replacer(match):
            path = match.group(1).strip()
            val = self.get(path)
            return '' if val is None else str(val)

        return re.sub(r'\{\{([^}]+)\}\}', replacer, value)


# ──────────────────────────────────────────────────────────────
# 2. Built-in core components
# ──────────────────────────────────────────────────────────────
class Component:
    """Base component class."""

    def __init__(self, node: Dict, path: str, store: Store):
        self.node = node
        self.path = path
        self.store = store

    def render(self) -> str:
        """Render the component to a string."""
        return ""


class TextComponent(Component):
    """Text display component."""

    def render(self) -> str:
        data = self.store.resolve(self.node.get('data', ''))
        return str(data)


class ButtonComponent(Component):
    """Button component (CLI representation)."""

    def render(self) -> str:
        data = self.store.resolve(self.node.get('data', 'Button'))
        return f"[{data}]"


class InputComponent(Component):
    """Input component (CLI representation)."""

    def render(self) -> str:
        value = self.store.get(self.path) or ''
        placeholder = self.node.get('data', '')
        if value:
            return f"<{value}>"
        return f"<{placeholder}...>"


class ContainerComponent(Component):
    """Container component."""

    def render(self) -> str:
        children = self.node.get('children', [])
        rendered = []
        for i, child in enumerate(children):
            child_path = f"{self.path}.children[{i}]"
            rendered.append(render_node(child, child_path, self.store))
        return '\n'.join(rendered)


class ListComponent(Component):
    """List component with item templating."""

    def render(self) -> str:
        data = self.node.get('data', [])
        children = self.node.get('children', [])

        if not children or not data:
            return ""

        template = children[0]
        rendered = []

        for i, item in enumerate(data):
            # Store item for template access
            self.store.set(f'{self.path}.item', item)
            child_path = f"{self.path}.children[{i}]"
            rendered.append(render_node(template, child_path, self.store))

        return '\n'.join(rendered)


class ConditionalComponent(Component):
    """Conditional rendering component."""

    def render(self) -> str:
        condition = self.store.resolve(self.node.get('data', False))
        if condition:
            children = self.node.get('children', [])
            rendered = []
            for i, child in enumerate(children):
                child_path = f"{self.path}.children[{i}]"
                rendered.append(render_node(child, child_path, self.store))
            return '\n'.join(rendered)
        return ""


# Built-in components registry
BUILTINS = {
    'Text': TextComponent,
    'Button': ButtonComponent,
    'Input': InputComponent,
    'Container': ContainerComponent,
    'List': ListComponent,
    'Conditional': ConditionalComponent,
    'HTTPRequest': lambda *args: None,  # No-op in CLI
    'Navigation': lambda *args: None,   # No-op in CLI
    'LocalState': lambda *args: None,   # No-op in CLI
    'TextOutput': TextComponent,        # Alias for Text
}


# ──────────────────────────────────────────────────────────────
# 3. Dynamic component loader
# ──────────────────────────────────────────────────────────────
component_cache: Dict[str, type] = {}

# Component base URL - loads from GitHub raw content
COMPONENT_BASE = 'https://raw.githubusercontent.com/Devignite25/apibb/main/v1/components'


def load_component(component_type: str) -> Optional[type]:
    """Load a component class by type name."""
    # Check cache
    if component_type in component_cache:
        return component_cache[component_type]

    # Check builtins
    if component_type in BUILTINS:
        component_cache[component_type] = BUILTINS[component_type]
        return BUILTINS[component_type]

    # Try to load from GitHub
    try:
        url = f"{COMPONENT_BASE}/{component_type}/package.zip"
        response = requests.get(url, timeout=5)
        response.raise_for_status()

        # Extract and load component from zip
        with zipfile.ZipFile(BytesIO(response.content)) as zf:
            # For now, just cache as TextComponent
            # In production, would parse schema.json and create dynamic component
            component_cache[component_type] = TextComponent
            return TextComponent

    except Exception as e:
        print(f"Warning: Could not load component {component_type}: {e}")
        # Return a missing component placeholder
        class MissingComponent(Component):
            def render(self):
                return f"[Missing: {component_type}]"

        component_cache[component_type] = MissingComponent
        return MissingComponent


# ──────────────────────────────────────────────────────────────
# 4. Node renderer
# ──────────────────────────────────────────────────────────────
def render_node(node: Dict, path: str, store: Store) -> str:
    """Render a single node."""
    if not isinstance(node, dict):
        return str(node)

    node_type = node.get('type')
    if not node_type:
        return ""

    # Load component
    ComponentClass = load_component(node_type)
    if not ComponentClass:
        return f"[Error: Unknown component {node_type}]"

    # Create and render component
    try:
        component = ComponentClass(node, path, store)
        return component.render()
    except Exception as e:
        return f"[Error rendering {node_type}: {e}]"


# ──────────────────────────────────────────────────────────────
# 5. Public Renderer
# ──────────────────────────────────────────────────────────────
class ApibbRenderer:
    """Main APIBB renderer for Python/CLI."""

    def __init__(self, app: Dict):
        self.app = app
        self.store = Store()

        # Initialize state
        initial_state = app.get('initialState', {})
        for key, value in initial_state.items():
            self.store.set(key, value)

    def render(self) -> str:
        """Render the app to a string."""
        if not self.app or 'type' not in self.app:
            return "Error: No valid APIBB app"

        return render_node(self.app, 'root', self.store)

    def update(self, path: str, value: Any) -> None:
        """Update a value in the store."""
        self.store.set(path, value)

    def get(self, path: str) -> Any:
        """Get a value from the store."""
        return self.store.get(path)


# ──────────────────────────────────────────────────────────────
# 6. CLI entry point
# ──────────────────────────────────────────────────────────────
def main():
    """CLI entry point for testing."""
    import sys

    if len(sys.argv) < 2:
        print("Usage: python index.py <app.json>")
        sys.exit(1)

    app_path = sys.argv[1]

    try:
        with open(app_path, 'r') as f:
            app = json.load(f)

        renderer = ApibbRenderer(app)
        output = renderer.render()
        print(output)

    except FileNotFoundError:
        print(f"Error: App file not found: {app_path}")
        sys.exit(1)
    except json.JSONDecodeError as e:
        print(f"Error: Invalid JSON: {e}")
        sys.exit(1)
    except Exception as e:
        print(f"Error: {e}")
        sys.exit(1)


if __name__ == '__main__':
    main()
