# APIBB Python Renderer

**Frozen v1.0 - 25 November 2025**

Python/CLI renderer for APIBB applications. Renders APIBB JSON apps to terminal output following the 7-property contract.

## Features

- ✅ Full 7-property contract support (`type`, `data`, `value`, `onChange`, `trigger`, `children`, `style`)
- ✅ Reactive state store with `{{path}}` templating
- ✅ Dynamic component loading from CDN
- ✅ Built-in core components (Text, Button, Input, Container, List, Conditional)
- ✅ Terminal/CLI rendering
- ✅ Zero dependencies (except `requests` for component loading)

## Installation

```bash
cd renderers/apibb-python
pip install -e .
```

Or install requirements directly:

```bash
pip install -r requirements.txt
```

## Usage

### Command Line

```bash
python src/index.py path/to/app.json
```

### Python API

```python
from src.index import ApibbRenderer

# Load your app
app = {
    "type": "Container",
    "children": [
        {
            "type": "Text",
            "data": "Hello from APIBB Python!"
        },
        {
            "type": "Button",
            "data": "Click Me"
        }
    ],
    "initialState": {
        "counter": 0
    }
}

# Create renderer
renderer = ApibbRenderer(app)

# Render to string
output = renderer.render()
print(output)

# Update state
renderer.update('counter', 1)

# Re-render
output = renderer.render()
print(output)
```

## Built-in Components

### Text
```python
{
    "type": "Text",
    "data": "Hello, {{user.name}}!"
}
```

### Button
```python
{
    "type": "Button",
    "data": "Click Me"
}
```
Renders as: `[Click Me]`

### Input
```python
{
    "type": "Input",
    "data": "Enter your name",
    "value": "{{user.name}}"
}
```
Renders as: `<John>` or `<Enter your name...>`

### Container
```python
{
    "type": "Container",
    "children": [
        {"type": "Text", "data": "Child 1"},
        {"type": "Text", "data": "Child 2"}
    ]
}
```

### List
```python
{
    "type": "List",
    "data": ["Apple", "Banana", "Cherry"],
    "children": [
        {"type": "Text", "data": "{{item}}"}
    ]
}
```

### Conditional
```python
{
    "type": "Conditional",
    "data": "{{showContent}}",
    "children": [
        {"type": "Text", "data": "This shows when true"}
    ]
}
```

## Store & Templating

The renderer includes a reactive store that supports `{{path}}` templating:

```python
renderer = ApibbRenderer(app)

# Set values
renderer.update('user.name', 'Alice')
renderer.update('counter', 5)

# Get values
name = renderer.get('user.name')  # 'Alice'

# Templates are resolved automatically during render
# "Hello, {{user.name}}!" becomes "Hello, Alice!"
```

## Dynamic Component Loading

Components not in the built-in set are automatically loaded from GitHub:

```
https://raw.githubusercontent.com/Devignite25/apibb/main/v1/components/{ComponentName}/package.zip
```

The package is downloaded, verified, and cached automatically.

## Example Apps

### Counter App

```python
counter_app = {
    "type": "Container",
    "children": [
        {
            "type": "Text",
            "data": "Count: {{counter}}"
        },
        {
            "type": "Button",
            "data": "Increment"
        }
    ],
    "initialState": {
        "counter": 0
    }
}

renderer = ApibbRenderer(counter_app)
print(renderer.render())
# Output:
# Count: 0
# [Increment]

renderer.update('counter', 1)
print(renderer.render())
# Output:
# Count: 1
# [Increment]
```

### Todo List

```python
todo_app = {
    "type": "Container",
    "children": [
        {
            "type": "Text",
            "data": "My Todos"
        },
        {
            "type": "List",
            "data": ["{{todos}}"],
            "children": [
                {"type": "Text", "data": "- {{item.text}}"}
            ]
        }
    ],
    "initialState": {
        "todos": [
            {"text": "Learn APIBB"},
            {"text": "Build an app"},
            {"text": "Deploy it"}
        ]
    }
}

renderer = ApibbRenderer(todo_app)
print(renderer.render())
# Output:
# My Todos
# - Learn APIBB
# - Build an app
# - Deploy it
```

## Architecture

The Python renderer follows the same architecture as the React renderer:

1. **Store** - Global reactive state with `{{path}}` resolution
2. **Component Classes** - Each component type has a Python class
3. **Dynamic Loading** - Components loaded from CDN when needed
4. **Node Rendering** - Recursive rendering of the app tree

## Limitations

- Terminal/CLI only (no GUI)
- No interactive input (read-only rendering)
- Limited styling (terminal constraints)
- Synchronous rendering only

## Future Enhancements

For interactive CLI apps, consider integrating with:
- `rich` - Rich text and beautiful formatting
- `textual` - Terminal user interfaces
- `prompt_toolkit` - Interactive prompts
- `curses` - Full-screen terminal apps

## License

APIBB v1.0 Python Renderer - Frozen 25 November 2025

Part of the APIBB v1.0 specification.
