# AppServer SDK Python AI

[![Python Version](https://img.shields.io/badge/python-3.11+-blue.svg)](https://www.python.org/downloads/)
[![License: MIT](https://img.shields.io/badge/License-MIT-yellow.svg)](https://opensource.org/licenses/MIT)
[![Code style: ruff](https://img.shields.io/endpoint?url=https://raw.githubusercontent.com/astral-sh/ruff/main/assets/badge/v2.json)](https://github.com/astral-sh/ruff)

SDK Python para integração com serviços de IA da AppServer.

## 🚀 Características

- Cliente HTTP assíncrono e síncrono
- Modelos Pydantic para validação de dados
- Retry automático com backoff exponencial  
- Type hints completos
- Suporte a múltiplos provedores de IA
- Logging estruturado
- Testes abrangentes

## 📦 Módulos Disponíveis

### 🤖 LLM (Large Language Models)
Módulo profissional para integração com modelos de linguagem e APIs de inteligência artificial.

**Características principais:**
- Cliente assíncrono e síncrono
- Retry automático com backoff exponencial
- Suporte a múltiplos provedores de IA
- Modelos Pydantic com type hints completos
- Sistema de logging estruturado
- Gerenciamento seguro de API keys

📖 **[Documentação completa do LLM](src/appserver_sdk_python_ai/llm/README.md)**

### 🔍 WebScraping
Módulo profissional de web scraping com conversão para markdown usando Docling.

**Características principais:**
- Scraping robusto com retry automático
- Conversão de alta qualidade usando Docling (IBM)
- Processamento paralelo de múltiplas URLs
- Sistema de cache inteligente
- Limpeza automática de conteúdo
- Extração de metadados ricos
- **OCR integrado**: Processamento de imagens e PDFs
- **Múltiplos engines**: Tesseract, EasyOCR, PaddleOCR

📖 **[Documentação completa do WebScraping](src/appserver_sdk_python_ai/webscraping/README.md)**

### 👁️ OCR (Optical Character Recognition)
Módulo especializado para extração de texto de imagens e documentos.

**Características principais:**
- Múltiplos engines de OCR (Tesseract, EasyOCR, PaddleOCR)
- Seleção automática do melhor engine disponível
- Formatos suportados: JPEG, PNG, GIF, TIFF, BMP, WEBP
- Pré-processamento automático de imagens
- Cache inteligente de resultados
- Processamento em lote paralelo
- Suporte a múltiplos idiomas
- Integração com processamento de PDFs

📖 **[Documentação completa do OCR](src/appserver_sdk_python_ai/ocr/README.md)**

## 📦 Instalação

### Instalação Básica

Para funcionalidades básicas (sem dependências pesadas):

```bash
pip install appserver-sdk-python-ai
```

Esta instalação inclui:
- `pydantic` - Validação de dados
- `httpx` - Cliente HTTP
- `structlog` - Logging estruturado
- `typing-extensions` - Extensões de tipagem

### Instalação com Funcionalidades Específicas

#### Módulo LLM Básico
```bash
pip install appserver-sdk-python-ai[llm]
```
Adiciona: `psutil` para monitoramento de sistema

#### Modelos OpenAI
```bash
pip install appserver-sdk-python-ai[openai]
```
Adiciona: `tiktoken` para tokenização OpenAI

#### Modelos HuggingFace
```bash
pip install appserver-sdk-python-ai[huggingface]
```
Adiciona: `transformers`, `torch`

#### Modelos Locais
```bash
pip install appserver-sdk-python-ai[local-models]
```
Adiciona: `transformers`, `torch`, `llama-cpp-python`, `onnxruntime`

#### Análise Avançada
```bash
pip install appserver-sdk-python-ai[analysis]
```
Adiciona: `nltk`, `spacy`, `textblob`, `pandas`, `matplotlib`, `seaborn`

#### Instalação Completa
```bash
pip install appserver-sdk-python-ai[full]
```
Inclui as principais dependências para LLM e modelos

### Combinando Extras

Você pode combinar múltiplos extras:

```bash
# LLM + OpenAI + Análise
pip install appserver-sdk-python-ai[llm,openai,analysis]

# Instalação completa + desenvolvimento
pip install appserver-sdk-python-ai[full,dev]
```

### Via Poetry (Recomendado para Desenvolvimento)
```bash
# Instalação básica
poetry add appserver-sdk-python-ai

# Com extras
poetry add appserver-sdk-python-ai[full]
```

### Via GitHub (Desenvolvimento)
```bash
# Via Poetry
poetry add git+https://github.com/appserver/appserver-sdk-python-ai.git

# Via pip
pip install git+https://github.com/appserver/appserver-sdk-python-ai.git
```

### Resolução de Problemas

#### Erro de Importação
Se você receber erros de importação, verifique se instalou os extras necessários:

```python
# Verificar módulos disponíveis
from appserver_sdk_python_ai import health_check_all
health_check_all()
```

#### Dependências Conflitantes
Se houver conflitos de versão, use um ambiente virtual:

```bash
# Criar ambiente virtual
python -m venv venv

# Ativar (Windows)
venv\Scripts\activate

# Ativar (Linux/Mac)
source venv/bin/activate

# Instalar SDK
pip install appserver-sdk-python-ai[full]
```

## 🔧 Uso Básico

### Verificação da Instalação

```python
from appserver_sdk_python_ai import get_sdk_info, print_sdk_status

# Informações do SDK
info = get_sdk_info()
print(info)

# Status dos módulos
print_sdk_status()
```

### Tratamento de Dependências Opcionais

O SDK foi projetado para funcionar mesmo sem dependências opcionais:

```python
from appserver_sdk_python_ai import llm

if llm is not None:
    # Módulo LLM disponível
    from appserver_sdk_python_ai.llm import get_token_count
    tokens = get_token_count("Texto de exemplo")
else:
    print("Módulo LLM não disponível")
    print("Instale com: pip install appserver-sdk-python-ai[llm]")
```

### Módulo WebScraping

```python
from appserver_sdk_python_ai.webscraping import quick_scrape

# Scraping simples
markdown = quick_scrape("https://example.com")
print(markdown)
```

### Módulo OCR

```python
from appserver_sdk_python_ai.ocr import quick_ocr

# OCR simples de uma imagem
texto = quick_ocr("documento.png")
print(texto)

# OCR com configurações específicas
from appserver_sdk_python_ai.ocr import OCRProcessor, OCRConfig

config = OCRConfig(
    languages=["por", "eng"],
    engine="tesseract"
)

processor = OCRProcessor(config)
resultado = processor.process_image("imagem.jpg")
print(resultado.text)
print(f"Confiança: {resultado.confidence}%")
```

### Módulo LLM - Uso Básico

```python
try:
    from appserver_sdk_python_ai.llm import (
        get_token_count,
        list_available_models,
        get_portuguese_models
    )
    
    # Contar tokens
    tokens = get_token_count("Olá, mundo!")
    print(f"Tokens: {tokens}")
    
    # Listar modelos
    models = list_available_models()
    print(f"Modelos disponíveis: {len(models)}")
    
except ImportError as e:
    print(f"Módulo LLM não disponível: {e}")
    print("Instale com: pip install appserver-sdk-python-ai[llm]")
```

### Módulo LLM - Cliente Síncrono

```python
from appserver_sdk_python_ai import AIClient
from appserver_sdk_python_ai.models import AIRequest

# Configurar cliente
client = AIClient(
    base_url="https://api.appserver.com.br/ai/v1",
    api_key="sua-api-key"
)

# Fazer requisição
request = AIRequest(
    prompt="Explique machine learning em termos simples",
    model="gpt-4",
    max_tokens=500
)

response = client.chat_completion(request)
print(response.content)
```

### Módulo LLM - Cliente Assíncrono

```python
import asyncio
from appserver_sdk_python_ai import AsyncAIClient

async def main():
    client = AsyncAIClient(
        base_url="https://api.appserver.com.br/ai/v1",
        api_key="sua-api-key"
    )
    
    request = AIRequest(
        prompt="O que é inteligência artificial?",
        model="gpt-3.5-turbo"
    )
    
    response = await client.chat_completion(request)
    print(response.content)
    
    await client.close()

asyncio.run(main())
```

### Módulo LLM - Configuração Avançada

```python
from appserver_sdk_python_ai import AIClient, AIConfig

config = AIConfig(
    base_url="https://api.appserver.com.br/ai/v1",
    api_key="sua-api-key",
    timeout=30,
    max_retries=3,
    retry_delay=1.0,
    debug=True
)

client = AIClient(config=config)
```

## 🛠️ Desenvolvimento

### Pré-requisitos

- Python 3.11+
- Poetry

### Configuração do Ambiente

```bash
# Clonar repositório
git clone https://github.com/appserver/appserver-sdk-python-ai.git
cd appserver-sdk-python-ai

# Instalar dependências
poetry install

# Configurar pre-commit hooks
poetry run pre-commit install

# Ativar ambiente virtual
poetry shell
```

### Executar Testes

```bash
# Todos os testes
poetry run pytest

# Com cobertura
poetry run pytest --cov=appserver_sdk_python_ai --cov-report=html

# Apenas testes unitários
poetry run pytest -m unit

# Apenas testes de integração
poetry run pytest -m integration
```

### Linting e Formatação

```bash
# Verificar e corrigir código
poetry run ruff check . --fix
poetry run ruff format .

# Verificar tipos
poetry run mypy src/

# Verificar segurança
poetry run bandit -r src/
poetry run safety check
```

### Executar Exemplo

```bash
# Exemplos básicos
poetry run python examples/basic_usage.py
poetry run python examples/async_usage.py

# Exemplos do módulo LLM
poetry run python examples/llm/custom_model_example.py
poetry run python examples/llm/metrics_example.py

# Demonstrações completas do módulo LLM
poetry run python examples/llm/features_demo.py
poetry run python examples/llm/improvements_demo.py
poetry run python examples/llm/medium_priority_demo.py
```

## 📚 Documentação

### Documentação dos Módulos

- **[LLM](src/appserver_sdk_python_ai/llm/README.md)** - Guia completo do módulo de modelos de linguagem
- **[WebScraping](src/appserver_sdk_python_ai/webscraping/README.md)** - Guia completo do módulo de web scraping
- **[OCR](src/appserver_sdk_python_ai/ocr/README.md)** - Guia completo do módulo de reconhecimento óptico de caracteres

### Estrutura do Projeto

```
appserver-sdk-python-ai/
├── src/
│   └── appserver_sdk_python_ai/
│       ├── __init__.py
│       ├── llm/                    # Módulo LLM
│       │   ├── __init__.py
│       │   ├── README.md           # Documentação do LLM
│       │   ├── core/
│       │   ├── service/
│       │   └── exceptions/
│       ├── webscraping/            # Módulo WebScraping
│       │   ├── __init__.py
│       │   ├── README.md           # Documentação do WebScraping
│       │   ├── core/
│       │   ├── docling/
│       │   ├── utils/
│       │   └── tests/
│       ├── ocr/                    # Módulo OCR
│       │   ├── __init__.py
│       │   ├── README.md           # Documentação do OCR
│       │   ├── core/
│       │   ├── engines/
│       │   └── utils/
│       └── shared/                 # Utilitários compartilhados
│           ├── __init__.py
│           └── exceptions.py
├── tests/
├── examples/                       # Exemplos de uso e demonstrações
├── scripts/
└── pyproject.toml
```

### Modelos Disponíveis

- `AIRequest`: Modelo de requisição
- `AIResponse`: Modelo de resposta
- `AIConfig`: Configuração do cliente
- `AIError`: Modelo de erro

### Exceções

- `AIException`: Exceção base
- `AIConnectionError`: Erro de conexão
- `AIAuthenticationError`: Erro de autenticação
- `AIRateLimitError`: Erro de limite de taxa
- `AITimeoutError`: Erro de timeout

## 🤖 Módulo LLM

O módulo LLM oferece funcionalidades avançadas para interação com modelos de linguagem:

### 🚀 Funcionalidades Principais

- **Cliente LLM**: Interface unificada para diferentes provedores
- **Cache Inteligente**: Sistema de cache LRU thread-safe para otimização
- **Validação Robusta**: Validação de entrada e saída com múltiplos níveis
- **Logging Estruturado**: Sistema de logging avançado com contexto
- **Métricas e Monitoramento**: Coleta automática de métricas de performance
- **Configuração Centralizada**: Gerenciamento unificado de configurações

### 📊 Sistema de Métricas

O módulo inclui um sistema abrangente de métricas:

```python
from appserver_sdk_python_ai.llm import (
    get_metrics_summary,
    export_metrics,
    record_operation_metric
)

# Coleta automática durante operações
summary = get_metrics_summary()
print(f"Operações realizadas: {len(summary['operation_stats'])}")

# Exportação para análise
export_metrics(format_type="json", file_path="metrics.json")
export_metrics(format_type="prometheus", file_path="metrics.prom")
```

**Tipos de Métricas Coletadas:**
- ⏱️ Latência e duração de operações
- 📈 Contadores de sucesso/erro
- 💾 Uso de memória e CPU
- 🔢 Estatísticas de tokens processados
- 📊 Histogramas de performance

### 📖 Documentação Completa

Para documentação detalhada do módulo LLM, consulte:
- **[📚 README do Módulo LLM](src/appserver_sdk_python_ai/llm/README.md)** - Documentação completa e consolidada
- **[🔧 Documentação Interativa](src/appserver_sdk_python_ai/llm/docs/interactive_docs.py)** - Acesso dinâmico à documentação
- **[💡 Exemplos Práticos](examples/llm/)** - Exemplos de uso organizados por funcionalidade
- **[📊 Métricas e Outputs](output/metrics/)** - Arquivos de métricas e relatórios
- **[📝 Logs do Sistema](logs/)** - Logs estruturados da aplicação

> **Nota**: A documentação do módulo LLM foi consolidada em um único local para evitar redundância. O sistema de documentação interativa carrega dinamicamente o conteúdo do README.md do módulo.

## 🤝 Contribuindo

1. Fork o projeto
2. Crie uma branch para sua feature (`git checkout -b feature/nova-feature`)
3. Commit suas mudanças (`git commit -m 'feat: adiciona nova feature'`)
4. Push para a branch (`git push origin feature/nova-feature`)
5. Abra um Pull Request

### Padrões de Commit

Seguimos o padrão [Conventional Commits](https://www.conventionalcommits.org/):

- `feat:` nova funcionalidade
- `fix:` correção de bug
- `docs:` mudanças na documentação
- `style:` formatação de código
- `refactor:` refatoração de código
- `test:` adição ou correção de testes
- `chore:` tarefas de manutenção

### Convenções de Documentação

#### Nomenclatura de Arquivos
- **Preferência**: Documentação no `README.md` do próprio módulo
- **Formato alternativo**: `{modulo}-{tipo}.md` (apenas quando necessário)

#### Estrutura dos Documentos
1. **Título e Descrição**
2. **Índice** (se necessário)
3. **Conteúdo Principal**
4. **Exemplos Práticos**
5. **Referências e Links**

#### Adicionando Nova Documentação
1. **Criar Arquivo**: Use a convenção `{modulo}-{tipo}.md`
2. **Seguir Estrutura**: Mantenha consistência com documentos existentes
3. **Atualizar README**: Adicione referência neste arquivo
4. **Links Cruzados**: Atualize links relevantes em outros documentos

#### Manutenção da Documentação
- **Revisão Regular**: Mensal ou a cada release
- **Verificações**: Links funcionais, informações atualizadas, exemplos válidos
- **Sincronização**: Manter sincronizado com mudanças no código
- **Versionamento**: Indicar mudanças significativas

## 📄 Licença

Este projeto está licenciado sob a Licença MIT - veja o arquivo [LICENSE](LICENSE) para detalhes.

## 🆘 Suporte

- **Email**: suporte@appserver.com.br
- **Issues**: [GitHub Issues](https://github.com/appserver/appserver-sdk-python-ai/issues)
- **Documentação**: [Wiki](https://github.com/appserver/appserver-sdk-python-ai/wiki)

## 📊 Status do Projeto

- ✅ Cliente básico implementado
- ✅ Modelos Pydantic
- ✅ Testes unitários
- ✅ Módulo LLM com funcionalidades avançadas
- ✅ Sistema de métricas e monitoramento
- ✅ Cache LRU thread-safe
- ✅ Validação robusta de dados
- ✅ Logging estruturado
- ✅ Configuração centralizada
- 🔄 Documentação (em andamento)
- 🔄 Testes de integração (em andamento)
- ⏳ Suporte a streaming (planejado)

---

**Desenvolvido com ❤️ pela equipe AppServer**
