"""Exceções customizadas do AppServer SDK."""

from __future__ import annotations


class AppServerSDKError(Exception):
    """Exceção base para erros do AppServer SDK."""

    def __init__(self, message: str, error_code: str | None = None) -> None:
        """Inicializa exceção.

        Args:
            message: Mensagem de erro
            error_code: Código de erro opcional
        """
        super().__init__(message)
        self.message = message
        self.error_code = error_code


class TokenizationError(AppServerSDKError):
    """Erro durante processo de tokenização."""

    def __init__(
        self,
        message: str,
        model_name: str | None = None,
        original_error: Exception | None = None,
    ) -> None:
        """Inicializa erro de tokenização.

        Args:
            message: Mensagem de erro
            model_name: Nome do modelo que causou o erro
            original_error: Erro original que causou este erro
        """
        super().__init__(message, "TOKENIZATION_ERROR")
        self.model_name = model_name
        self.original_error = original_error


class ModelNotFoundError(AppServerSDKError):
    """Modelo não encontrado ou não registrado."""

    def __init__(self, model_name: str) -> None:
        """Inicializa erro de modelo não encontrado.

        Args:
            model_name: Nome do modelo não encontrado
        """
        super().__init__(
            f"Modelo '{model_name}' não encontrado ou não registrado", "MODEL_NOT_FOUND"
        )
        self.model_name = model_name


class DependencyNotInstalledError(AppServerSDKError):
    """Dependência necessária não está instalada."""

    def __init__(self, dependency: str, install_command: str | None = None) -> None:
        """Inicializa erro de dependência.

        Args:
            dependency: Nome da dependência
            install_command: Comando para instalar a dependência
        """
        message = f"Dependência '{dependency}' não está instalada"
        if install_command:
            message += f". Instale com: {install_command}"

        super().__init__(message, "DEPENDENCY_NOT_INSTALLED")
        self.dependency = dependency
        self.install_command = install_command
